/**
 ******************************************************************************
 * @file    main.cpp
 * @author  Davide Aliprandi, STMicroelectronics
 * @version V1.0.0
 * @date    October 16th, 2015
 * @brief   mbed vertical application using the STMicroelectronics
 *          X-NUCLEO-IHM01A1 Motor Control Expansion Board and the
 *          X-NUCLEO-IKS01A2 MEMS Inertial & Environmental Sensors Expansion
 *          Board to get a MEMS-based motor control (direction and speed).
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */


/* Includes ------------------------------------------------------------------*/

/* mbed specific header files. */
#include "mbed.h"

/* Helper header files. */
#include "DevSPI.h"

/* Components and expansion boards specific header files. */
#include "XNucleoIKS01A2.h"
#include "L6474.h"


/* Definitions ---------------------------------------------------------------*/

/* Absolute value of the threshold on the Y axis acceleration. */
#define ACCELERATION_TH 50

/* Rotation gain. */
#define ROTATION_SPEED_GAIN 20


/* Variables -----------------------------------------------------------------*/

/* MEMS Expansion Board. */
XNucleoIKS01A2 *sensors;

/* Motor Control Component. */
L6474 *motor;


/* Main ----------------------------------------------------------------------*/

int main()
{
    /*----- Initialization. -----*/

    /* Initializing I2C bus. */
    DevI2C dev_i2c(D14, D15);

    /* Initializing SPI bus. */
    DevSPI dev_spi(D11, D12, D13);

    /* Initializing MEMS Expansion Board. */
    sensors = XNucleoIKS01A2::instance(&dev_i2c);

    /* Retrieving the accelerometer. */
    LSM303AGRAccSensor *accelerometer = sensors->accelerometer;
    accelerometer->enable();
    int acceleration_axis = 1;

    /* Initializing Motor Control Component. */
    motor = new L6474(D2, D8, D7, D9, D10, dev_spi);
    if (motor->init() != COMPONENT_OK)
        exit(EXIT_FAILURE);

    /* Set defaults. */
    motor->set_acceleration(10000);
    motor->set_deceleration(10000);
    motor->set_min_speed(100);
    int status = 0;
    int speed = 0;


    /*----- Infinite Loop. -----*/

    /* Printing to the console. */
    printf("Motor Control with MEMS\r\n\n");

    /* Main Loop. */
    while(true)
    {
        /* Reading Accelerometer. */
        int accelerometer_data[3];
        accelerometer->get_x_axes(accelerometer_data);

        /* Motor Control. */
        int module = abs(accelerometer_data[acceleration_axis]);
        if (module > ACCELERATION_TH)
        {
            int sign = accelerometer_data[acceleration_axis] < 0 ? -1 : 1;
            speed = module * ROTATION_SPEED_GAIN;

            /* Requesting to run. */
            if (status != sign)
            {
                motor->run(sign == -1 ? StepperMotor::BWD : StepperMotor::FWD);
                status = sign;
            }

            /* Setting Speed. */
            motor->set_max_speed(speed);

            /* Printing to the console. */
            printf("Speed: %c%d\r\n", sign == -1 ? '-' : '+', motor->get_speed());
        }
        else if (status != 0)
        {
            /* Requesting to stop. */
            motor->soft_stop();
            status = 0;
            speed = 0;

            /* Printing to the console. */
            printf("Stop.\r\n");
        }

        /* Waiting. */
        wait_ms(50);
    }
}
