/**
  ******************************************************************************
  * @file    CustomConsoleService.h
  * @author  Central Labs / AST
  * @version V0.9.0
  * @date    23-Dec-2015
  * @brief   BLE console service
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */

#ifndef __CUSTOM_BLE_CONSOLE_SERVICE_H__
#define __CUSTOM_BLE_CONSOLE_SERVICE_H__

#include "BLE.h"
#include "ble_debug.h"

const UUID::LongUUIDBytes_t CONS_SERVICE_UUID_128 =                   { 0x00,0x00,0x00,0x00,0x00,0x0E,0x11,0xe1,0x9a,0xb4,0x00,0x02,0xa5,0xd5,0xc5,0x1b }; 
const UUID::LongUUIDBytes_t CONS_TERM_CHAR_UUID_128 =                 { 0x00,0x00,0x00,0x01,0x00,0x0E,0x11,0xe1,0xac,0x36,0x00,0x02,0xa5,0xd5,0xc5,0x1b };  
const UUID::LongUUIDBytes_t CONS_STDERR_CHAR_UUID_128 =               { 0x00,0x00,0x00,0x02,0x00,0x0E,0x11,0xe1,0xac,0x36,0x00,0x02,0xa5,0xd5,0xc5,0x1b };  

#define W2ST_CONSOLE_MAX_CHAR_LEN   20

/* Custom Sensors Service */
class CustomConsoleService  {
public:
    CustomConsoleService(BLE &_ble) :
        ble(_ble),
        consTermCharacteristic(CONS_TERM_CHAR_UUID_128,LastTermBuffer, W2ST_CONSOLE_MAX_CHAR_LEN, W2ST_CONSOLE_MAX_CHAR_LEN, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE |
                                GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        consStderrCharacteristic(CONS_STDERR_CHAR_UUID_128, LastStderrBuffer, W2ST_CONSOLE_MAX_CHAR_LEN, W2ST_CONSOLE_MAX_CHAR_LEN,
                                 GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY)
        {
 
            static bool serviceAdded = false; /* We should only ever need to add the env service once. */
            if (serviceAdded) {
            return;
            }

            GattCharacteristic *charTable[] = {&consTermCharacteristic, &consStderrCharacteristic};        
                        
            GattService   consService(CONS_SERVICE_UUID_128, charTable, sizeof(charTable) / sizeof(GattCharacteristic *));               
                
            ble.gattServer().addService(consService);             

            memset (LastTermBuffer, 0, W2ST_CONSOLE_MAX_CHAR_LEN);  
            memset (LastStderrBuffer, 0, W2ST_CONSOLE_MAX_CHAR_LEN);           
                                                                         
            isBTLEConnected                     = DISCONNECTED;  
            isEnabledTermNotify                 = false;
            isEnabledStderrNotify               = false;                                                                            
            serviceAdded                        = true;                                                                                                                                  
        }

    tBleStatus updateTerm (uint8_t *data,uint8_t length) {
        tBleStatus ret;
        uint8_t Offset;
        uint8_t DataToSend;
//        uint8_t BufferToWrite[W2ST_CONSOLE_MAX_CHAR_LEN];
//        uint8_t BytesToWrite;    

        if (isEnabledTermNotify && isBTLEConnected==CONNECTED) {
        /* Split the code in packages */
            for(Offset =0; Offset<length; Offset +=W2ST_CONSOLE_MAX_CHAR_LEN){
                DataToSend = (length-Offset);
                DataToSend = (DataToSend>W2ST_CONSOLE_MAX_CHAR_LEN) ?  W2ST_CONSOLE_MAX_CHAR_LEN : DataToSend;

                /* keep a copy */
                memcpy(LastTermBuffer,data+Offset,DataToSend);
                LastTermLen = DataToSend;       
                PRINTF("updateTerm handle: %d string: %s\n\r", consTermCharacteristic.getValueAttribute().getHandle(), LastTermBuffer);
                ret = ble.gattServer().write(consTermCharacteristic.getValueAttribute().getHandle(), data+Offset, DataToSend, 0);
                if (ret != BLE_ERROR_NONE) {  /* FIXME the wrong errcode from BLE requires BLE fix */
//                    BytesToWrite = sprintf((char *)BufferToWrite, "Error Updating Stdout Char\r\n");
//                    updateStderr(BufferToWrite,BytesToWrite);               
                }
                return ret;               
            }
        }
        return BLE_STATUS_SUCCESS;
    }
       
    tBleStatus updateStderr (uint8_t *data,uint8_t length) {

        uint8_t Offset;
        uint8_t DataToSend;
        
        if (isEnabledStderrNotify && isBTLEConnected==CONNECTED) {
        /* Split the code in packages */
            for(Offset =0; Offset<length; Offset +=W2ST_CONSOLE_MAX_CHAR_LEN){
                DataToSend = (length-Offset);
                DataToSend = (DataToSend>W2ST_CONSOLE_MAX_CHAR_LEN) ?  W2ST_CONSOLE_MAX_CHAR_LEN : DataToSend;

                /* keep a copy */
                memcpy(LastStderrBuffer,data+Offset,DataToSend);
                LastStderrLen = DataToSend;
                PRINTF("updateStderr handle: %d string: %s\n\r", consStderrCharacteristic.getValueAttribute().getHandle(), LastStderrBuffer);
                return (ble.gattServer().write(consStderrCharacteristic.getValueAttribute().getHandle(), data+Offset, DataToSend, 0));                          
            }
        }
        return BLE_STATUS_SUCCESS;
    }

    void updateConnectionStatus (ConnectionStatus_t status){
            isBTLEConnected = status;                      
            isEnabledTermNotify                 = false;
            isEnabledStderrNotify               = false;                 
    }             

    bool isTermHandle (Gap::Handle_t handle) {
            if (handle == consTermCharacteristic.getValueAttribute().getHandle() - BLE_HANDLE_VALUE_OFFSET) return true;
            return false;
    }

    bool isStderrHandle (Gap::Handle_t handle) {
            if (handle == consStderrCharacteristic.getValueAttribute().getHandle() - BLE_HANDLE_VALUE_OFFSET) return true;
            return false;
    }
    
    void enNotify (Gap::Handle_t handle)  {
            if (isTermHandle(handle - BLE_HANDLE_EN_DIS_OFFSET))   { isEnabledTermNotify = true; return; }    
            if (isStderrHandle(handle - BLE_HANDLE_EN_DIS_OFFSET)) { isEnabledStderrNotify = true; return; }                
    }
        
    void disNotify (Gap::Handle_t handle) {
            if (isTermHandle(handle - BLE_HANDLE_EN_DIS_OFFSET))   { isEnabledTermNotify = false; return; }    
            if (isStderrHandle(handle - BLE_HANDLE_EN_DIS_OFFSET)) { isEnabledStderrNotify = false; return; }                            
    }


private:
    BLE                 &ble;
    uint8_t             LastTermBuffer     [W2ST_CONSOLE_MAX_CHAR_LEN]; 
    uint8_t             LastTermLen;
    uint8_t             LastStderrBuffer   [W2ST_CONSOLE_MAX_CHAR_LEN];   
    uint8_t             LastStderrLen;                   
     
    GattCharacteristic  consTermCharacteristic;     
    GattCharacteristic  consStderrCharacteristic;   
    ConnectionStatus_t  isBTLEConnected;
    bool                isEnabledTermNotify;
    bool                isEnabledStderrNotify;    
};

#endif /* #ifndef __CUSTOM_BLE_CONSOLE_SERVICE_H__*/