/**
  ******************************************************************************
  * @file    CustomSensorService.h
  * @author  Central Labs / AST
  * @version V0.9.0
  * @date    23-Dec-2015
  * @brief   BLE MEMS and environmental sensors service
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */

#ifndef __CUSTOM_BLE_SENSORS_SERVICE_H__
#define __CUSTOM_BLE_SENSORS_SERVICE_H__

#include "BLE.h"
#include "ble_utils.h"
#include "ble_debug.h"
#include "UUID.h"

//#define DISAGGREGATED_ENV_CHAR  // uncomment to enable the disaggregated environmental temp,hum,pres characteristics, attention because of BLE mem limitations                                // the maximum number of characteristics is limited check BLUENRG_STACK_MODE
//#define DISAGGREGATED_MOT_CHAR  // "                                     motion acc, gyro, mag "

const UUID::LongUUIDBytes_t SENS_SERVICE_UUID_128 =           { 0x00,0x00,0x00,0x00,0x00,0x01,0x11,0xe1,0x9a,0xb4,0x00,0x02,0xa5,0xd5,0xc5,0x1b }; // temp, pressure, humidity,
#ifdef DISAGGREGATED_ENV_CHAR
const UUID::LongUUIDBytes_t SENS_TEMP_CHAR_UUID_128 =         { 0x00,0x04,0x00,0x00,0x00,0x01,0x11,0xe1,0xac,0x36,0x00,0x02,0xa5,0xd5,0xc5,0x1b };
const UUID::LongUUIDBytes_t SENS_HUMI_CHAR_UUID_128 =         { 0x00,0x08,0x00,0x00,0x00,0x01,0x11,0xe1,0xac,0x36,0x00,0x02,0xa5,0xd5,0xc5,0x1b };
const UUID::LongUUIDBytes_t SENS_PRES_CHAR_UUID_128 =         { 0x00,0x10,0x00,0x00,0x00,0x01,0x11,0xe1,0xac,0x36,0x00,0x02,0xa5,0xd5,0xc5,0x1b };
#endif
#ifdef DISAGGREGATED_MOT_CHAR
const UUID::LongUUIDBytes_t SENS_MAGN_CHAR_UUID_128   =       { 0x00,0x20,0x00,0x00,0x00,0x01,0x11,0xe1,0xac,0x36,0x00,0x02,0xa5,0xd5,0xc5,0x1b };
const UUID::LongUUIDBytes_t SENS_GYRO_CHAR_UUID_128   =       { 0x00,0x40,0x00,0x00,0x00,0x01,0x11,0xe1,0xac,0x36,0x00,0x02,0xa5,0xd5,0xc5,0x1b };
const UUID::LongUUIDBytes_t SENS_ACCE_CHAR_UUID_128   =       { 0x00,0x80,0x00,0x00,0x00,0x01,0x11,0xe1,0xac,0x36,0x00,0x02,0xa5,0xd5,0xc5,0x1b };
#endif
const UUID::LongUUIDBytes_t SENS_ACC_GYRO_MAG_CHAR_UUID_128 = { 0x00,0xE0,0x00,0x00,0x00,0x01,0x11,0xe1,0xac,0x36,0x00,0x02,0xa5,0xd5,0xc5,0x1b };
const UUID::LongUUIDBytes_t SENS_PRES_HUM_TEMP_CHAR_UUID_128= { 0x00,0x1C,0x00,0x00,0x00,0x01,0x11,0xe1,0xac,0x36,0x00,0x02,0xa5,0xd5,0xc5,0x1b };


#define TEMP_DATA_LEN   2+2
#define HUM_DATA_LEN    2+2
#define PRES_DATA_LEN   2+4
#define ACC_DATA_LEN    6+2
#define MAG_DATA_LEN    6+2
#define GYRO_DATA_LEN   6+2
#define ACCGYROMAG_DATA_LEN 2+3*3*2
#define PRESHUMTEMP_DATA_LEN 2+4+2+2

/* Custom Sensors Service */
class CustomSensorService
{
public:
    CustomSensorService(BLE &_ble) :
        ble(_ble),
#ifdef DISAGGREGATED_ENV_CHAR        
        envTemperatureCharacteristic(SENS_TEMP_CHAR_UUID_128,envTemperature, TEMP_DATA_LEN, TEMP_DATA_LEN,
                                     GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        envHumidityCharacteristic(SENS_HUMI_CHAR_UUID_128, envHumidity, HUM_DATA_LEN, HUM_DATA_LEN,
                                  GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        envPressureCharacteristic(SENS_PRES_CHAR_UUID_128, envPressure, PRES_DATA_LEN, PRES_DATA_LEN,
                                  GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
#endif 
#ifdef DISAGGREGATED_MOT_CHAR                                         
        envMagnetometerCharacteristic(SENS_MAGN_CHAR_UUID_128,envMagn, MAG_DATA_LEN, MAG_DATA_LEN,
                                      GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        envAccelerometerCharacteristic(SENS_ACCE_CHAR_UUID_128,envAcce, ACC_DATA_LEN, ACC_DATA_LEN,
                                       GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        envGyroCharacteristic(SENS_GYRO_CHAR_UUID_128,envGyro, GYRO_DATA_LEN, GYRO_DATA_LEN,
                              GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
#endif                              
        envAccGyroMagCharacteristic(SENS_ACC_GYRO_MAG_CHAR_UUID_128,envAccGyroMag, ACCGYROMAG_DATA_LEN, ACCGYROMAG_DATA_LEN,
                                    GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY), 
        envPresHumTempCharacteristic(SENS_PRES_HUM_TEMP_CHAR_UUID_128,envPresHumTemp, PRESHUMTEMP_DATA_LEN, PRESHUMTEMP_DATA_LEN,
                                    GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY)
    {

        static bool serviceAdded = false; /* We should only ever need to add the env service once. */
        if (serviceAdded) {
            return;
        }

        GattCharacteristic *charTable[] = {
#ifdef DISAGGREGATED_ENV_CHAR            
                                           &envTemperatureCharacteristic, &envHumidityCharacteristic, &envPressureCharacteristic,
#endif
                                           &envPresHumTempCharacteristic,
#ifdef  DISAGGREGATED_MOT_CHAR                                          
                                           &envMagnetometerCharacteristic, &envAccelerometerCharacteristic, &envGyroCharacteristic, 
#endif                                           
                                           &envAccGyroMagCharacteristic
                                          };

        GattService   envService(SENS_SERVICE_UUID_128, charTable, sizeof(charTable) / sizeof(GattCharacteristic *));

        ble.gattServer().addService(envService);

#ifdef DISAGGREGATED_ENV_CHAR        
        isEnabledTempNotify             = false;
        isEnabledHumNotify              = false;
        isEnabledPresNotify             = false;
#endif
#ifdef DISAGGREGATED_ENV_CHAR                
        isEnabledGyroNotify             = false;
        isEnabledAccNotify              = false;
        isEnabledMagNotify              = false;
#endif        
        isEnabledAccGyroMagNotify       = false;
        isEnabledPresHumTempNotify      = false;
        
        isTempCalibrated                = false;
        isHumCalibrated                 = false;
        isPresCalibrated                = false;
        isMagCalibrated                 = false;
        isAccCalibrated                 = false;
        isAGyroCalibrated               = false;

        memset (pastenvTemperature,     0, TEMP_DATA_LEN);
        memset (pastenvHumidity,        0, HUM_DATA_LEN);
        memset (pastenvPressure,        0, PRES_DATA_LEN);

        isBTLEConnected                 = DISCONNECTED;
        serviceAdded                    = true;
    }
#ifdef DISAGGREGATED_ENV_CHAR
    uint32_t sendEnvTemperature (int16_t Temp, uint16_t TimeStamp) {
        STORE_LE_16(envTemperature,TimeStamp);
        STORE_LE_16(envTemperature+2,Temp);
        memcpy (pastenvTemperature, envTemperature, TEMP_DATA_LEN);
        return ble.gattServer().write(envTemperatureCharacteristic.getValueAttribute().getHandle(), envTemperature, TEMP_DATA_LEN, 0);
    }

    /**
     * Update the temperature with a new value. Valid values range from
     * 0..100. Anything outside this range will be ignored.
     * @param newLevel New level. */
    uint32_t updateEnvTemperature (int16_t Temp, uint16_t TimeStamp) {
        if (memcmp (&pastenvTemperature[2], &Temp, 2) != 0) {
            return sendEnvTemperature (Temp,  TimeStamp);
        }
        return 0;
    }

    uint32_t sendEnvHumidity(uint16_t Hum, uint16_t TimeStamp) {
        STORE_LE_16(envHumidity,TimeStamp);
        STORE_LE_16(envHumidity+2,Hum);
        memcpy (pastenvHumidity, envHumidity, HUM_DATA_LEN);
        return ble.gattServer().write(envHumidityCharacteristic.getValueAttribute().getHandle(), envHumidity, HUM_DATA_LEN, 0);
    }

    uint32_t updateEnvHumidity(uint16_t Hum, uint16_t TimeStamp) {
        if (memcmp (&pastenvHumidity[2], &Hum, 2) != 0) {
            return sendEnvHumidity(Hum, TimeStamp);
        }
        return 0;
    }

    uint32_t sendEnvPressure(uint32_t Press, uint16_t TimeStamp) {
        STORE_LE_16(envPressure,TimeStamp);
        STORE_LE_32(envPressure+2,Press);
        memcpy (pastenvPressure, envPressure, PRES_DATA_LEN);
        return ble.gattServer().write(envPressureCharacteristic.getValueAttribute().getHandle(), envPressure, PRES_DATA_LEN, 0);
    }

    uint32_t updateEnvPressure(uint32_t Press, uint16_t TimeStamp) {
        if (memcmp (&pastenvPressure[2], &Press, 2) != 0) {
            return sendEnvPressure(Press, TimeStamp);
        }
        return 0;
    }
#endif

    uint32_t sendEnvPresHumTemp(uint32_t Press, uint16_t Hum, int16_t Temp, uint16_t TimeStamp) {
        STORE_LE_16(envPressure,TimeStamp);
        STORE_LE_32(envPressure+2,Press);
        memcpy (pastenvPressure, envPressure, PRES_DATA_LEN);
        STORE_LE_16(envHumidity,TimeStamp);
        STORE_LE_16(envHumidity+2,Hum);
        memcpy (pastenvHumidity, envHumidity, HUM_DATA_LEN);            
        STORE_LE_16(envTemperature,TimeStamp);
        STORE_LE_16(envTemperature+2,Temp);
        memcpy (pastenvTemperature, envTemperature, TEMP_DATA_LEN);
        STORE_LE_16(envPresHumTemp, TimeStamp);             
        memcpy (envPresHumTemp+2, envPressure+2, 4);
        memcpy (envPresHumTemp+2+4, envHumidity+2, 2);
        memcpy (envPresHumTemp+2+4+2, envTemperature+2, 2);           
        return ble.gattServer().write(envPresHumTempCharacteristic.getValueAttribute().getHandle(), envPresHumTemp, PRESHUMTEMP_DATA_LEN, 0);    
    }
    
    uint32_t updateEnvPresHumTemp(uint32_t Press, uint16_t Hum, int16_t Temp, uint16_t TimeStamp) {
        if ((memcmp (&pastenvPressure[2], &Press, 2) != 0) ||
            (memcmp (&pastenvHumidity[2], &Hum, 2) != 0)   ||
            (memcmp (&pastenvTemperature[2], &Temp, 2) != 0)) {
            return sendEnvPresHumTemp (Press, Hum, Temp, TimeStamp);
        } 
        return 0;
    }        
#ifdef DISAGGREGATED_MOT_CHAR                
    uint32_t sendEnvMagnetometer(AxesRaw_TypeDef *Magn, uint16_t TimeStamp, osxMFX_calibFactor *magOffset) {
        STORE_LE_16(envMagn,TimeStamp);
        STORE_LE_16(envMagn+2,(Magn->AXIS_X - magOffset->magOffX));
        STORE_LE_16(envMagn+4,(Magn->AXIS_Y - magOffset->magOffY));
        STORE_LE_16(envMagn+6,(Magn->AXIS_Z - magOffset->magOffZ));
        return ble.gattServer().write(envMagnetometerCharacteristic.getValueAttribute().getHandle(), envMagn, MAG_DATA_LEN, 0);
    }

    uint32_t updateEnvMagnetometer(AxesRaw_TypeDef *Magn, uint16_t TimeStamp, osxMFX_calibFactor *magOffset) {
        if (isMagNotificationEn()) return sendEnvMagnetometer(Magn, TimeStamp, magOffset);
        return 0;
    }

    uint32_t sendEnvAccelerometer (AxesRaw_TypeDef *Acc, uint16_t TimeStamp) {
        STORE_LE_16(envAcce,TimeStamp);
        STORE_LE_16(envAcce+2,Acc->AXIS_X);
        STORE_LE_16(envAcce+4,Acc->AXIS_Y);
        STORE_LE_16(envAcce+6,Acc->AXIS_Z);
        return ble.gattServer().write(envAccelerometerCharacteristic.getValueAttribute().getHandle(), envAcce, ACC_DATA_LEN, 0);
    }

    uint32_t updateEnvAccelerometer (AxesRaw_TypeDef *Acc, uint16_t TimeStamp) {
        if (isAccNotificationEn()) return sendEnvAccelerometer (Acc, TimeStamp);
        return 0;
    }

    uint32_t sendEnvGyroscope (AxesRaw_TypeDef *Gyro, uint16_t TimeStamp) {
        STORE_LE_16(envGyro,TimeStamp);
        STORE_LE_16(envGyro+2,Gyro->AXIS_X);
        STORE_LE_16(envGyro+4,Gyro->AXIS_Y);
        STORE_LE_16(envGyro+6,Gyro->AXIS_Z);
        return ble.gattServer().write(envGyroCharacteristic.getValueAttribute().getHandle(), envGyro, GYRO_DATA_LEN, 0);
    }

    uint32_t updateEnvGyroscope (AxesRaw_TypeDef *Gyro, uint16_t TimeStamp) {
        if (isGyroNotificationEn()) return sendEnvGyroscope (Gyro, TimeStamp);
        return 0;
    }
#endif
    uint32_t sendEnvAccGyroMag (AxesRaw_TypeDef *Acc, AxesRaw_TypeDef *Gyro, AxesRaw_TypeDef *Magn, uint16_t TimeStamp, osxMFX_calibFactor *magOffset) {
        STORE_LE_16(envAccGyroMag,TimeStamp);
        STORE_LE_16(envAccGyroMag+2,Acc->AXIS_X);
        STORE_LE_16(envAccGyroMag+4,Acc->AXIS_Y);
        STORE_LE_16(envAccGyroMag+6,Acc->AXIS_Z);

        STORE_LE_16(envAccGyroMag+8,Gyro->AXIS_X);
        STORE_LE_16(envAccGyroMag+10,Gyro->AXIS_Y);
        STORE_LE_16(envAccGyroMag+12,Gyro->AXIS_Z);

        STORE_LE_16(envAccGyroMag+14,(Magn->AXIS_X  - magOffset->magOffX));
        STORE_LE_16(envAccGyroMag+16,(Magn->AXIS_Y  - magOffset->magOffY));
        STORE_LE_16(envAccGyroMag+18,(Magn->AXIS_Z  - magOffset->magOffZ));
        return ble.gattServer().write(envAccGyroMagCharacteristic.getValueAttribute().getHandle(), envAccGyroMag, ACCGYROMAG_DATA_LEN, 0);
    }

    uint32_t updateEnvAccGyroMag (AxesRaw_TypeDef *Acc, AxesRaw_TypeDef *Gyro, AxesRaw_TypeDef *Magn, uint16_t TimeStamp, osxMFX_calibFactor *magOffset) {
        if (isAccGyroMagNotificationEn()) return sendEnvAccGyroMag (Acc, Gyro, Magn, TimeStamp, magOffset);
        return 0;
    }
    void enNotify (Gap::Handle_t a_handle) {
        Gap::Handle_t handle = a_handle - BLE_HANDLE_EN_DIS_OFFSET;
#ifdef DISAGGREGATED_ENV_CHAR        
        if (isTempHandle(handle)) {
            isEnabledTempNotify = true;
            memset(pastenvTemperature,0,TEMP_DATA_LEN);
            printf ("isTempHandle en\n\r");                 
            return;
        }
        if (isHumHandle(handle))  {
            isEnabledHumNotify  = true;
            memset(pastenvHumidity,0,HUM_DATA_LEN);
            printf ("isHumHandle en\n\r");                 
            return;
        }
        if (isPresHandle(handle)) {
            isEnabledPresNotify = true;
            memset(pastenvPressure,0,PRES_DATA_LEN);
            printf ("isPresHandle en\n\r");                 
            return;
        }
#endif
#ifdef DISAGGREGATED_ENV_CHAR                
        if (isGyroHandle(handle)) {
            isEnabledGyroNotify = true;
            printf ("isGyroHandle en\n\r");                 
            return;
        }
        if (isAccHandle(handle))  {
            isEnabledAccNotify  = true;
            printf ("isAccHandle en\n\r");                 
            return;
        }
        if (isMagHandle(handle))  {
            isEnabledMagNotify  = true;
            printf ("isMagHandle en\n\r");                 
            return;
        }
#endif        
        if (isAccGyroMagHandle(handle)) {
            isEnabledAccGyroMagNotify = true;
            printf ("isAccGyroMagHandle en\n\r");            
            return;
        }
        if (isPresHumTempHandle(handle)) {
            isEnabledPresHumTempNotify = true;
            printf ("isPresHumTempHandle en\n\r");
            return;
        }        
    }

    void disNotify (Gap::Handle_t a_handle) {
        Gap::Handle_t handle = a_handle - BLE_HANDLE_EN_DIS_OFFSET;        
#ifdef DISAGGREGATED_ENV_CHAR        
        if (isTempHandle(handle)) {     
            isEnabledTempNotify = false;
            memset(pastenvTemperature,0,TEMP_DATA_LEN);
            printf ("isTempHandle dis\n\r");            
            return;
        }        
        if (isHumHandle(handle))  {
            isEnabledHumNotify  = false;
            memset(pastenvHumidity,0,HUM_DATA_LEN);
            printf ("isHumHandle dis\n\r");                     
            return;
        }
        if (isPresHandle(handle)) {
            isEnabledPresNotify = false;
            memset(pastenvPressure,0,PRES_DATA_LEN);
            printf ("isPresHandle dis\n\r");                     
            return;
        }
#endif
#ifdef DISAGGREGATED_ENV_CHAR                
        if (isGyroHandle(handle)) {
            isEnabledGyroNotify = false;
            printf ("isGyroHandle dis\n\r");                     
            return;
        }
        if (isAccHandle(handle))  {
            isEnabledAccNotify  = false;
            printf ("isAccHandle dis\n\r");                     
            return;
        }
        if (isMagHandle(handle))  {            
            isEnabledMagNotify  = false;
            printf ("isMagHandle dis\n\r");                     
            return;
        }
#endif        
        if (isAccGyroMagHandle(handle)) {
            isEnabledAccGyroMagNotify = false;
            printf ("isAccGyroMagHandle dis\n\r");                     
            return;
        }
        if (isPresHumTempHandle(handle)) {
            isEnabledPresHumTempNotify = false;
            printf ("isPresHumTempHandle dis\n\r");                     
            return;
        }                
    }
#ifdef DISAGGREGATED_ENV_CHAR
    bool isTempNotificationEn (void) {
        return isEnabledTempNotify;
    }

    bool isHumNotificationEn (void) {
        return isEnabledHumNotify;
    }

    bool isPresNotificationEn (void) {
        return isEnabledPresNotify;
    }
#endif
#ifdef DISAGGREGATED_MOT_CHAR        
    bool isGyroNotificationEn (void) {
        return isEnabledGyroNotify;
    }

    bool isAccNotificationEn (void) {
        return isEnabledAccNotify;
    }

    bool isMagNotificationEn (void) {
        return isEnabledMagNotify;
    }
#endif
    bool isAccGyroMagNotificationEn (void) {
        return isEnabledAccGyroMagNotify;
    }
    
    bool isPresHumTempNotificationEn (void) {
        return isEnabledPresHumTempNotify;
    }    
#ifdef DISAGGREGATED_ENV_CHAR
    bool isTempHandle (Gap::Handle_t handle) {
        if (handle == envTemperatureCharacteristic.getValueAttribute().getHandle() - BLE_HANDLE_VALUE_OFFSET) return true;
        return false;
    }

    bool isHumHandle (Gap::Handle_t handle) {
        if (handle == envHumidityCharacteristic.getValueAttribute().getHandle() - BLE_HANDLE_VALUE_OFFSET) return true;
        return false;
    }

    bool isPresHandle (Gap::Handle_t handle) {
        if (handle == envPressureCharacteristic.getValueAttribute().getHandle() - BLE_HANDLE_VALUE_OFFSET) return true;
        return false;
    }
#endif
#ifdef DISAGGREGATED_MOT_CHAR        
    bool isMagHandle (Gap::Handle_t handle) {
        if (handle == envMagnetometerCharacteristic.getValueAttribute().getHandle() - BLE_HANDLE_VALUE_OFFSET) return true;
        return false;
    }
    bool isAccHandle (Gap::Handle_t handle) {
        if (handle == envAccelerometerCharacteristic.getValueAttribute().getHandle() - BLE_HANDLE_VALUE_OFFSET) return true;
        return false;
    }
    bool isGyroHandle (Gap::Handle_t handle) {
        if (handle == envGyroCharacteristic.getValueAttribute().getHandle() - BLE_HANDLE_VALUE_OFFSET) return true;
        return false;
    }
#endif    
    bool isAccGyroMagHandle (Gap::Handle_t handle) {
        if (handle == envAccGyroMagCharacteristic.getValueAttribute().getHandle() - BLE_HANDLE_VALUE_OFFSET) return true;
        return false;
    }
    bool isPresHumTempHandle (Gap::Handle_t handle) {
        if (handle == envPresHumTempCharacteristic.getValueAttribute().getHandle() - BLE_HANDLE_VALUE_OFFSET) return true;
        return false;
    }
    

    void updateConnectionStatus(ConnectionStatus_t status) {
#ifdef DISAGGREGATED_ENV_CHAR        
        isEnabledTempNotify             = false;
        isEnabledHumNotify              = false;
        isEnabledPresNotify             = false;
#endif
#ifdef DISAGGREGATED_MOT_CHAR                
        isEnabledGyroNotify             = false;
        isEnabledAccNotify              = false;
        isEnabledMagNotify              = false;
#endif        
        isEnabledPresHumTempNotify      = false;
        isEnabledAccGyroMagNotify       = false;

        isTempCalibrated                = false;
        isHumCalibrated                 = false;
        isPresCalibrated                = false;
        isMagCalibrated                 = false;
        isAccCalibrated                 = false;
        isAGyroCalibrated               = false;

        memset (pastenvTemperature, 0, TEMP_DATA_LEN);
        memset (pastenvHumidity, 0, HUM_DATA_LEN);
        memset (pastenvPressure, 0, PRES_DATA_LEN);
        isBTLEConnected = status;
    }


private:
    BLE                  &ble;
    uint8_t              envTemperature     [TEMP_DATA_LEN];                /* in C  */
    uint8_t              pastenvTemperature [TEMP_DATA_LEN];
    uint8_t              envHumidity        [HUM_DATA_LEN];                     /* in %  */
    uint8_t              pastenvHumidity    [HUM_DATA_LEN];
    uint8_t              envPressure        [PRES_DATA_LEN];                        /* in mBar */
    uint8_t              pastenvPressure    [PRES_DATA_LEN];
    uint8_t              envMagn            [MAG_DATA_LEN];
    uint8_t              envGyro            [GYRO_DATA_LEN];
    uint8_t              envAcce            [ACC_DATA_LEN];
    uint8_t              envAccGyroMag      [ACCGYROMAG_DATA_LEN];
    uint8_t              envPresHumTemp     [PRESHUMTEMP_DATA_LEN];    
#ifdef DISAGGREGATED_ENV_CHAR
    GattCharacteristic   envTemperatureCharacteristic;
    GattCharacteristic   envHumidityCharacteristic;
    GattCharacteristic   envPressureCharacteristic;
#endif
#ifdef DISAGGREGATED_MOT_CHAR                    
    GattCharacteristic   envMagnetometerCharacteristic;
    GattCharacteristic   envAccelerometerCharacteristic;
    GattCharacteristic   envGyroCharacteristic;
#endif    
    GattCharacteristic   envAccGyroMagCharacteristic;
    GattCharacteristic   envPresHumTempCharacteristic;    

    ConnectionStatus_t   isBTLEConnected;
#ifdef DISAGGREGATED_ENV_CHAR
    bool                 isEnabledTempNotify;
    bool                 isEnabledHumNotify;
    bool                 isEnabledPresNotify;
#endif    
#ifdef DISAGGREGATED_MOT_CHAR                
    bool                 isEnabledGyroNotify;
    bool                 isEnabledAccNotify;
    bool                 isEnabledMagNotify;
#endif    
    bool                 isEnabledAccGyroMagNotify;
    bool                 isEnabledPresHumTempNotify;

    bool                 isTempCalibrated;
    bool                 isHumCalibrated;
    bool                 isPresCalibrated;
    bool                 isMagCalibrated;
    bool                 isAccCalibrated;
    bool                 isAGyroCalibrated;
};

#endif /* #ifndef __CUSTOM_BLE_SENSORS_SERVICE_H__*/
