/**
 ******************************************************************************
 * @file    main.cpp
 * @author  Davide Aliprandi, STMicroelectronics
 * @version V1.0.0
 * @date    November 25th, 2016
 * @brief   mbed test application which combines the STMicroelectronics
 *          X-NUCLEO-CCA01M1 and the X-NUCLEO-CCA02M1 Expansion Boards.
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2016 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */


/* Includes ------------------------------------------------------------------*/

/* mbed specific header files. */
#include "mbed.h"
#include "rtos.h"


#if DEVICE_I2S

/* Helper header files. */
#include "DevI2C.h"

/* Component specific header files. */
#include "STA350BW.h"

/* Expansion Board specific header files. */
#include "XNucleoCCA02M1.h"


/* Variables -----------------------------------------------------------------*/

/* Initialization parameters. */
STA350BW_init_t sound_terminal_init =
{
    32000,   /* Default Sampling Frequency [Hz]. */
    100      /* Default Volume. */
};
XNucleoCCA02M1_init_t microphones_init =
{
    32000,   /* Default Sampling Frequency [Hz]. */
    2        /* Default number of channels. */
};

/* Sound Terminal Component. */
STA350BW *sound_terminal;

/* MEMS Microphones Expansion Board. */
XNucleoCCA02M1 *microphones;

/* Thread to manage I2S peripherals. */
static rtos::Thread i2s_bh_daemon;


/* Functions -----------------------------------------------------------------*/

/**
 * @brief  Callback to start playing from the sound terminal the PCM buffer
 *         filled by the microphones.
 * @param  PCM_buffer       Buffer containing the PCM data to play.
 * @param  PCM_buffer_bytes Size of the buffer.
 * @retval None.
 */
void StartPlaying(int16_t *PCM_buffer, uint16_t PCM_buffer_bytes)
{
    static bool first_time = true;
    if (first_time)
    {
        /* Start playing from the sound terminal. */
        sound_terminal->play(PCM_buffer, PCM_buffer_bytes, true);
        first_time = false;
    }
}

/**
 * @brief  Entry point function of mbedOS.
 * @param  None
 * @retval None
 */
int main(void)
{
    /*----- Initialization. -----*/

    /* Initializing I2C bus. */
    DevI2C *dev_i2c = new DevI2C(PB_9, PB_8);

    /* Initializing Sound Terminal Component on the I2S2 interface. */
    sound_terminal = new STA350BW(PA_10, STA350BW_ADDRESS_2, *dev_i2c, PC_12, PC_10, PA_4, NC, PC_7);
    if (sound_terminal->init(&sound_terminal_init) != COMPONENT_OK)
    {
        error("Initialization of the Sound Terminal Expansion Board failed.\r\n");
        exit(EXIT_FAILURE);
    }

    /* Initializing MEMS Microphones Expansion Board on the I2S1 interface. */
    microphones = new XNucleoCCA02M1(PB_15, PB_13);
    if (microphones->init(&microphones_init) != COMPONENT_OK)
    {
        error("Initialization of the MEMS Microphones Expansion Board failed.\r\n");
        exit(EXIT_FAILURE);
    }

    /* Asking the I2S peripherals to work at frequencies one multiple of the
       other in order to avoid audio glitches that may be caused by a drift
       between the I2S clock signals.*/
    int ret = I2S::harmonize(sound_terminal->dev_i2s, microphones->dev_i2s);
    if (ret != 0)
        error("Unable to synchronize audio frequencies.\r\n");

    /* Starting a thread to manage I2S peripherals. */
    Callback<void()> i2s_bh_task(&I2S::i2s_bh_queue, &events::EventQueue::dispatch_forever);
    i2s_bh_daemon.start(i2s_bh_task);

    /* Printing to the console. */
    printf("Audio Karaoke Application Example\r\n\n");


    /*----- Recording and playing. -----*/

    /* Printing to the console. */
    printf("--> Sing a song and have fun!\r\n");

    /* Attaching a callback which will be called each time the microphones have
       recorded 1 ms of audio data, and start recording. */
    microphones->attach(&StartPlaying);
    microphones->record();
}

#else // DEVICE_I2S

int main(void)
{
    printf("The target does not support I2S API.\r\n");
}

#endif // DEVICE_I2S
