/**
  ******************************************************************************
  * @file       main.cpp
  * @author     ST Central Labs
  * @version V2.0.0
  * @date    28 Apr 2017
  * @brief      This demo writes an ndef message with an url inside.
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */

#include "mbed.h"
#include "XNucleoNFC01A1.h"
#include "NDefLib/NDefNfcTag.h"
#include "NDefLib/RecordType/RecordURI.h"

/**
 * Write a Ndef URI message linking to st.com site.
 * Write an NDef message with a Uri record linking the st.com site
 * @param nfcNucleo expansion board where write the NDef message
 */  
static void write_url(XNucleoNFC01A1 *nfcNucleo) {
    //retrieve the NdefLib interface
    NDefLib::NDefNfcTag& tag = nfcNucleo->get_M24SR().get_NDef_tag();
    
    //open the i2c session with the nfc chip
    if (tag.open_session()) {
        printf("Session opened\n\r");
        nfcNucleo->get_led1()=1;
        
        //create the NDef message and record
        NDefLib::Message msg;
        NDefLib::RecordURI rUri(NDefLib::RecordURI::HTTP_WWW,"st.com");
        msg.add_record(&rUri);

        //write the tag
        if (tag.write(msg)) {
            printf("Tag written\n\r");
            nfcNucleo->get_led2()=1;
        } else {
            printf("Error writing \n\r");
        }//if-else

        //close the i2c session
        if (tag.close_session()) {
           printf("Session closed\n\r");
           nfcNucleo->get_led3()=1;
        } else {
           printf("Error closing the session\n\r");
        }//if-else
            
    } else {
        printf("Error opening the session\n\r");
    }
}

/**
 * Read a NDef message and print content and type of all the uri record inside 
 * the message
 * @param nfcNucleo expansion board from where read the message
 */
static void read_url(XNucleoNFC01A1 *nfcNucleo) {
    //retrieve the NdefLib interface
    NDefLib::NDefNfcTag& tag = nfcNucleo->get_M24SR().get_NDef_tag();
    
    //open the i2c session with the nfc chip
    if (tag.open_session()) {
        printf("Session opened\n\r");
        nfcNucleo->get_led1() = 1;
        
        //create the NDef message and record
        NDefLib::Message msg;
        
        //read the tag
        if (tag.read(&msg)) {
            const uint32_t nRecords = msg.get_N_records();
            printf("Tag read: %d record\n\r", msg.get_N_records());
            for (int i =0 ;i<nRecords ;i++) {
                if (msg[i]->get_type()== NDefLib::Record::TYPE_URI) {
                    NDefLib::RecordURI *rUri = (NDefLib::RecordURI *)msg[i];
                    printf("UriType: %x\n\rUriContent: %s\n\r",
                        rUri->get_uri_id(),
                        rUri->get_content().c_str());
                }//if
            }//for
            //free the read records
            NDefLib::Message::remove_and_delete_all_record(msg);
        } else {
            printf("Error Reading \n\r");
        }//if-else

        //close the i2c session
        if (tag.close_session()) {
           printf("Session closed\n\r");
           nfcNucleo->get_led3() = 1;
        } else {
           printf("Error closing the session\n\r");
        }//if-else
            
    } else {
        printf("Error opening the session\n\r");
    }
}

static volatile bool buttonPress=false;

static void set_button_press(){
    buttonPress=true;
}//if buttonPress

/**
 * Write a Ndef URI message linking to st.com site and than read the message from
 * the Nfc tag.
 * 
 */
int main(void){
    
    //use default board pinout
    I2C i2cChannel(XNucleoNFC01A1::DEFAULT_SDA_PIN,XNucleoNFC01A1::DEFAULT_SDL_PIN);
    XNucleoNFC01A1 *nfcNucleo = XNucleoNFC01A1::instance(i2cChannel,NULL,
        XNucleoNFC01A1::DEFAULT_GPO_PIN,XNucleoNFC01A1::DEFAULT_RF_DISABLE_PIN,
        XNucleoNFC01A1::DEFAULT_LED1_PIN,XNucleoNFC01A1::DEFAULT_LED2_PIN,
        XNucleoNFC01A1::DEFAULT_LED3_PIN);
    
    printf("System Init done: !\n\r");
    
    // write an URI message
    write_url(nfcNucleo);
    
//if run on a nucleo board enable the user button for read the ndef record
#if defined(TARGET_STM)
    InterruptIn userButton(USER_BUTTON);    
    userButton.fall(set_button_press);
     while(true){
        if (buttonPress) {
            read_url(nfcNucleo);
            buttonPress=false;
        }//if
        //wait next event
        __WFE();
    }//while
#else
    read_url(nfcNucleo);
#endif
}


/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/ 
