/**
  ******************************************************************************
  * @file       main.cpp
  * @author     ST Central Labs
  * @version    V1.0.0
  * @date       21 Dic 2015
  * @brief      This demo writes an ndef message with an url inside.
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */

#include "mbed.h"

#include "X_NUCLEO_NFC01A1.h"
#include "NDefLib/NDefNfcTag.h"
#include "NDefLib/RecordType/RecordURI.h"

/**
 * Write an NDef message with a Uri record linking the st.com site
 * @param nfcNucleo expansion board where write the NDef message
 */
static void writeUrl(X_NUCLEO_NFC01A1 *nfcNucleo){
    NDefLib::NDefNfcTag& tag = nfcNucleo->getM24SR().getNDefTag();
    //open the i2c session with the nfc chip
    if(tag.openSession()){
        printf("Session opened\n\r");
        nfcNucleo->getLed1()=1;
        
        //create the NDef message and record
        NDefLib::Message msg;
        NDefLib::RecordURI rUri(NDefLib::RecordURI::HTTP_WWW,"st.com");
        msg.addRecord(&rUri);

        //write the tag
        if(tag.write(msg)){
            printf("Tag written\n\r");
            nfcNucleo->getLed2()=1;
        }else{
            printf("Error writing \n\r");
        }//if-else

        //close the i2c session
        if(tag.closeSession()){
           printf("Session closed\n\r");
           nfcNucleo->getLed3()=1;
        }else{
           printf("Error closing the session\n\r");
        }//if-else
            
    }else
        printf("Error opening the session\n\r");
}

/**
 * Read a NDef message and print content and type of all the uri record inside 
 * the message
 * @param nfcNucleo expansion board from where read the message
 */
static void readUrl(X_NUCLEO_NFC01A1 *nfcNucleo){
    NDefLib::NDefNfcTag& tag = nfcNucleo->getM24SR().getNDefTag();
    //open the i2c session with the nfc chip
    if(tag.openSession()){
        printf("Session opened\n\r");
        nfcNucleo->getLed1()=1;
        
        //create the NDef message and record
        NDefLib::Message msg;
        
        //read the tag
        if(tag.read(&msg)){
            const uint32_t nRecords = msg.getNRecords();
            printf("Tag read: %d record\n\r",msg.getNRecords());
            for(int i =0 ;i<nRecords ;i++){
                if(msg[i]->getType()== NDefLib::Record::TYPE_URI){
                    NDefLib::RecordURI *rUri = (NDefLib::RecordURI *)msg[i];
                    printf("UriType: %x\n\rUriContent: %s\n\r",
                        rUri->getUriId(),
                        rUri->getContent().c_str());
                }//if
            }//for
            //free the read records
            NDefLib::Message::removeAndDeleteAllRecord(msg);
        }else{
            printf("Error Reading \n\r");
        }//if-else

        //close the i2c session
        if(tag.closeSession()){
           printf("Session closed\n\r");
           nfcNucleo->getLed3()=1;
        }else{
           printf("Error closing the session\n\r");
        }//if-else
            
    }else
        printf("Error opening the session\n\r");
}

static volatile bool buttonPress=false;

static void setButtonPress(){
    buttonPress=true;
}//if buttonPress

/**
 * Write a Ndef URI message linking to st.com site.
 */
int main(void){
    
    //use default board pinout
    I2C i2cChannel(X_NUCLEO_NFC01A1::DEFAULT_SDA_PIN,X_NUCLEO_NFC01A1::DEFAULT_SDL_PIN);
    X_NUCLEO_NFC01A1 *nfcNucleo = X_NUCLEO_NFC01A1::Instance(i2cChannel,NULL,
            X_NUCLEO_NFC01A1::DEFAULT_GPO_PIN,X_NUCLEO_NFC01A1::DEFAULT_RF_DISABLE_PIN,
            X_NUCLEO_NFC01A1::DEFAULT_LED1_PIN,X_NUCLEO_NFC01A1::DEFAULT_LED2_PIN,
            X_NUCLEO_NFC01A1::DEFAULT_LED3_PIN);
    
    //retrieve the NdefLib interface
    NDefLib::NDefNfcTag& tag =nfcNucleo->getM24SR().getNDefTag();
    printf("System Init done: !\n\r");
    
    // write an URI message
    writeUrl(nfcNucleo);
    
//if run on a nucleo board enable the user button for read the ndef record
#if defined(TARGET_STM)
    InterruptIn userButton(USER_BUTTON);    
    userButton.fall(setButtonPress);
     while(true){
        if(buttonPress){
            readUrl(nfcNucleo);
            buttonPress=false;
        }//if
        //wait next event
        __WFE();
    }//while
#else
    readUrl(nfcNucleo);
#endif
}