/**
 ******************************************************************************
 * @file    main.cpp
 * @author  Davide Aliprandi, STMicroelectronics
 * @version V1.0.0
 * @date    February 4h, 2016
 * @brief   mbed test application for the STMicroelectronics X-NUCLEO-LED61A1
 *          LED expansion board.
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */


/* Includes ------------------------------------------------------------------*/

/* mbed specific header files. */
#include "mbed.h"

/* Component specific header files. */
#include "led6001_class.h"


/* Definitions ---------------------------------------------------------------*/

/* PI. */
#ifndef M_PI
    #define M_PI                          (3.14159265358979323846f)
#endif

/* Loop period in micro-seconds. */
#define LOOP_PERIOD_us                    (5E5) /* 0.5 seconds. */

/* Sin period in micro-seconds. */
#define PWM_SIN_PERIOD_us                 (1E7) /* 10 seconds. */


/* Variables -----------------------------------------------------------------*/

/* Main loop's ticker. */
static Ticker ticker;

/* LED Control Component. */
LED6001 *led;

/* Interrupt flags. */
static volatile bool ticker_irq_triggered = false;
static volatile bool xfault_irq_triggered = false;


/* Functions -----------------------------------------------------------------*/

/**
 * @brief  Handling the LED capabilities.
 * @param  None.
 * @retval None.
 */
void LEDHandler(void)
{
    static int tick = 0;

    /* Handling the LED dimming when powered ON. */
    float pwm_dimming = 0.5f * sin(2 * M_PI * (tick++ * LOOP_PERIOD_us) / PWM_SIN_PERIOD_us) + 0.5f;
    tick %= (int) (PWM_SIN_PERIOD_us / LOOP_PERIOD_us);

    /* Printing to the console. */
    printf("Sinusoidal PWM Dimming --> %0.2f\r", pwm_dimming);

    /*
       Writing PWM dimming values to the LED.

       Notes:
         + Replace "SetPWMDimming()" with "SetAnalogDimming()" for an analog control.
    */
    led->SetPWMDimming(pwm_dimming);
}

/**
 * @brief  Interrupt Request for the main loop's ticker related interrupt.
 * @param  None.
 * @retval None.
 */
void TickerIRQ(void)
{
    ticker_irq_triggered = true;
}

/**
 * @brief  Interrupt Request for the component's XFAULT interrupt.
 * @param  None.
 * @retval None.
 */
void XFaultIRQ(void)
{
    xfault_irq_triggered = true;
    led->DisableXFaultIRQ();
}

/**
 * @brief  Interrupt Handler for the component's XFAULT interrupt.
 * @param  None.
 * @retval None.
 */
void XFaultHandler(void)
{
    /* Printing to the console. */
    printf("XFAULT Interrupt detected! Re-initializing LED driver...");

    /* Re-starting-up LED Control Component. */
    led->StartUp();

    /* Printing to the console. */
    printf("Done.\r\n\n");

    led->EnableXFaultIRQ();
}


/* Main ----------------------------------------------------------------------*/

int main()
{
    /*----- Initialization. -----*/

    /* Printing to the console. */
    printf("LED Control Application Example\r\n\n");

    /* Initializing LED Control Component. */
    led = new LED6001(D4, A3, D7, D5);
    if (led->Init() != COMPONENT_OK)
        exit(EXIT_FAILURE);

    /* Attaching interrupt request functions. */
    led->AttachXFaultIRQ(&XFaultIRQ);
    led->EnableXFaultIRQ();
    ticker.attach_us(TickerIRQ, LOOP_PERIOD_us);

    /* Starting-up LED Control Component. */
    led->StartUp();


    /*----- LED Control. -----*/

    /* Either performing the component handler, interrupt handlers, or waiting for events. */
    while (true)
    {
        if (ticker_irq_triggered)
        {
            ticker_irq_triggered = false;
            LEDHandler();
        } else if (xfault_irq_triggered)
        {
            xfault_irq_triggered = false;
            XFaultHandler();
        } else {
            /* It is recommended that SEVONPEND in the System Control Register is NOT set. */
            __WFE();
        }
    }
}
