/**
 ******************************************************************************
 * @file    main.cpp
 * @author  APG Mass Market
 * @version V1.0.1
 * @date    16-Nov-2015
 * @brief   Example application for using the X_NUCLEO_IPS02A1 
 *          Intelligent Power Switch Nucleo expansion board.
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 *  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
*/ 

/**
 * @mainpage X_NUCLEO_IPS02A1 Intelligent Power Switch Nucleo Expansion Board Firmware Package
 *
 * <b>Introduction</b>
 *
 * This firmware package includes Components Device Drivers, Board Support Package
 * and example application for STMicroelectronics X_NUCLEO_IPS02A1 Intelligent Power Switch
 * Nucleo Expansion Board
 * 
 * <b>Example Application</b>
 *
 */
/*** Includes ----------------------------------------------------------------- ***/
#include "mbed.h"
#include "assert.h"
#include "x_nucleo_ips02a1.h"

/*** Static variables --------------------------------------------------------- ***/
#ifdef DBG_MCU
#include "DbgMCU.h"
static DbgMCU enable_dbg;
#endif // DBG_MCU

/* HW settings */
/* Pay attention before changing HW settings, they must be coherent with you HW design */
/* Power Switch Connection to Arduino connectors */
#define IPS02A1_PIN_IN_1             (D5)
#define IPS02A1_PIN_IN_2             (D6)
#define IPS02A1_PIN_FR_STBY          (D4)
#define IPS02A1_PIN_CURRENTSENSE1    (A2)
#define IPS02A1_PIN_CURRENTSENSE2    (A3)


/* V-Ref */
#define V_REF 3.3
/* Rsense Value */
#define R_SENSE 1e3
/* R_D1 */
#define R_D1 56e3
/* R_D2 */
#define R_D2 36e3

/* End of HW settings */

static X_NUCLEO_IPS02A1 &ips_expansion_board = X_NUCLEO_IPS02A1::Instance(IPS02A1_PIN_IN_1,
                                                                          IPS02A1_PIN_IN_2,
                                                                          IPS02A1_PIN_FR_STBY,
                                                                          IPS02A1_PIN_CURRENTSENSE1,
                                                                          IPS02A1_PIN_CURRENTSENSE2,                                                                         
                                                                          V_REF,
                                                                          R_SENSE,
                                                                          R_D1,
                                                                          R_D2);
    
                          
static Ticker ticker;
DigitalOut UserLed(LED1);    

/*** Main function ------------------------------------------------------------- ***/
/* Generic main function/loop, interrupt based cyclic execution
*/

float Multisense_Signal = 0;    // Multisense pin - signal level
bool  ButtonPressed = 0;         // User Button
int   TestSequence = 1;           // Test sequence counter

//------------------------------------
// Hyperterminal configuration
// 9600 bauds, 8-bit data, no parity
//------------------------------------

InterruptIn UserButton(USER_BUTTON);  // B1 is the User Button
void B1_pressed (void);
void LedBlink (int TestSequence);
void Write_Serial (void);
void Reset_Pins (void);


int main(void){
    UserButton.fall(&B1_pressed);       //interrupt User Button
   
    printf("############################################################ \n\r");
    printf("###################  TEST PROCEDURE ######################## \n\r");
    printf("############################################################ \n\n\r");
    printf("This demo performs current measurements on Ch1 and Ch2  \n\r");
    printf("in the following conditions: \n\r\n\r");
    printf(" 1) Ch1 OFF, Ch2 OFF \n\r");
    printf(" 2) Ch1 ON,  Ch2 OFF \n\r");
    printf(" 3) Ch1 OFF, Ch2 ON \n\r");
    printf(" 4) Ch1 ON,  Ch2 ON \n\r\n\r");
      
    printf("Start test Procedure.... \n\r\n\r");
    printf("PRESS USER BUTTON (Blue One) on NUCLEO to perform single test \n\r\n\r\n\r");
    
    
    while (true) {
        // wait for User button is pressed 
        while (!ButtonPressed) {
        }
           
        ButtonPressed = 0;
        
        LedBlink(TestSequence);
     
        switch (TestSequence) {
          case (1):
            printf("############################################################ \n\r");
            printf("###################  TEST PROCEDURE ######################## \n\r");
            printf("############################################################ \n\n\r");
            printf("This demo performs current measurements on Ch1 and Ch2  \n\r");
            printf("in the following conditions: \n\r\n\r");
            printf(" 1) Ch1 OFF, Ch2 OFF \n\r");
            printf(" 2) Ch1 ON,  Ch2 OFF \n\r");
            printf(" 3) Ch1 OFF, Ch2 ON \n\r");
            printf(" 4) Ch1 ON,  Ch2 ON \n\r\n\r");
            printf("\n\r\n\r");
            break;
          case (2): {
            printf("Test 1: StandBy\n\r");
            Reset_Pins();
            wait (0.1);
            Write_Serial();
            Multisense_Signal= ips_expansion_board.GetCurrent(CHANNEL_1);
            printf("Current Ch1 = %2.3fA \n\r", Multisense_Signal);
            Multisense_Signal= ips_expansion_board.GetCurrent(CHANNEL_2);
            printf("Current Ch2 = %2.3fA \n\r", Multisense_Signal);
            printf("\n\r\n\r");
           }
           break;

          
          case(3):{
            printf("Test 2: Ch1=ON, CH2=OFF\n\r");
            ips_expansion_board.vps235h2.In_1 = 1;
            ips_expansion_board.vps235h2.In_2 = 0;
            ips_expansion_board.vps235h2.Fr_Stby = 1;
           
            wait (0.1);
            Write_Serial();
            Multisense_Signal= ips_expansion_board.GetCurrent(CHANNEL_1);
            printf("Current Ch1 = %2.3fA \n\r", Multisense_Signal);
            Multisense_Signal= ips_expansion_board.GetCurrent(CHANNEL_2);
            printf("Current Ch2 = %2.3fA \n\r", Multisense_Signal);
            wait (0.5);
            Reset_Pins();
            printf("\n\r\n\r");
           }
           break;

          case(4):{
            printf("Test 3: Ch1=OFF, CH2=ON\n\r");
            ips_expansion_board.vps235h2.In_1 = 0;
            ips_expansion_board.vps235h2.In_2 = 1;
            ips_expansion_board.vps235h2.Fr_Stby = 1;

            wait (0.1);
            Write_Serial();
            Multisense_Signal= ips_expansion_board.GetCurrent(CHANNEL_1);
            printf("Current Ch1 = %2.3fA \n\r", Multisense_Signal);
            Multisense_Signal= ips_expansion_board.GetCurrent(CHANNEL_2);
            printf("Current Ch2 = %2.3fA \n\r", Multisense_Signal);
            wait (.5);
            Reset_Pins();
            printf("\n\r\n\r");
           }
           break;

          case(5):{
            printf("Test 4: Ch1=ON, CH2=ON  \n\r");
            ips_expansion_board.vps235h2.In_1= 1;
            ips_expansion_board.vps235h2.In_2 = 1;
            ips_expansion_board.vps235h2.Fr_Stby = 1;
            wait (0.1);
            Write_Serial();
            Multisense_Signal= ips_expansion_board.GetCurrent(CHANNEL_1);
            printf("Current Ch1 = %2.3fA \n\r", Multisense_Signal);
            Multisense_Signal= ips_expansion_board.GetCurrent(CHANNEL_2);
            printf("Current Ch2 = %2.3fA \n\r", Multisense_Signal);
            wait (.5);
            Reset_Pins();
            printf("\n\r\n\r");
           }
          break;

          default: {
            printf("End Test Cycle...press to user button to continue\n\n\n\r");  
            TestSequence = 0;
            Reset_Pins();
           }
          break;

        } 
        
    }
    
 }


void B1_pressed (){
  // Interrupt procedure - User button is pressed
    TestSequence ++;

    UserLed = 1; // LED is ON
    wait(0.05); // 50 ms
    UserLed = 0; // LED is OFF

    ButtonPressed = 1;

}

void LedBlink (int TestSequence){
// Option feedback by usingUser LED
    for (int TestCounter =0; TestCounter<TestSequence; TestCounter++) {
        UserLed = 1; // LED is ON
        wait(0.05); // 50 ms
        UserLed = 0; // LED is OFF
        wait(0.05); // 50 msec
    }
    wait(1-(TestSequence*2*0.05));
}

void Write_Serial (){
// This code send messages and data to the serial port
// send info to serial port

    printf("Input 1= %d\t", ips_expansion_board.vps235h2.In_1.read());
    printf("Input 2= %d\t", ips_expansion_board.vps235h2.In_2.read());
    printf("Fr_Stby= %d\t\n\r", ips_expansion_board.vps235h2.Fr_Stby.read());
    

 }
 
 void Reset_Pins(){
// reset input pins to  
   ips_expansion_board.vps235h2.In_1= 0;
   ips_expansion_board.vps235h2.In_2 = 0;
   ips_expansion_board.vps235h2.Fr_Stby = 0;
}


