/**
 ******************************************************************************
 * @file    main.cpp
 * @author  IPC Rennes
 * @version V1.0.0
 * @date    May 26th, 2016
 * @brief   mbed simple application for the STMicroelectronics X-NUCLEO-IHM06A1
 *          Motor Control Expansion Board: control of 1 motor.
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2016 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/

/* mbed specific header files. */
#include "mbed.h"

/* Component specific header files. */
#include "STSpin220.h"

/* Definitions ---------------------------------------------------------------*/
#ifdef TARGET_NUCLEO_F302R8
#define PWM_REF_PIN D11 /* HW mandatory patch: bridge manually D9 with D10 */
#else
#define PWM_REF_PIN D9
#endif
/* Uncomment the line below to enable the step clock monitoring duration */
//#define STEP_CLOCK_HANDLER_DURATION_MONITORING
#ifdef STEP_CLOCK_HANDLER_DURATION_MONITORING
#define MONITORING_PIN D15
#endif

/* Variables -----------------------------------------------------------------*/

/* Initialization parameters of the motor connected to the expansion board. */
STSpin220_init_t init =
{
  10000,           //Acceleration rate in pulse/s2 (must be greater than 0)
  10000,           //Deceleration rate in pulse/s2 (must be greater than 0)
  10000,           //Running speed in pulse/s (8 pulse/s < Maximum speed <= 10 000 pulse/s )
  400,             //Minimum speed in pulse/s (8 pulse/s <= Minimum speed < 10 000 pulse/s)
  20,              //Acceleration current torque in % (from 0 to 100)
  15,              //Deceleration current torque in % (from 0 to 100)
  10,              //Running current torque in % (from 0 to 100)
  25,              //Holding current torque in % (from 0 to 100)
  TRUE,            //Torque boost speed enable
  200,             //Torque boost speed threshold in fullstep/s
  STEP_MODE_1_32,  //Step mode via enum motorStepMode_t  
  HIZ_MODE,        //Automatic HIZ STOP
  100000           //REF frequency (Hz)
};

/* Motor Control Component. */
STSpin220 *motor;

/* Functions -----------------------------------------------------------------*/

/**
 * @brief  This is an example of user handler for the flag interrupt.
 * @param  None
 * @retval None
 * @note   If needed, implement it, and then attach and enable it:
 *           + motor->attach_flag_irq(&my_flag_irq_handler);
 *           + motor->enable_flag_irq();
 *         To disable it:
 *           + motor->DisbleFlagIRQ();
 */
void my_flag_irq_handler(void)
{
  printf("    WARNING: \"FLAG\" interrupt triggered:\r\n");
  motor->disable();
  printf("    Motor disabled.\r\n\n");
}

/**
 * @brief  This is an example of error handler.
 * @param[in] error Number of the error
 * @retval None
 * @note   If needed, implement it, and then attach it:
 *           + motor->attach_error_handler(&my_error_handler);
 */
void my_error_handler(uint16_t error)
{
  /* Printing to the console. */
  printf("Error %d detected\r\n\n", error);
  
  /* Infinite loop */
  while (true) {
  }    
}

/* Main ----------------------------------------------------------------------*/

int main()
{
  /* Printing to the console. */
  printf("STARTING MAIN PROGRAM\r\n");
  printf("    Reminder:\r\n");
  printf("    The position, speed, acceleration and deceleration units\r\n");
  printf("    are in agreement to the step mode.\r\n");
  printf("    In example if the step mode is 1/32th,\r\n");
  printf("    the position is in 1/32th step,\r\n");
  printf("    the speed is in 1/32th step/s,\r\n");
  printf("    the acceleration and the deceleration are in 1/32th step/s^2.\r\n");
    
//----- Initialization 
  /* Initializing Motor Control Component. */
#ifdef STEP_CLOCK_HANDLER_DURATION_MONITORING
  motor = new STSpin220(D2, D8, D7, D5, D10, D3, PWM_REF_PIN, MONITORING_PIN);
#else
  motor = new STSpin220(D2, D8, D7, D5, D10, D3, PWM_REF_PIN);
#endif

  if (motor->init(&init) != COMPONENT_OK) {
    exit(EXIT_FAILURE);
  }

  /* Attaching and enabling an interrupt handler. */
  motor->attach_flag_irq(&my_flag_irq_handler);
  motor->enable_flag_irq();
    
  /* Attaching an error handler */
  motor->attach_error_handler(&my_error_handler);

  /* Printing to the console. */
  printf("Motor Control Application Example for 1 Motor\r\n");
  
//----- Get the step mode after initialization 
  StepperMotor::step_mode_t myStepMode = motor->get_step_mode();

//----- run the motor BACKWARD
  printf("--> Running the motor backward.\r\n");
  motor->run(StepperMotor::BWD);
  
  while (motor->get_status()!=STEADY)
  {
    /* Print reached speed to the console in step/s or microsteps/s */
    printf("    Reached Speed: %d step/s.\r\n", motor->get_speed());
    wait_ms(50);    
  }
  printf("    Reached Speed: %d step/s.\r\n", motor->get_speed());
     
  /* Wait for 1 second */  
  wait_ms(1000);
  
//----- Decrease speed while running to one quarter of the previous speed
  printf("    Motor init step mode: %d\r\n", myStepMode);
  int currentStepMode = motor->get_step_mode();
  printf("    Motor current step mode: %d\r\n", currentStepMode);
  int newSpeed = (motor->get_speed()<<(myStepMode-currentStepMode))>>2;
  printf("    Set motor max speed to: %d step/s.\r\n", newSpeed);
  if (!motor->set_max_speed(newSpeed)) {
    printf("    Failed: target speed below min speed.\r\n");
    if (motor->set_max_speed(motor->get_min_speed())) {
      printf("    Motor max speed set to min speed: %d step/s.\r\n", motor->get_min_speed());
    } else {
      printf("    Failed: check all speed and boost setting.\r\n");
    }
  }
  
  /* Wait until the motor starts decelerating */
  while (motor->get_status()==STEADY);
  /* Wait and print speed while the motor is not steady running */
  while (motor->get_status()!=STEADY) {
    /* Print reached speed to the console in step/s or microsteps/s */
    printf("    Reached Speed: %d step/s.\r\n", motor->get_speed());
    wait_ms(50);    
  }
  printf("    Reached Speed: %d step/s.\r\n", motor->get_speed());  

  /* Wait for 5 seconds */  
  wait_ms(5000);
  
//----- Soft stop required while running
  printf("--> Soft stop requested.\r\n");
  motor->soft_stop();
  
  /* Wait for the motor of device ends moving */  
  motor->wait_while_active();

  /* Wait for 2 seconds */
  wait_ms(2000);

//----- Change step mode to full step mode
  motor->set_step_mode(StepperMotor::STEP_MODE_FULL);
  printf("    Motor step mode: %d\r\n", motor->get_step_mode());
  printf("    0:FS      1:1/2     2:1/4\r\n    3:1/8     4:1/16    5:1/32\r\n");
  printf("    6:1/64    7:1/128   8:1/256\r\n");
    
  /* Get current position of device and print to the console */
  printf("    Position: %d.\r\n", motor->get_position());
  
  /* Set speed, acceleration and deceleration to scale with full step mode */
  motor->set_min_speed(init.minSpeed>>myStepMode);  
  motor->set_max_speed(init.maxSpeed>>myStepMode);
  motor->set_acceleration(motor->get_acceleration()>>myStepMode);
  motor->set_deceleration(motor->get_deceleration()>>myStepMode);
  /* Print parameters to the console */
  printf("    Motor Min Speed: %d step/s.\r\n", motor->get_min_speed());  
  printf("    Motor Max Speed: %d step/s.\r\n", motor->get_max_speed());
  printf("    Motor Acceleration: %d step/s.\r\n", motor->get_acceleration());
  printf("    Motor Deceleration: %d step/s.\r\n", motor->get_deceleration());
  
//----- move of 200 steps in the FW direction
  printf("--> Moving forward 200 steps.\r\n");
  motor->move(StepperMotor::FWD, 200);
  
  /* Waiting while the motor is active. */
  motor->wait_while_active();
  
  /* Get current position of device and print to the console */
  printf("    Position: %d.\r\n", motor->get_position());
  
  /* Disable the power bridges */
  motor->disable();
  
  /* Check that the power bridges are actually disabled */
  if (motor->check_status_hw()!=0) {
    printf("    Motor driver disabled.\r\n");
  } else {
    printf("    Failed to disable the motor driver.\r\n");
  }
    
  /* Wait for 2 seconds */  
  wait_ms(2000);
  
//----- Restore step mode to its initialization value
  motor->set_step_mode((StepperMotor::step_mode_t)init.stepMode);
  printf("    Motor step mode: %d\r\n", motor->get_step_mode());
  printf("    0:FS      1:1/2     2:1/4\r\n    3:1/8     4:1/16    5:1/32\r\n");
  printf("    6:1/64    7:1/128   8:1/256\r\n");

  /* Set speed, acceleration and deceleration to scale with microstep mode */
  motor->set_max_speed(motor->get_max_speed()<<myStepMode);
  motor->set_min_speed(motor->get_min_speed()<<myStepMode);
  motor->set_acceleration(motor->get_acceleration()<<myStepMode);
  motor->set_deceleration(motor->get_deceleration()<<myStepMode);
  /* Print parameters to the console */  
  printf("    Motor Max Speed: %d step/s.\r\n", motor->get_max_speed());
  printf("    Motor Min Speed: %d step/s.\r\n", motor->get_min_speed());
  printf("    Motor Acceleration: %d step/s.\r\n", motor->get_acceleration());
  printf("    Motor Deceleration: %d step/s.\r\n", motor->get_deceleration());

  /* Get current position of device and print to the console */
  printf("    Position: %d.\r\n", motor->get_position());
  
//----- Change torque
  motor->set_torque(ACC_TORQUE,30);
  motor->set_torque(DEC_TORQUE,20);
  printf("    Motor acceleration and deceleration torque changed to: %d and %d.\r\n", motor->get_torque(ACC_TORQUE), motor->get_torque(DEC_TORQUE));   
  
//----- Go to position -6400
  printf("--> Go to position -6400 steps.\r\n");
  motor->go_to(-6400);
  
  /* Wait for the motor ends moving */
  motor->wait_while_active();

  /* Get current position of device and print to the console */
  printf("    Position: %d.\r\n", motor->get_position());
  
  /* Wait for 2 seconds */  
  wait_ms(2000);
  
//----- Go Home
  printf("--> Go to home position.\r\n");
  motor->go_home();
 
  /* Wait for the motor ends moving */
  motor->wait_while_active();
  
  /* Wait for 1 second */
  wait_ms(1000);
  
  /* Infinite Loop. */
  printf("--> Infinite Loop...\r\n");
  while (true) {
    /* Request device to go position -3200 */
    motor->go_to(-3200);
    /* Waiting while the motor is active. */
    motor->wait_while_active();
    /* Request device to go position 3200 */
    motor->go_to(3200);
    /* Waiting while the motor is active. */
    motor->wait_while_active();
    wait_ms(500);
  }
}
/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/
