/**
  ******************************************************************************
  * @file    Multi/Examples/MotionControl/IHM04A1_ExampleFor4UniDirMotors/Src/main.c 
  * @author  IPC Rennes
  * @version V1.0.0
  * @date    January 06, 2015
  * @brief   This example shows how to use 1 IHM04A1 expansion board with 
  * 4 unidirectionnal Brush DC motors.
  * Each motor has one lead connected to one of the bridge output, 
  * the other lead to the ground. The input bridges are not parallelised.
  * The demo sequence starts when the user button is pressed.
  * Each time, the user button is pressed, the demo step is changed
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */

/* Includes ------------------------------------------------------------------*/

/* mbed specific header files. */
#include "mbed.h"

/* Component specific header files. */
#include "l6206_class.h"



/* Definitions ---------------------------------------------------------------*/


#define MAX_MOTOR (4)



/* Variables -----------------------------------------------------------------*/

static volatile uint16_t gLastError;
static volatile uint8_t gStep = 0;



/* Variables -----------------------------------------------------------------*/

/* Initialization parameters. */
L6206_Init_t init =
{
    L6206_CONF_PARAM_PARALLE_BRIDGES,
    {L6206_CONF_PARAM_FREQ_PWM1A, L6206_CONF_PARAM_FREQ_PWM2A, L6206_CONF_PARAM_FREQ_PWM1B, L6206_CONF_PARAM_FREQ_PWM2B},
    {100,100,100,100},
    {FORWARD,FORWARD,BACKWARD,FORWARD},
    {INACTIVE,INACTIVE,INACTIVE,INACTIVE},
    {FALSE,FALSE}
};

/* Motor Control Component. */
L6206 *motor;

/* User button on Nucleo board */
InterruptIn mybutton_irq(USER_BUTTON);



/* Functions -----------------------------------------------------------------*/

/**
  * @brief  This function is executed in case of error occurrence.
  * @param  error number of the error
  * @retval None
  */
void Error_Handler(uint16_t error)
{
  /* Backup error number */
  gLastError = error;
  
  /* Enter your own code here */
}

/**
* @brief  This function is the User handler for the flag interrupt
* @param  None
* @retval None
* @note   If needed, implement it, and then attach and enable it:
*           + motor->AttachFlagInterrupt(MyFlagInterruptHandler);
*/
void MyFlagInterruptHandler(void)
{
  /* Code to be customised */
  /************************/
  /* Get the state of bridge A */
  uint16_t bridgeState  = motor->CmdGetStatus(0);
  
  if (bridgeState == 0) 
  {
    if ((motor->GetDeviceState(0) != INACTIVE)||
        (motor->GetDeviceState(1) != INACTIVE))
    {
      /* Bridge A was disabling due to overcurrent or over temperature */
      /* When at least on of its  motor was running */
        Error_Handler(0XBAD0);
    }
  }
  
  /* Get the state of bridge B */
  bridgeState  = motor->CmdGetStatus(1);
  
  if (bridgeState == 0) 
  {
    if ((motor->GetDeviceState(2) != INACTIVE)||
        (motor->GetDeviceState(3) != INACTIVE))
    {
      /* Bridge A was disabling due to overcurrent or over temperature */
      /* When at least on of its  motor was running */
        Error_Handler(0XBAD1);
    }
  }  
 }


/* Private functions ---------------------------------------------------------*/

/**
  * @brief  Button Irq
  * @param  None
  * @retval None
  */

void button_pressed(void)
{
    mybutton_irq.disable_irq();
    gStep++;
    if (gStep > MAX_MOTOR)
    {
        gStep = 0;
    }
    wait_ms(200);
    mybutton_irq.enable_irq();

}


/**
  * @brief  Main program
  * @param  None
  * @retval None
  */
int main(void)
{
    /*----- Initialization. -----*/

    /* Initializing Motor Control Component. */
    motor = new L6206( D2, A4, D5, D4, A0, A1);

    /* When Init method is called with NULL pointer, the L6206 parameters are set   */
    /* with the predefined values from file l6206_target_config.h, otherwise the    */
    /* parameters are set using the initDeviceParameters structure values.          */
    if (motor->Init(&init) != COMPONENT_OK)
        exit(EXIT_FAILURE);

    /* Attach the function MyFlagInterruptHandler (defined below) to the flag interrupt */
    motor->AttachFlagInterrupt(MyFlagInterruptHandler);

    /* Attach the function Error_Handler (defined below) to the error Handler*/
    motor->AttachErrorHandler(Error_Handler);

    /* Printing to the console. */
    printf("Motor Control Application Example for 1 Motor\r\n\n");


    /* Select the configuration with no bridge paralleling, two unidirectionnal motors on bridge A 
       and two unidirectionnal motors on bridge B */
    motor->SetDualFullBridgeConfig(PARALLELING_NONE___2_UNDIR_MOTOR_BRIDGE_A__2_UNDIR_MOTOR_BRIDGE_B);
 
    /* Set PWM Frequency of bridge A inputs to 10000 Hz */ 
    motor->SetBridgeInputPwmFreq(0,1000);
  
    /* Set PWM Frequency of bridge B inputs to 25000 Hz */ 
    motor->SetBridgeInputPwmFreq(1,2000);
  
    // Attach button_pressed function to Irq
    mybutton_irq.fall(&button_pressed);

    /* Infinite loop */
    while(1)
    {

        if (gStep > 0)
        {
            /* Set speed of motor 0 to 20 % */
            motor->SetMaxSpeed(0,20);
            /* start motor 0 */
            motor->Run(0, FORWARD);
        }

        if (gStep > 1)
        {
            /* Set speed of motor 1 to 30 % */
            motor->SetMaxSpeed(1,30);
            /* start motor 1 */
            motor->Run(1, FORWARD);
        }

        if (gStep > 2)
        {
            /* Set speed of motor 2 to 40 % */
            motor->SetMaxSpeed(2,40);
            /* start motor 2 */
            motor->Run(2, FORWARD);
        }

        if (gStep > 3)
        {
            /* Set speed of motor 3 to 50 % */
            motor->SetMaxSpeed(3,50);
            /* start motor 3 */
            motor->Run(3, FORWARD);      
        }

        if (gStep > 0)
        {
            wait_ms(1000);
        
            motor->CmdHardHiZ(0);   
            motor->CmdHardHiZ(1);   
            motor->CmdHardHiZ(2);   
            motor->CmdHardHiZ(3);
            
            wait_ms(1000);
        }
    }
}



/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/

