/**
 ******************************************************************************
 * @file    main.cpp
 * @author  Davide Aliprandi / AST
 * @version V1.0.0
 * @date    November 4th, 2015
 * @brief   mbed test application for the STMicrolectronics X-NUCLEO-IHM01A1
 *          Motor Control Expansion Board: control of 1 motor.
 *          This application makes use of a C++ component architecture obtained
 *          from the C component architecture through the Stm32CubeTOO tool.
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */


/* Includes ------------------------------------------------------------------*/

/* mbed specific header files. */
#include "mbed.h"

/* Helper header files. */
#include "DevSPI.h"

/* Expansion Board specific header files. */
#include "x_nucleo_ihm02a1_class.h"


/* Definitions ---------------------------------------------------------------*/

/* Number of movements per revolution. */
#define MPR_1 4
#define MPR_2 8

/* Delay in milliseconds. */
#define DELAY_1 500
#define DELAY_2 2000
#define DELAY_3 5000


/* Variables -----------------------------------------------------------------*/

/* Motor Control Expansion Board. */
X_NUCLEO_IHM02A1 *x_nucleo_ihm02a1;


/* Main ----------------------------------------------------------------------*/

int main()
{
    /* Led. */
    DigitalOut led(LED1);

    /* Initializing SPI bus. */
    DevSPI dev_spi(D11, D12, D3);

    /* Initializing Motor Control Expansion Board. */
    x_nucleo_ihm02a1 = X_NUCLEO_IHM02A1::Instance(A4, D4, A2, &dev_spi);

    /* Building a list of motors. */
    L6470 *motors[L6470DAISYCHAINSIZE] = {x_nucleo_ihm02a1->l6470_0, x_nucleo_ihm02a1->l6470_1};

    /* Printing to the console. */
    printf("Motor Control Application Example for 2 Motors\r\n\n");

    /* Main Loop. */
    while(true)
    {
        /*----- Doing a full revolution on each motor, one after the other. -----*/

        /* Printing to the console. */
        printf("--> Doing a full revolution on each motor, one after the other.\r\n");

        /* Doing a full revolution on each motor, one after the other. */
        for (int m = 0; m < 2; m++)
            for (int i = 0; i < MPR_1; i++)
            {
                /* Computing the number of steps. */
                int steps = (int) (((int) X_NUCLEO_IHM02A1::MotorParameterInitData[0][m].fullstepsperrevolution * pow(2.0f, X_NUCLEO_IHM02A1::MotorParameterInitData[0][m].step_sel)) / MPR_1);

                /* Moving. */
                motors[m]->Move(StepperMotor::FWD, steps);
                
                /* Waiting while active. */
                motors[m]->WaitWhileActive();

                /* Waiting. */
                wait_ms(DELAY_1);
            }

        /* Waiting. */
        wait_ms(DELAY_2);


        /*----- Running together for a certain amount of time. -----*/

        /* Printing to the console. */
        printf("--> Running together for %d seconds.\r\n", DELAY_3 / 1000);

        /* Preparing each motor to perform a run at a specified speed. */
        for (int m = 0; m < 2; m++)
            motors[m]->PrepareRun(StepperMotor::BWD, 10000);

        /* Performing the action on each motor at the same time. */
        for (int m = 0; m < 2; m++)
            motors[m]->PerformAction();

        /* Waiting. */
        wait_ms(DELAY_3);


        /*----- Hard Stop. -----*/

        /* Printing to the console. */
        printf("--> Hard Stop.\r\n");

        /* Preparing each motor to perform a hard stop. */
        for (int m = 0; m < 2; m++)
            motors[m]->PrepareHardStop();

        /* Performing the action on each motor at the same time. */
        for (int m = 0; m < 2; m++)
            motors[m]->PerformAction();

        /* Waiting. */
        wait_ms(DELAY_2);
        

        /*----- Doing a full revolution on each motor, one after the other. -----*/

        /* Printing to the console. */
        printf("--> Doing a full revolution on each motor, one after the other.\r\n");

        /* Doing a full revolution on each motor, one after the other. */
        for (int m = 0; m < 2; m++)
            for (int i = 0; i < MPR_2; i++)
            {
                /* Computing the number of steps. */
                int steps = (int) (((int) X_NUCLEO_IHM02A1::MotorParameterInitData[0][m].fullstepsperrevolution * pow(2.0f, X_NUCLEO_IHM02A1::MotorParameterInitData[0][m].step_sel)) / MPR_2);

                /* Moving. */
                motors[m]->Move(StepperMotor::FWD, steps);

                /* Waiting while active. */
                motors[m]->WaitWhileActive();

                /* Waiting. */
                wait_ms(DELAY_1);
            }

        /* Waiting. */
        wait_ms(DELAY_2);


        /*----- High Impedance State. -----*/

        /* Printing to the console. */
        printf("--> High Impedance State.\r\n");

        /* Preparing each motor to set High Impedance State. */
        for (int m = 0; m < 2; m++)
            motors[m]->PrepareHardHiZ();

        /* Performing the action on each motor at the same time. */
        for (int m = 0; m < 2; m++)
            motors[m]->PerformAction();

        /* Waiting. */
        wait_ms(DELAY_2);


        /*----- Led Blinking. -----*/
        
        /* Led Blinking. */
        led = 1;
        wait_ms(DELAY_2);
        led = 0;
    }
}