/**
 ******************************************************************************
 * @file    main.cpp
 * @author  Davide Aliprandi, STMicrolectronics
 * @version V1.0.0
 * @date    October 14th, 2015
 * @brief   mbed test application for the STMicrolectronics X-NUCLEO-IHM01A1
 *          Motor Control Expansion Board: control of 1 motor.
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */


/* Includes ------------------------------------------------------------------*/

/* mbed specific header files. */
#include "mbed.h"

/* Helper header files. */
#include "DevSPI.h"

/* Component specific header files. */
#include "l6474_class.h"


/* Definitions ---------------------------------------------------------------*/

/* Number of steps to move. */
#define STEPS 3200


/* Variables -----------------------------------------------------------------*/

/* Motor Control Component. */
L6474 *motor;


/* Main ----------------------------------------------------------------------*/

int main()
{
    /*----- Initialization. -----*/

    /* Initializing SPI bus. */
    DevSPI dev_spi(D11, D12, D13);

    /* Initializing Motor Control Component. */
    motor = new L6474(D2, D8, D7, D9, D10, dev_spi);
    if (motor->Init(NULL) != COMPONENT_OK)
        return false;

    /* Printing to the console. */
    printf("Motor Control Application Example for 1 Motor\r\n\n");


    /*----- Moving. -----*/

    /* Printing to the console. */
    printf("--> Moving forward %d steps.\r\n", STEPS);

    /* Moving N steps in the forward direction. */
    motor->Move(StepperMotor::FWD, STEPS);

    /* Waiting while the motor is active. */
    motor->WaitWhileActive();

    /* Getting current position. */
    int position = motor->GetPosition();
    
    /* Printing to the console. */
    printf("    Position: %d.\r\n", position);

    /* Waiting 2 seconds. */
    wait_ms(2000);

    
    /*----- Moving. -----*/
    
    /* Printing to the console. */
    printf("--> Moving backward %d steps.\r\n", STEPS >> 1);

    /* Moving N steps in the backward direction. */
    motor->Move(StepperMotor::BWD, STEPS >> 1);
    
    /* Waiting while the motor is active. */
    motor->WaitWhileActive();

    /* Getting current position. */
    position = motor->GetPosition();
    
    /* Printing to the console. */
    printf("    Position: %d.\r\n", position);
    printf("--> Setting Home.\r\n");

    /* Setting the current position to be the home position. */
    motor->SetHome();

    /* Waiting 2 seconds. */
    wait_ms(2000);


    /*----- Going to a specified position. -----*/

    /* Printing to the console. */
    printf("--> Going to position %d.\r\n", STEPS >> 1);
    
    /* Requesting to go to a specified position. */
    motor->GoTo(STEPS >> 1);
    
    /* Waiting while the motor is active. */
    motor->WaitWhileActive();

    /* Getting current position. */
    position = motor->GetPosition();
    
    /* Printing to the console. */
    printf("    Position: %d.\r\n", position);
    
    /* Waiting 2 seconds. */
    wait_ms(2000);

    
    /*----- Going Home. -----*/

    /* Printing to the console. */
    printf("--> Going Home.\r\n");
    
    /* Requesting to go to home. */
    motor->GoHome();
    
    /* Waiting while the motor is active. */
    motor->WaitWhileActive();

    /* Getting current position. */
    position = motor->GetPosition();

    /* Printing to the console. */
    printf("    Position: %d.\r\n", position);

    /* Waiting 2 seconds. */
    wait_ms(2000);


    /*----- Running. -----*/

    /* Printing to the console. */
    printf("--> Running backward.\r\n");

    /* Requesting to run backward. */
    motor->Run(StepperMotor::BWD);

    /* Waiting until delay has expired. */
    wait_ms(6000);

    /* Getting current speed. */
    int speed = motor->GetSpeed();

    /* Printing to the console. */
    printf("    Speed: %d.\r\n", speed);

    /*----- Increasing the speed while running. -----*/

    /* Printing to the console. */
    printf("--> Increasing the speed while running.\r\n");

    /* Increasing speed to 2400 step/s. */
    motor->SetMaxSpeed(2400);

    /* Waiting until delay has expired. */
    wait_ms(6000);

    /* Getting current speed. */
    speed = motor->GetSpeed();

    /* Printing to the console. */
    printf("    Speed: %d.\r\n", speed);


    /*----- Decreasing the speed while running. -----*/

    /* Printing to the console. */
    printf("--> Decreasing the speed while running.\r\n");

    /* Decreasing speed to 1200 step/s. */
    motor->SetMaxSpeed(1200);

    /* Waiting until delay has expired. */
    wait_ms(8000);

    /* Getting current speed. */
    speed = motor->GetSpeed();

    /* Printing to the console. */
    printf("    Speed: %d.\r\n", speed);


    /*----- Requiring hard-stop while running. -----*/

    /* Printing to the console. */
    printf("--> Requiring hard-stop while running.\r\n");

    /* Requesting to immediatly stop. */
    motor->HardStop();

    /* Waiting while the motor is active. */
    motor->WaitWhileActive();

    /* Waiting 2 seconds. */
    wait_ms(2000);


    /*----- Infinite Loop. -----*/

    /* Printing to the console. */
    printf("--> Infinite Loop...\r\n");

    /* Setting the current position to be the home position. */
    motor->SetHome();

    /* Infinite Loop. */
    while (1)
    {
        /* Requesting to go to a specified position. */
        motor->GoTo(STEPS >> 1);

        /* Waiting while the motor is active. */
        motor->WaitWhileActive();

        /* Requesting to go to a specified position. */
        motor->GoTo(- (STEPS >> 1));

        /* Waiting while the motor is active. */
        motor->WaitWhileActive();
    }
}
