/**
 ******************************************************************************
 * @file    main.cpp
 * @author  Davide Aliprandi / AST
 * @version V1.0.0
 * @date    October 14th, 2015
 * @brief   mbed test application for the STMicrolectronics X-NUCLEO-IHM01A1
 *          Motor Control Expansion Board: control of 2 motors.
 *          This application makes use of a C++ component architecture obtained
 *          from the C component architecture through the Stm32CubeTOO tool.
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */


/* Includes ------------------------------------------------------------------*/

/* mbed specific header files. */
#include "mbed.h"

/* Helper header files. */
#include "DevSPI.h"

/* Component specific header files. */
#include "l6474_class.h"


/* Definitions ---------------------------------------------------------------*/

/* Number of steps corresponding to one round angle of the motor. */
#define ROUND_ANGLE_STEPS 3200


/* Variables -----------------------------------------------------------------*/

/* Motor Control Component. */
L6474 *motor1;
L6474 *motor2;


/* Main ----------------------------------------------------------------------*/

int main()
{
    /* Initializing SPI bus. */
    DevSPI dev_spi(D11, D12, D13);

    /* Initializing Motor Control Component. */
    motor1 = new L6474(D8, D7, D9, D10, dev_spi);
    if (motor1->Init(NULL) != COMPONENT_OK)
        return false;
    motor2 = new L6474(D8, D4, D3, D10, dev_spi);
    if (motor2->Init(NULL) != COMPONENT_OK)
        return false;

    /* Printing to the console. */
    printf("Motor Control Application Example for 2 Motors\r\n\n");

    /* Main Loop. */
    while(true)
    {
        /*----- Moving. -----*/

        /* Printing to the console. */
        printf("--> Moving forward: M1 %d steps, M2 %d steps.\r\n", ROUND_ANGLE_STEPS >> 1, ROUND_ANGLE_STEPS);

        /* Moving N steps in the forward direction. */
        motor1->Move(StepperMotor::FWD, ROUND_ANGLE_STEPS >> 1);
        motor2->Move(StepperMotor::FWD, ROUND_ANGLE_STEPS);

        /* Waiting while the motor is active. */
        motor1->WaitWhileActive();
        motor2->WaitWhileActive();

        /* Getting current position. */
        int position1 = motor1->GetPosition();
        int position2 = motor2->GetPosition();
        
        /* Printing to the console. */
        printf("    Position: M1 %d, M2 %d.\r\n", position1, position2);

        /* Waiting 2 seconds. */
        wait_ms(2000);

        
        /*----- Moving. -----*/
        
        /* Printing to the console. */
        printf("--> Moving backward: M1 %d steps, M2 %d steps.\r\n", ROUND_ANGLE_STEPS >> 1, ROUND_ANGLE_STEPS);

        
        /* Moving N steps in the backward direction. */
        motor1->Move(StepperMotor::BWD, ROUND_ANGLE_STEPS >> 1);
        motor2->Move(StepperMotor::BWD, ROUND_ANGLE_STEPS);
        
        /* Waiting while the motor is active. */
        motor1->WaitWhileActive();
        motor2->WaitWhileActive();

        /* Getting current position. */
        position1 = motor1->GetPosition();
        position2 = motor2->GetPosition();
        
        /* Printing to the console. */
        printf("    Position: M1 %d, M2 %d.\r\n", position1, position2);
        printf("--> Setting Home.\r\n");

        /* Setting the current position to be the home position. */
        motor1->SetHome();
        motor2->SetHome();

        /* Waiting 2 seconds. */
        wait_ms(2000);


        /*----- Going to a specified position. -----*/

        /* Printing to the console. */
        printf("--> Going to position: M1 %d, M2 %d.\r\n", ROUND_ANGLE_STEPS, ROUND_ANGLE_STEPS >> 1);
        
        /* Requesting to go to a specified position. */
        motor1->GoTo(ROUND_ANGLE_STEPS);
        motor2->GoTo(ROUND_ANGLE_STEPS >> 1);
        
        /* Waiting while the motor is active. */
        motor1->WaitWhileActive();
        motor2->WaitWhileActive();

        /* Getting current position. */
        position1 = motor1->GetPosition();
        position2 = motor2->GetPosition();
        
        /* Printing to the console. */
        printf("    Position: M1 %d, M2 %d.\r\n", position1, position2);
        
        /* Waiting 2 seconds. */
        wait_ms(2000);

        
        /*----- Going Home. -----*/

        /* Printing to the console. */
        printf("--> Going Home.\r\n");
        
        /* Requesting to go to home. */
        motor1->GoHome();
        motor2->GoHome();
        
        /* Waiting while the motor is active. */
        motor1->WaitWhileActive();
        motor2->WaitWhileActive();

        /* Getting current position. */
        position1 = motor1->GetPosition();
        position2 = motor2->GetPosition();

        /* Printing to the console. */
        printf("    Position: M1 %d, M2 %d.\r\n", position1, position2);

        /* Waiting 2 seconds. */
        wait_ms(2000);


        /*----- Running. -----*/

        /* Printing to the console. */
        printf("--> M1 running backward, M2 running forward.\r\n");

        /* Requesting to run backward. */
        motor1->Run(StepperMotor::BWD);
        motor2->Run(StepperMotor::FWD);

        /* Waiting until delay has expired. */
        wait_ms(6000);

        /* Getting current speed. */
        int speed1 = motor1->GetSpeed();
        int speed2 = motor2->GetSpeed();

        /* Printing to the console. */
        printf("    Speed: M1 %d, M2 %d.\r\n", speed1, speed2);

        /*----- Increasing the speed while running. -----*/

        /* Printing to the console. */
        printf("--> Increasing the speed while running.\r\n");

        /* Increasing speed to 2400 step/s. */
        motor1->SetMaxSpeed(2400);
        motor2->SetMaxSpeed(2400);

        /* Waiting until delay has expired. */
        wait_ms(6000);

        /* Getting current speed. */
        speed1 = motor1->GetSpeed();
        speed2 = motor2->GetSpeed();

        /* Printing to the console. */
        printf("    Speed: M1 %d, M2 %d.\r\n", speed1, speed2);


        /*----- Decreasing the speed while running. -----*/

        /* Printing to the console. */
        printf("--> Decreasing the speed while running.\r\n");

        /* Decreasing speed to 1200 step/s. */
        motor1->SetMaxSpeed(1200);
        motor2->SetMaxSpeed(1200);

        /* Waiting until delay has expired. */
        wait_ms(8000);

        /* Getting current speed. */
        speed1 = motor1->GetSpeed();
        speed2 = motor2->GetSpeed();

        /* Printing to the console. */
        printf("    Speed: M1 %d, M2 %d.\r\n", speed1, speed2);


        /*----- Requiring hard-stop while running. -----*/

        /* Printing to the console. */
        printf("--> Requiring hard-stop while running.\r\n");

        /* Requesting to immediatly stop. */
        motor1->HardStop();
        motor2->HardStop();

        /* Waiting while the motor is active. */
        motor1->WaitWhileActive();
        motor2->WaitWhileActive();

        /* Waiting 2 seconds. */
        wait_ms(2000);


        /*----- Infinite Loop. -----*/

        /* Printing to the console. */
        printf("--> Infinite Loop...\r\n");

        /* Setting the current position to be the home position. */
        motor1->SetHome();
        motor2->SetHome();

        /* Infinite Loop. */
        while(1)
        {
            /* Requesting to go to a specified position. */
            motor1->GoTo(ROUND_ANGLE_STEPS >> 1);
            motor2->GoTo(- (ROUND_ANGLE_STEPS >> 1));

            /* Waiting while the motor is active. */
            motor1->WaitWhileActive();
            motor2->WaitWhileActive();

            /* Requesting to go to a specified position. */
            motor1->GoTo(- (ROUND_ANGLE_STEPS >> 1));
            motor2->GoTo(ROUND_ANGLE_STEPS >> 1);

            /* Waiting while the motor is active. */
            motor1->WaitWhileActive();
            motor2->WaitWhileActive();
        }
    }
}
