/**
 ******************************************************************************
 * @file    main.cpp
 * @author  Rosarium PILA, STMicroelectronics
 * @version V1.0.0
 * @date    June 19th, 2017
 * @brief   mbed test application for the STMicroelectronics X-NUCLEO-IDB01A4/5
 *          Spirit1 Expansion Board
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2017 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */

#include "mbed.h"
#include "SimpleSpirit1.h"

#define TEST_STR_LEN (32) 
static uint8_t send_buf[TEST_STR_LEN] ={'S','P','I','R','I','T','1',' ','H','E','L','L','O',' ','W','O','R','L','D',' ','P','2','P',' ','D','E','M','O'};
static uint8_t read_buf[TEST_STR_LEN] ={'0','0','0','0','0','0','0','0','0','0','0','0','0','0','0','0','0','0','0','0','0','0','0','0','0','0','0','0','0','0','0','0'};
          
static SimpleSpirit1 &myspirit = SimpleSpirit1::CreateInstance(D11, D12, D3, D9, D10, D2); /* (SPI_CLK) = (D13:PA5:resistorR7 --> D3:PB3:resistorR4)*/

static volatile bool rx_done_flag = false; 
static volatile bool tx_done_flag = false; 
static volatile bool send_data_flag = false;

static DigitalOut TestLED(D5);         /* LED of IDS01A4/5 */
static InterruptIn event(USER_BUTTON); /* Interrupt event to give userinterface to send pkt. */

/**
 * @brief  callback_func
 * @param  callback event
 * @retval None.
 */
static void callback_func(int event) 
{
  if(event == SimpleSpirit1::RX_DONE) 
  {              
    rx_done_flag = 1;
  }
  else if (event == SimpleSpirit1::TX_DONE) 
  {
    tx_done_flag = 1;
  }
}

/**
 * @brief  set_send_data_flag
 * @param  None
 * @retval None
 */
static void set_send_data_flag(void)
{  
  send_data_flag = 1 ;
}

/**
 * @brief  send_data
 * @param  None
 * @retval None
 */
static void send_data(void)
{
  printf("\r\n***Sending a packet***\r\n");  
  
  while(myspirit.is_receiving()); /* wait for ongoing RX ends */
 
  size_t curr_len = strlen((const char*)send_buf) + 1;
  myspirit.send(send_buf, curr_len);
}

/**
 * @brief  read_rcvd_data
 * @param  None
 * @retval None
 */
static void read_rcvd_data(void)
{  
 
  for(unsigned int flush_count = 0; flush_count < TEST_STR_LEN; flush_count++) read_buf[flush_count] = 0 ;/* clear the read buffer */
      
      int ret = myspirit.read(read_buf, sizeof(read_buf));
      
      TestLED = !TestLED;     /* Toggle LED at the receiver */
      
      if(ret == 0) 
      {
        printf("\nNothing to read\n\r");
        return;
      }
      printf("\r\n***Received a packet***\r\n\rReceived string = '%s' (len=%d) \n\r", read_buf, ret); 
}


/**
 * @brief  main routine
 * @param  None
 * @retval int
 */
int main() 
{  
  TestLED = 0;                  /* LED off */
  
  myspirit.attach_irq_callback(callback_func);
  
  myspirit.on();
  
  printf("\n**************HelloWorld mbed demo for Spirit1 (X-NUCLEO-IDS01A4/5)**************\r\n");
  printf("\nPress User Button on one of the two boards to send a packaet to the other and the LED D1 on the receiver X-NUCLEO-IDS01A4/5 should toggle \n\r\n");
  
  event.rise(&set_send_data_flag); /*User button interrupt trigger to set send data flag */
  
  while(1) 
  {
    __WFE(); /* low power in idle condition., waiting for an event */
    
    if(rx_done_flag) 
    {
      rx_done_flag = false; 
      read_rcvd_data();
    }
    
    else if (send_data_flag)
    {
      send_data_flag = false;
      send_data();
    }
    
    else if (tx_done_flag)
    {
      tx_done_flag = false;
      printf("\r\n***Packet sent ***\r\nSent string ='%s' (len=%d)\n\r", send_buf, strlen((const char*)send_buf) + 1);
    }
  }
  
  /* unreachable */
  // myspirit.off();
  // return 0;
}
