/**
  ******************************************************************************
  * @file       main.cpp
  * @date       22/01/2016
  * @brief      Test the async comunication api
  ******************************************************************************
  *
  * COPYRIGHT(c) 2015 STMicroelectronics
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */

#include "mbed.h"
#include "XNucleoNFC01A1.h"
#include "ReadUriCallbacks.h"
#include "WriteUriCallbacks.h"
#include "NDefNfcTagM24SR.h"
#include "NDefLib/RecordType/RecordURI.h"

 
/** variable set to true when we receive an interrupt from the nfc component*/
static volatile bool nfcInterruptFlag=false;
 
/** variable set to true when the user press the board user button*/
static volatile bool buttonPress=false;
 
/** Nfc ISR called when the nfc component has a message ready*/
static void nfc_interrupt_callback() {
    nfcInterruptFlag=true;
}//nfcInterruptCallback
 
static void set_button_press() {
    buttonPress=true;
}//if buttonPress
 
int main(int argc,char *args[]) {
    (void)argc; (void)args;
 
#if defined(TARGET_STM)
    InterruptIn userButton(USER_BUTTON);    
    userButton.fall(set_button_press);
#endif
 
    //create the nfc component
    I2C i2cChannel(XNucleoNFC01A1::DEFAULT_SDA_PIN,XNucleoNFC01A1::DEFAULT_SDL_PIN);

    XNucleoNFC01A1 *nfcNucleo = XNucleoNFC01A1::instance(i2cChannel,&nfc_interrupt_callback,
        XNucleoNFC01A1::DEFAULT_GPO_PIN,XNucleoNFC01A1::DEFAULT_RF_DISABLE_PIN,
        XNucleoNFC01A1::DEFAULT_LED1_PIN,XNucleoNFC01A1::DEFAULT_LED2_PIN,
        XNucleoNFC01A1::DEFAULT_LED3_PIN);
 
    //No call back needed since default behavior is sync
    nfcNucleo->get_M24SR().get_session();
    nfcNucleo->get_M24SR().manage_I2C_GPO(M24SR::I2C_ANSWER_READY); //switch to async mode
 
    NDefLib::NDefNfcTag &tag = nfcNucleo->get_M24SR().get_NDef_tag();
    printf("System Init done!\n\r");
 
    //crate the callback to use for write a tag
    WriteUriCallbacks NDefWriteCallback(nfcNucleo->get_led1(),nfcNucleo->get_led2(),nfcNucleo->get_led3());
    ReadUriCallbacks NDefReadCallback(nfcNucleo->get_led1(),nfcNucleo->get_led2(),nfcNucleo->get_led3());
    tag.set_callback(&NDefWriteCallback); //set the callback
    tag.open_session(); //start the callback chain
 
    printf("Start Main Loop\n\r");
    while(true) {
        if (nfcInterruptFlag) {
            nfcInterruptFlag=false;
            //manage an async event from the nfc component
            nfcNucleo->get_M24SR().manage_event();
        }//if
#if defined(TARGET_STM)
        if (buttonPress) {
            buttonPress=false;
            printf("Pressed");
            tag.set_callback(&NDefReadCallback);
            tag.open_session();
        }
#endif
        __WFE();
    }//while
 
    //return 0;
}
