/**
  ******************************************************************************
  * @file       Sample_async.cpp
  * @date       22/01/2016
  * @brief      Test the async comunication api
  ******************************************************************************
  *
  * COPYRIGHT(c) 2015 STMicroelectronics
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */

#include "mbed.h"

#include "X_NUCLEO_NFC01A1.h"
#include "m24sr/NDefNfcTagM24SR.h"
#include "NDefLib/RecordType/RecordURI.h"

/** variable set to true when we receive an interrupt from the nfc component*/
static volatile bool nfcInterruptFlag=false;

/** variable set to true when the user press the board user button*/
static volatile bool buttonPress=false;

/** Nfc ISR called when the nfc component has a message ready*/
static void nfcInterruptCallback(){
    nfcInterruptFlag=true;
}//nfcInterruptCallback

static void setButtonPress(){
    buttonPress=true;
}//if buttonPress

/**
 * Chain of callback that will crate a Uri record and write it.
 * After each operation the class will switch on a led
 */
class WriteUriCallbacks : public NDefLib::NDefNfcTag::Callbacks{

    DigitalOut &mOnOpenSession;
    DigitalOut &mOnWrite;
    DigitalOut &mOnCloseSession;
    NDefLib::Message msg;

public:

    /**
     * create the callback chain
     * @param onOpenSession led to switch on when the session open
     * @param onWrite led to switch on when the write end
     * @param onCloseSession led to switch on when the session end
     */
    WriteUriCallbacks(DigitalOut &onOpenSession,DigitalOut &onWrite,
            DigitalOut &onCloseSession):mOnOpenSession(onOpenSession),
                    mOnWrite(onWrite),mOnCloseSession(onCloseSession){};

    /**
     * crate the new message and write it
     * @param tag tag where write the message
     * @param success true if the session correctly open
     */
    virtual void onSessionOpen(NDefLib::NDefNfcTag *tag,bool success){
        if(!success){
            printf("Error opening the session\r\n");
        }//else
        printf("Session opened\r\n");
        //ask to have an interrupt when the command finish
        mOnOpenSession=1;
        
        NDefLib::RecordURI rUri(NDefLib::RecordURI::HTTP_WWW,"http://www.st.com");
        msg.addRecord(&rUri);

        tag->write(msg);
    }

    /**
     * request to close the session
     * @param tag tag where close the session
     * @param success true if the message is correctly wrote
     * @param message wrote
     */
    virtual void onMessageWrite(NDefLib::NDefNfcTag *tag,bool success,
            const NDefLib::Message&){

        if(!success)
            printf("Error writing tag!\r\n");
        else{
            printf("Tag written!\r\n");
            mOnWrite=1;
        }//if-else
        tag->closeSession();
    }

    /**
     * switch on the led
     * @param tag where the session is closed
     * @param success true if the session is correctly close
     */
    virtual void onSessionClose(NDefLib::NDefNfcTag*,bool success){
        if(success){
            printf("Session closed\r\n");
            mOnCloseSession=1;
        }else
            printf("Error closing the session\r\n");
    }

};

/**
 * Chain of callback that will read a NDef Message and print all the
 * record of type URI.
 * After each operation the class will switch on a led
 */
class ReadUriCallbacks : public NDefLib::NDefNfcTag::Callbacks{

    DigitalOut &mOnOpenSession;
    DigitalOut &mOnRead;
    DigitalOut &mOnCloseSession;

    NDefLib::Message mMsg;

public:

    /**
     * create the callback chain
     * @param onOpenSession led to switch on when the session open
     * @param onWrite led to switch on when the write end
     * @param onCloseSession led to switch on when the session end
     */
    ReadUriCallbacks(DigitalOut &onOpenSession,DigitalOut &onRead,
            DigitalOut &onCloseSession):mOnOpenSession(onOpenSession),
                    mOnRead(onRead),mOnCloseSession(onCloseSession){};

    /**
     * crate the new message and write it
     * @param tag tag where write the message
     * @param success true if the session correctly open
     */
    virtual void onSessionOpen(NDefLib::NDefNfcTag *tag,bool success){
        if(!success){
            printf("Error opening the session\r\n");
        }//else
        printf("Session opened\r\n");
        //ask to have an interrupt when the command finish
        mOnOpenSession=1;
        
        tag->read(&mMsg);
    }

    /**
     * request to close the session
     * @param tag tag where close the session
     * @param success true if the message is correctly wrote
     * @param message wrote
     */
    virtual void onMessageRead(NDefLib::NDefNfcTag *tag,bool success,
            const NDefLib::Message*){

        if(!success)
            printf("Error Reading tag!\r\n");
        else{
            const uint32_t nRecords =mMsg.getNRecords();
            printf("Read %d records!\r\n",nRecords);
            for(uint32_t i=0;i<nRecords;i++){
                if(mMsg[i]->getType()== NDefLib::Record::TYPE_URI){
                    NDefLib::RecordURI *rUri = (NDefLib::RecordURI *)mMsg[i];
                    printf("UriType: %x\r\nUriContent: %s\r\n",
                        rUri->getUriId(),
                        rUri->getContent().c_str());
                }//if
            }//for
            NDefLib::Message::removeAndDeleteAllRecord(mMsg);
            mOnRead=1;
        }//if-else
        tag->closeSession();
    }

    /**
     * switch on the led
     * @param tag where the session is closed
     * @param success true if the session is correctly close
     */
    virtual void onSessionClose(NDefLib::NDefNfcTag*,bool success){
        if(success){
            printf("Session closed\r\n");
            mOnCloseSession=1;
        }else
            printf("Error opening the session\r\n");
    }

};



int main(int argc,char *args[]){
    (void)argc; (void)args;

#if defined(TARGET_STM)
    InterruptIn userButton(USER_BUTTON);    
    userButton.fall(setButtonPress);
#endif

    //create the nfc component
    I2C i2cChannel(X_NUCLEO_NFC01A1::DEFAULT_SDA_PIN,X_NUCLEO_NFC01A1::DEFAULT_SDL_PIN);
    X_NUCLEO_NFC01A1 *nfcNucleo = X_NUCLEO_NFC01A1::Instance(i2cChannel,&nfcInterruptCallback,
            X_NUCLEO_NFC01A1::DEFAULT_GPO_PIN,X_NUCLEO_NFC01A1::DEFAULT_RF_DISABLE_PIN,
            X_NUCLEO_NFC01A1::DEFAULT_LED1_PIN,X_NUCLEO_NFC01A1::DEFAULT_LED2_PIN,
            X_NUCLEO_NFC01A1::DEFAULT_LED3_PIN);

    //No call back needed since default behavior is sync
    nfcNucleo->getM24SR().GetSession();
    nfcNucleo->getM24SR().ManageI2CGPO(I2C_ANSWER_READY); //switch to async mode

    NDefLib::NDefNfcTag &tag = nfcNucleo->getM24SR().getNDefTag();
    printf("System Init done!\n\r");

    //crate the callback to use for write a tag
    WriteUriCallbacks NDefWriteCallback(nfcNucleo->getLed1(),nfcNucleo->getLed2(),nfcNucleo->getLed3());
    ReadUriCallbacks NDefReadCallback(nfcNucleo->getLed1(),nfcNucleo->getLed2(),nfcNucleo->getLed3());
    tag.setCallback(&NDefWriteCallback); //set the callback
    tag.openSession(); //start the callback chain

    printf("Start Main Loop\n\r");
    while(true){
        if(nfcInterruptFlag){
            nfcInterruptFlag=false;
            //manage an async event from the nfc component
            nfcNucleo->getM24SR().ManageEvent();

        }//if
#if defined(TARGET_STM)
        if(buttonPress){
            buttonPress=false;
            printf("Pressed");
            tag.setCallback(&NDefReadCallback);
            tag.openSession();
        }
#endif
        __WFE();
    }//while

    //return 0;
}
