#include "mbed.h"
#include "stm32l496g_discovery.h"
#include "stm32l496g_discovery_lcd.h"
#include "stm32l496g_discovery_sd.h"

#define NUM_OF_BLOCKS            5
#define BLOCK_START_ADDR         0
#define BLOCK_SIZE               512
#define BLOCK_END_ADDR           (BLOCK_SIZE * NUM_OF_BLOCKS)
#define BUFFER_WORDS_SIZE        ((BLOCK_SIZE * NUM_OF_BLOCKS) >> 2) // Total data size in bytes

uint32_t aTxBuffer[BUFFER_WORDS_SIZE];
uint32_t aRxBuffer[BUFFER_WORDS_SIZE];

static void print_demo_title(void);
static void print_PASS(void);
static void print_FAIL(void);
static void Fill_Buffer(uint32_t *pBuffer, uint32_t uwBufferLenght, uint32_t uwOffset);
static uint8_t Buffercmp(uint32_t* pBuffer1, uint32_t* pBuffer2, uint16_t BufferLength);

int main()
{
    uint8_t status;
    print_demo_title();
    wait(0.2);
    
    // Initialization
    status = BSP_SD_Init();
    if (status == MSD_OK) {
      BSP_LCD_DisplayStringAt(0, LINE(5), (uint8_t *)"SD INIT OK", LEFT_MODE);
    }
    else if (status == MSD_ERROR_SD_NOT_PRESENT) {
      BSP_LCD_DisplayStringAt(0, LINE(5), (uint8_t *)"SD CARD NOT FOUND", LEFT_MODE);
      print_FAIL();
    }
    else {
      BSP_LCD_DisplayStringAt(0, LINE(5), (uint8_t *)"SD INIT FAIL", LEFT_MODE);
      print_FAIL();
    }
    
    // Erase
    status = BSP_SD_Erase(BLOCK_START_ADDR, BLOCK_END_ADDR);
    if (status == MSD_OK) {
      BSP_LCD_DisplayStringAt(0, LINE(6), (uint8_t *)"SD ERASE OK", LEFT_MODE);
    }
    else {
      BSP_LCD_DisplayStringAt(0, LINE(6), (uint8_t *)"SD ERASE FAIL", LEFT_MODE);
      print_FAIL();
    }
    
    // Prepare the buffer to write
    Fill_Buffer(aTxBuffer, BUFFER_WORDS_SIZE, 0x22FF);
    
    // Write
    status = BSP_SD_WriteBlocks(aTxBuffer, BLOCK_START_ADDR, NUM_OF_BLOCKS, SD_DATATIMEOUT);
    if (status == MSD_OK) {
      BSP_LCD_DisplayStringAt(0, LINE(7), (uint8_t *)"SD WRITE OK", LEFT_MODE);
    }
    else {
      BSP_LCD_DisplayStringAt(0, LINE(7), (uint8_t *)"SD WRITE FAIL", LEFT_MODE);
      print_FAIL();
    }
    
    // Read
    status = BSP_SD_ReadBlocks(aRxBuffer, BLOCK_START_ADDR, NUM_OF_BLOCKS, SD_DATATIMEOUT);
    if (status == MSD_OK) {
      BSP_LCD_DisplayStringAt(0, LINE(8), (uint8_t *)"SD READ OK", LEFT_MODE);
    }
    else {
      BSP_LCD_DisplayStringAt(0, LINE(8), (uint8_t *)"SD READ FAIL", LEFT_MODE);
      print_FAIL();
    }
    
    // Compare data
    if (Buffercmp(aTxBuffer, aRxBuffer, BUFFER_WORDS_SIZE) == 0) {
      BSP_LCD_DisplayStringAt(0, LINE(9), (uint8_t *)"SD DATA OK", LEFT_MODE);
    }
    else {
      BSP_LCD_DisplayStringAt(0, LINE(9), (uint8_t *)"SD DATA FAIL", LEFT_MODE);
      print_FAIL();
    }

    print_PASS();
}

static void print_demo_title(void)
{
    BSP_LCD_Init();
    BSP_LCD_Clear(LCD_COLOR_WHITE);
    BSP_LCD_SetTextColor(LCD_COLOR_BLUE);
    BSP_LCD_FillRect(0, 0, BSP_LCD_GetXSize(), 80);
    BSP_LCD_SetTextColor(LCD_COLOR_WHITE);
    BSP_LCD_SetBackColor(LCD_COLOR_BLUE);
    BSP_LCD_SetFont(&Font24);
    BSP_LCD_DisplayStringAt(0, 0, (uint8_t *)"SDCARD", CENTER_MODE);
    BSP_LCD_SetFont(&Font12);
    BSP_LCD_DisplayStringAt(0, 30, (uint8_t *)"This example shows how to write", CENTER_MODE);
    BSP_LCD_DisplayStringAt(0, 45, (uint8_t *)"and read data on the SDCard", CENTER_MODE);
    BSP_LCD_SetFont(&Font20);
}

static void print_PASS(void)
{
    BSP_LCD_SetTextColor(LCD_COLOR_BLACK);
    BSP_LCD_SetBackColor(LCD_COLOR_GREEN);
    BSP_LCD_DisplayStringAt(0, LINE(11), (uint8_t *)"Demo OK", CENTER_MODE);
    while(1);
}

static void print_FAIL(void)
{
    BSP_LCD_SetBackColor(LCD_COLOR_RED);
    BSP_LCD_DisplayStringAt(0, LINE(11), (uint8_t *)"Demo FAILED", CENTER_MODE);
    while(1);
}

/**
  * @brief  Fills buffer with user predefined data.
  * @param  pBuffer: pointer on the buffer to fill
  * @param  uwBufferLenght: size of the buffer to fill
  * @param  uwOffset: first value to fill on the buffer
  * @retval None
  */
static void Fill_Buffer(uint32_t *pBuffer, uint32_t uwBufferLength, uint32_t uwOffset)
{
  uint32_t tmpIndex = 0;
  /* Put in global buffer different values */
  for (tmpIndex = 0; tmpIndex < uwBufferLength; tmpIndex++ ) {
    pBuffer[tmpIndex] = tmpIndex + uwOffset;
  }
}
 
/**
  * @brief  Compares two buffers.
  * @param  pBuffer1, pBuffer2: buffers to be compared.
  * @param  BufferLength: buffer's length
  * @retval 0: pBuffer2 identical to pBuffer1
  *         1: pBuffer2 differs from pBuffer1
  */
static uint8_t Buffercmp(uint32_t* pBuffer1, uint32_t* pBuffer2, uint16_t BufferLength)
{
  while (BufferLength--) {
    if (*pBuffer1 != *pBuffer2) {
      return 1;
    }
    pBuffer1++;
    pBuffer2++;
  }
  return 0;
}
