#include "mbed.h"
#include "stm32f413h_discovery.h"
#include "stm32f413h_discovery_lcd.h"
#include "stm32f413h_discovery_sd.h"

#define BLOCK_START_ADDR         0     /* Block start address      */
#define NUM_OF_BLOCKS            5     /* Total number of blocks   */
#define BUFFER_WORDS_SIZE        ((512 * NUM_OF_BLOCKS) >> 2) /* Total data size in bytes */

uint32_t aTxBuffer[BUFFER_WORDS_SIZE];
uint32_t aRxBuffer[BUFFER_WORDS_SIZE];

static void Fill_Buffer(uint32_t *pBuffer, uint32_t uwBufferLenght, uint32_t uwOffset);
static uint8_t Buffercmp(uint32_t* pBuffer1, uint32_t* pBuffer2, uint16_t BufferLength);
static void print_error(uint8_t *msg);

int main()
{
    /* Init LCD and display example information */
    BSP_LCD_Init();
    BSP_LCD_Clear(LCD_COLOR_WHITE);
    BSP_LCD_SetFont(&Font16);

    /* SD init */
    if (BSP_SD_Init() == MSD_OK) {
        BSP_LCD_SetTextColor(LCD_COLOR_GREEN);
        BSP_LCD_FillRect(0, 0, BSP_LCD_GetXSize(), 40);
        BSP_LCD_SetTextColor(LCD_COLOR_BLACK);
        BSP_LCD_SetBackColor(LCD_COLOR_GREEN);
        BSP_LCD_DisplayStringAt(0, 15, (uint8_t *)"SD detected", CENTER_MODE);
        BSP_LCD_SetBackColor(LCD_COLOR_WHITE);
    } else {
        BSP_LCD_SetTextColor(LCD_COLOR_RED);
        BSP_LCD_FillRect(0, 0, BSP_LCD_GetXSize(), 40);
        BSP_LCD_SetTextColor(LCD_COLOR_BLACK);
        BSP_LCD_SetBackColor(LCD_COLOR_RED);
        BSP_LCD_DisplayStringAt(0, 15, (uint8_t *)"SD not detected", CENTER_MODE);
        print_error((uint8_t *)"Demo aborted");
    }

    wait(1);
    // Check if SD card is detected
    if (BSP_SD_IsDetected() == SD_PRESENT) {
        BSP_LCD_DisplayStringAt(0, 50, (uint8_t *)"SD detected OK", CENTER_MODE);
    } else {
        print_error((uint8_t *)"SD not detected");
    }

    wait(1);
    // Erase SD card
    if(BSP_SD_Erase(BLOCK_START_ADDR, (512 * NUM_OF_BLOCKS)) == MSD_OK) {
        BSP_LCD_DisplayStringAt(0, 70, (uint8_t *)"SD erased OK", CENTER_MODE);
    } else {
        print_error((uint8_t *)"Erased failed");
    }

    wait(1);
    // Write data in SD card
    Fill_Buffer(aTxBuffer, BUFFER_WORDS_SIZE, 0x22FF);
    if (BSP_SD_WriteBlocks(aTxBuffer, BLOCK_START_ADDR, NUM_OF_BLOCKS, SD_DATATIMEOUT) == MSD_OK) {
        BSP_LCD_DisplayStringAt(0, 90, (uint8_t *)"SD write OK", CENTER_MODE);
    } else {
        print_error((uint8_t *)"Write failed");
    }

    wait(1);
    // Read data from SD card and verify them
    if (BSP_SD_ReadBlocks(aRxBuffer, BLOCK_START_ADDR, NUM_OF_BLOCKS, SD_DATATIMEOUT) == MSD_OK) {
        BSP_LCD_DisplayStringAt(0, 110, (uint8_t *)"SD read OK", CENTER_MODE);
        if (Buffercmp(aTxBuffer, aRxBuffer, BUFFER_WORDS_SIZE) == 0) {
            BSP_LCD_DisplayStringAt(0, 130, (uint8_t *)"Data OK", CENTER_MODE);
        } else {
            print_error((uint8_t *)"Data wrong");
        }
    } else {
        print_error((uint8_t *)"Read failed");
    }

    wait(1);
    BSP_LCD_DisplayStringAt(0, 200, (uint8_t *)"Demo finished OK", CENTER_MODE);
    while(1);
}

/**
  * @brief  Fills buffer with user predefined data.
  * @param  pBuffer: pointer on the buffer to fill
  * @param  uwBufferLenght: size of the buffer to fill
  * @param  uwOffset: first value to fill on the buffer
  * @retval None
  */
static void Fill_Buffer(uint32_t *pBuffer, uint32_t uwBufferLength, uint32_t uwOffset)
{
    uint32_t tmpIndex = 0;

    /* Put in global buffer different values */
    for (tmpIndex = 0; tmpIndex < uwBufferLength; tmpIndex++ ) {
        pBuffer[tmpIndex] = tmpIndex + uwOffset;
    }
}

/**
  * @brief  Compares two buffers.
  * @param  pBuffer1, pBuffer2: buffers to be compared.
  * @param  BufferLength: buffer's length
  * @retval 1: pBuffer identical to pBuffer1
  *         0: pBuffer differs from pBuffer1
  */
static uint8_t Buffercmp(uint32_t* pBuffer1, uint32_t* pBuffer2, uint16_t BufferLength)
{
    while (BufferLength--) {
        if (*pBuffer1 != *pBuffer2) {
            return 1;
        }

        pBuffer1++;
        pBuffer2++;
    }

    return 0;
}

/**
  * @brief  Print an error message on LCD and stop
  * @param  Message string
  */
static void print_error(uint8_t *msg)
{
    BSP_LCD_SetBackColor(LCD_COLOR_WHITE);
    BSP_LCD_SetTextColor(LCD_COLOR_RED);
    BSP_LCD_DisplayStringAt(0, 200, msg, CENTER_MODE);
    while(1);
}
