/* SpwfInterface NetworkSocketAPI Example Program
 * Copyright (c) 2015 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include <string.h>
#include "easy-connect.h"
#include "MQTTClient.h"
#include "XNucleoIKS01A2.h"
#include "XNucleoNFC01A1.h"
#include "NDefNfcTag.h"
#include "NDefLib/RecordType/RecordURI.h"
#include "RecordWifiConf.h"
#include "MQTTmbed.h"
#include "MQTTNetwork.h"

/****  System configuration define   ****/
#define ORG_QUICKSTART           // comment to connect to play.internetofthings.ibmcloud.com
//#define SUBSCRIBE              // uncomment to subscribe to broker msgs (not to be used with IBM broker) 
#define X_NUCLEO_NFC01A1_PRESENT // uncomment to add NFC support
#ifndef ORG_QUICKSTART
//#define TLS_EN                   // uncomment to add TLS to NON quickstart connections
#endif

#ifdef TLS_EN     // Digicert Root Certificate in PEM format (from IBM website)
const char SSL_CA_PEM[] ="-----BEGIN CERTIFICATE-----\n"
"MIIDrzCCApegAwIBAgIQCDvgVpBCRrGhdWrJWZHHSjANBgkqhkiG9w0BAQUFADBh\n"
"MQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3\n"
"d3cuZGlnaWNlcnQuY29tMSAwHgYDVQQDExdEaWdpQ2VydCBHbG9iYWwgUm9vdCBD\n"
"QTAeFw0wNjExMTAwMDAwMDBaFw0zMTExMTAwMDAwMDBaMGExCzAJBgNVBAYTAlVT\n"
"MRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5j\n"
"b20xIDAeBgNVBAMTF0RpZ2lDZXJ0IEdsb2JhbCBSb290IENBMIIBIjANBgkqhkiG\n"
"9w0BAQEFAAOCAQ8AMIIBCgKCAQEA4jvhEXLeqKTTo1eqUKKPC3eQyaKl7hLOllsB\n"
"CSDMAZOnTjC3U/dDxGkAV53ijSLdhwZAAIEJzs4bg7/fzTtxRuLWZscFs3YnFo97\n"
"nh6Vfe63SKMI2tavegw5BmV/Sl0fvBf4q77uKNd0f3p4mVmFaG5cIzJLv07A6Fpt\n"
"43C/dxC//AH2hdmoRBBYMql1GNXRor5H4idq9Joz+EkIYIvUX7Q6hL+hqkpMfT7P\n"
"T19sdl6gSzeRntwi5m3OFBqOasv+zbMUZBfHWymeMr/y7vrTC0LUq7dBMtoM1O/4\n"
"gdW7jVg/tRvoSSiicNoxBN33shbyTApOB6jtSj1etX+jkMOvJwIDAQABo2MwYTAO\n"
"BgNVHQ8BAf8EBAMCAYYwDwYDVR0TAQH/BAUwAwEB/zAdBgNVHQ4EFgQUA95QNVbR\n"
"TLtm8KPiGxvDl7I90VUwHwYDVR0jBBgwFoAUA95QNVbRTLtm8KPiGxvDl7I90VUw\n"
"DQYJKoZIhvcNAQEFBQADggEBAMucN6pIExIK+t1EnE9SsPTfrgT1eXkIoyQY/Esr\n"
"hMAtudXH/vTBH1jLuG2cenTnmCmrEbXjcKChzUyImZOMkXDiqw8cvpOp/2PV5Adg\n"
"06O/nVsJ8dWO41P0jmP6P6fbtGbfYmbW0W5BjfIttep3Sp+dWOIrWcBAI+0tKIJF\n"
"PnlUkiaY4IBIqDfv8NZ5YBberOgOzW6sRBc4L0na4UU+Krk2U886UAb3LujEV0ls\n"
"YSEY1QSteDwsOoBrp+uvFRTp2InBuThs4pFsiv9kuXclVzDAGySj4dzp30d8tbQk\n"
"CAUw7C29C79Fv1C5qfPrmAESrciIxpg0X40KPMbp1ZWVbd4=\n"
"-----END CERTIFICATE-----\n";
#endif

//------------------------------------
// Hyperterminal configuration
// 9600 bauds, 8-bit data, no parity
//------------------------------------
static Serial pc(SERIAL_TX, SERIAL_RX); 
static DigitalOut myled(LED1);
static bool quickstartMode = true;      // set to false to connect with authentication tocken
static bool BlueButtonToggle = false;  
    
#define MQTT_MAX_PACKET_SIZE 400   
#define MQTT_MAX_PAYLOAD_SIZE 300 

 // Configuration values needed to connect to IBM IoT Cloud
#ifdef ORG_QUICKSTART
#define ORG "quickstart"     // connect to quickstart.internetofthings.ibmcloud.com/ For a registered connection, replace with your org 
#define ID ""
#define AUTH_TOKEN ""
//#define DEFAULT_TYPE_NAME "iotsample-mbed-Nucleo"
#define DEFAULT_TYPE_NAME "sensor"
#define DEFAULT_PORT  MQTT_PORT
#else   // not def ORG_QUICKSTART
#define ORG MQTT_ORG_ID            // connect to ORG.internetofthings.ibmcloud.com/ For a registered connection, replace with your org
#define ID MQTT_DEVICE_ID          // For a registered connection is your device id
#define AUTH_TOKEN  MQTT_DEVICE_PASSWORD  // For a registered connection is a device auth-token
#define DEFAULT_TYPE_NAME  MQTT_DEVICE_TYPE  // For a registered connection is device type
#ifdef TLS_EN
#define DEFAULT_PORT  MQTT_TLS_PORT
#else
#define DEFAULT_PORT  MQTT_PORT
#endif
#endif

#define TYPE DEFAULT_TYPE_NAME       // For a registered connection, replace with your type
#define IBM_IOT_PORT  DEFAULT_PORT
 
#define MAXLEN_MBED_CONF_APP_WIFI_SSID       32  // same as WIFI_SSID_MAX_LEN in easy_connect
#define MAXLEN_MBED_CONF_APP_WIFI_PASSWORD   64  // same as WIFI_PASSWORD_MAX_LEN
 
static char id[30] = ID;                 // mac without colons  
static char org[12] = ORG;        
static int connack_rc = 0; // MQTT connack return code
static char type[30] = TYPE;
static char auth_token[30] = AUTH_TOKEN; // Auth_token is only used in non-quickstart mode
static bool netConnecting = false;
static int connectTimeout = 1000;
static bool mqttConnecting = false;
static bool netConnected = false;
static bool connected = false;
static int retryAttempt = 0;
static char subscription_url[MQTT_MAX_PAYLOAD_SIZE];
static char  ssid[MAXLEN_MBED_CONF_APP_WIFI_SSID];     // Network must be visible otherwise it can't connect
static char  seckey[MAXLEN_MBED_CONF_APP_WIFI_PASSWORD]; 

static LPS22HBSensor *pressure_sensor;
static HTS221Sensor  *humidity_sensor;
static HTS221Sensor  *temp_sensor1;

#ifdef X_NUCLEO_NFC01A1_PRESENT  
// Read from NFC the Wifi record
void NFCReadRecordWIFI (XNucleoNFC01A1 *nfcNucleo) {   
   
    NDefLib::NDefNfcTag& tag = nfcNucleo->get_M24SR().get_NDef_tag();

    printf ("Write to NFC tag the WiFi record ...\n\r");   
    for (int ReadSSIDPassw=0; ReadSSIDPassw!=1; wait_ms(1000)) {   
    //open the i2c session with the nfc chip
	    NDefLib::Message readMsg;
        if(tag.open_session(1)){
        tag.read(&readMsg);
//        printf ("---- N record %d\n\r", readMsg.get_N_records());		
        if(readMsg.get_N_records()==0){
            printf("Error Read\r\n");
        }else {
            for(uint32_t i=0;i<readMsg.get_N_records();i++){
                NDefLib::Record *r = readMsg[i];
//              printf ("N record %d\n\r", readMsg.get_N_records());
				if (r != NULL) {
//					printf ("Record RecordType_t: %d\n\r", r->get_type());
                    if (r->get_type() == NDefLib::Record::TYPE_WIFI_CONF) {
                        NDefLib::RecordWifiConf * temp = (NDefLib::RecordWifiConf*)r;
                        sprintf (ssid, "%s", temp->get_network_ssid().c_str());
                        sprintf (seckey, "%s", temp->get_network_key().c_str());
                        printf ("Read SSID: %s Passw: %s\n\r", ssid, /*seckey*/"*****"); 
						ReadSSIDPassw =1;
                    }                        
                    else if (r->get_type() == NDefLib::Record::TYPE_UNKNOWN) { printf ("NFC RECORD TYPE_UNKNOWN\n\r"); }
                }
                if (r != NULL) delete r;
            }//for all the NFC records
        }//nfc n records              
        //close the i2c session
        if(!tag.close_session()){
            printf("Error Closing the session\r\n");
        }
     }else printf("Error open Session\r\n");  
   }    
}

// add to NFC the HTTP_BROKER_URL URL swapping the two NFC records in order to set the HTTP_BROKER_URL as a first 
// record allowing to read URL from any device. 
void NFCWriteRecordURI (XNucleoNFC01A1 *nfcNucleo) {
         
    NDefLib::NDefNfcTag& tag = nfcNucleo->get_M24SR().get_NDef_tag();
    //open the i2c session with the nfc chip
    if(tag.open_session()){
       //create the NDef message and record
        NDefLib::Message msg;
		tag.read(&msg);
//		printf ("---- N record present: %d\n\r", msg.get_N_records());	
		NDefLib::Record *r = msg[0];
        if ((r != NULL) && (r->get_type() == NDefLib::Record::TYPE_WIFI_CONF)) {  // first time record 0 must be wifi
 			NDefLib::RecordWifiConf * const pWIFI = (NDefLib::RecordWifiConf*)r;  
            NDefLib::Message msg2;				
            NDefLib::RecordURI rURL(NDefLib::RecordURI::HTTPS, subscription_url);
            msg2.add_record(&rURL);   // add a new HTTP record to msg2
            msg2.add_record(pWIFI);	  // copy a Wifi record to msg2
            //write the tag
            if(tag.write(msg2)){
                printf("Tag writed \r\n");
            }
		    NDefLib::Message::remove_and_delete_all_record(msg);										                
		}
        //close the i2c session
        if(!tag.close_session()){
            printf("Error Closing the session\r\n");
        }				
     }else printf("Error open Session\r\n"); 
}        

#endif

#ifndef TARGET_SENSOR_TILE
static void BlueButtonPressed ()
{
	BlueButtonToggle = true;	
}
#endif

#ifdef SUBSCRIBE
void subscribe_cb(MQTT::MessageData & msgMQTT) {
    char msg[MQTT_MAX_PAYLOAD_SIZE];
    msg[0]='\0';
    strncat (msg, (char*)msgMQTT.message.payload, msgMQTT.message.payloadlen);
    printf ("--->>> subscribe_cb msg: %s\n\r", msg);
}

int subscribe(char *pubTopic, MQTT::Client<MQTTNetwork, Countdown, MQTT_MAX_PACKET_SIZE>* client)
{
    return client->subscribe(pubTopic, MQTT::QOS1, subscribe_cb);
}
#endif

int connect(MQTT::Client<MQTTNetwork, Countdown, MQTT_MAX_PACKET_SIZE>* client, MQTTNetwork *mqttNetwork, NetworkInterface* network)
{
    const char* iot_ibm = MQTT_BROKER_URL;     
    char hostname[strlen(org) + strlen(iot_ibm) + 1];
    
    sprintf(hostname, "%s%s", org, iot_ibm);
    // Construct clientId - d:org:type:id
    char clientId[strlen(org) + strlen(type) + strlen(id) + 5];  
    sprintf(clientId, "d:%s:%s:%s", org, type, id);  
    sprintf(subscription_url, "%s.%s/#/device/%s/%s/", org, "internetofthings.ibmcloud.com", id, TYPE);

    // Network debug statements 
    LOG("=====================================\n\r");
    LOG("Nucleo IP ADDRESS: %s\n\r", network->get_ip_address());
    LOG("Nucleo MAC ADDRESS: %s\n\r", network->get_mac_address());
    LOG("Server Hostname: %s port: %d\n\r", hostname, IBM_IOT_PORT);
//    for(int i = 0; clientId[i]; i++){  // set lowercase mac
//       clientId[i] = tolower(clientId[i]); 
//    }    
    LOG("Client ID: %s\n\r", clientId);
    LOG("Topic: %s\n\r",MQTT_TOPIC);
    LOG("Subscription URL: %s\n\r", subscription_url);
    LOG("=====================================\n\r");    
    netConnecting = true;

#ifdef ORG_QUICKSTART
int tls = TLS_OFF;
const char * cert = NULL;
unsigned int sizeof_cert = 0;
#else  // if !QUICKSTART possible to connect with TLS or not
#ifdef TLS_EN
int tls = TLS_ON;
const char * cert = SSL_CA_PEM;
unsigned int sizeof_cert = sizeof(SSL_CA_PEM);
#else
int tls = TLS_OFF;
const char * cert = 0;
unsigned int sizeof_cert = 0;
#endif
#endif

    int rc = mqttNetwork->connect(hostname, IBM_IOT_PORT, tls, cert, sizeof_cert);
    if (rc != 0)
    {
        printf("rc from TCP connect is %d\r\n", rc);
        return rc;
    }
    netConnected = true;
    netConnecting = false;		

    // MQTT Connect
    mqttConnecting = true;
    MQTTPacket_connectData data = MQTTPacket_connectData_initializer;
    data.MQTTVersion = 4;
    data.struct_version=0;
    data.clientID.cstring = clientId; 
	  data.keepAliveInterval = 0; //MQTT_KEEPALIVE;  // in Sec
    if (!quickstartMode) 
    {        
        data.username.cstring = "use-token-auth";
        data.password.cstring = auth_token;
        printf ("AutToken: %s\n\r", auth_token);
    }   
    if ((rc = client->connect(data)) != MQTT::SUCCESS) {
        printf("rc from MQTT connect is %d\r\n", rc);
        connack_rc = rc;
        return rc;
    }
    connected = true;
    printf ("--->MQTT Connected\n\r"); 
#ifdef SUBSCRIBE
		    int rc=0;
        if ((rc=subscribe(MQTT_TOPIC, client)) == 0) LOG ("--->>>MQTT subscribed to: %s\n\r",MQTT_TOPIC);
		    else LOG ("--->>>ERROR MQTT subscribe : %s\n\r",MQTT_TOPIC);
#endif    
    mqttConnecting = false;
    connack_rc = rc;
    return rc;       
}


int getConnTimeout(int attemptNumber)
{  // First 10 attempts try within 3 seconds, next 10 attempts retry after every 1 minute
   // after 20 attempts, retry every 10 minutes
    return (attemptNumber < 10) ? 3 : (attemptNumber < 20) ? 60 : 600;
}


void attemptConnect(MQTT::Client<MQTTNetwork, Countdown, MQTT_MAX_PACKET_SIZE>* client, MQTTNetwork *mqttNetwork, NetworkInterface* network) 
{
    connected = false;
           
    while (connect(client, mqttNetwork, network) != MQTT_CONNECTION_ACCEPTED) 
    {    
        if (connack_rc == MQTT_NOT_AUTHORIZED || connack_rc == MQTT_BAD_USERNAME_OR_PASSWORD) {
            printf ("Error MQTT_BAD_USERNAME_OR_PASSWORDFile: %s, Line: %d Error: %d \n\r",__FILE__,__LINE__, connack_rc);        
            return; // don't reattempt to connect if credentials are wrong
        } 
        int timeout = getConnTimeout(++retryAttempt);
        WARN("Retry attempt number %d waiting %d\n", retryAttempt, timeout);
        
        // if ipstack and client were on the heap we could deconstruct and goto a label where they are constructed
        //  or maybe just add the proper members to do this disconnect and call attemptConnect(...)        
        // this works - reset the system when the retry count gets to a threshold
        if (retryAttempt == 5)
            NVIC_SystemReset(); 
        else
            wait(timeout);
    }    
}

int publish(MQTT::Client<MQTTNetwork, Countdown, MQTT_MAX_PACKET_SIZE>* client)
{
    MQTT::Message message;
    const char* pubTopic = MQTT_TOPIC;
            
    char buf[MQTT_MAX_PAYLOAD_SIZE];
    float temp, press, hum;

	if (!client->isConnected()) { printf ("publish failed: MQTT disconnected\n\r"); return MQTT::FAILURE; }
    temp_sensor1->get_temperature(&temp);
    pressure_sensor->get_pressure(&press);
    humidity_sensor->get_humidity(&hum);
    sprintf(buf,
     "{\"d\":{\"ST\":\"Nucleo-IoT-mbed\",\"Temp\":%0.4f,\"Pressure\":%0.4f,\"Humidity\":%0.4f}}",
              temp, press, hum);
    message.qos = MQTT::QOS0;
    message.retained = false;
    message.dup = false;
    message.payload = (void*)buf;
    message.payloadlen = strlen(buf);

    if( (message.payloadlen + strlen(pubTopic)+1) >= MQTT_MAX_PACKET_SIZE )
        printf("message too long!\r\n");
    
    LOG("Publishing %s\n\r", buf);
    return client->publish(pubTopic, message);    
}

 
int main()
{   
   myled=0;
   DevI2C *i2c = new DevI2C(I2C_SDA, I2C_SCL);
   i2c->frequency(400000);    

   XNucleoIKS01A2 *mems_expansion_board = XNucleoIKS01A2::instance(i2c);  
   pressure_sensor = mems_expansion_board->pt_sensor;
   temp_sensor1 = mems_expansion_board->ht_sensor;  
   humidity_sensor = mems_expansion_board->ht_sensor; 
   pressure_sensor->enable();
   temp_sensor1->enable();
   humidity_sensor->enable();
    
#if !defined (TARGET_SENSOR_TILE)
    InterruptIn BlueButton(USER_BUTTON);    
    BlueButton.fall(&BlueButtonPressed);
	  BlueButtonToggle = false;
#endif    
	
   pc.printf("\r\nCloud_IBM_MbedOS Application\r\n");     
#if defined(MBED_MAJOR_VERSION)
    printf("Using Mbed OS %d.%d.%d\n", MBED_MAJOR_VERSION, MBED_MINOR_VERSION, MBED_PATCH_VERSION);
#else
    printf("Using Mbed OS from master.\n");
#endif	
   pc.printf("\r\nconnecting to AP\r\n");            

   quickstartMode=false;
   if (strcmp(org, "quickstart") == 0){quickstartMode = true;}
   
#ifdef X_NUCLEO_NFC01A1_PRESENT      
   // program NFC with broker URL        
   XNucleoNFC01A1 *nfcNucleo = XNucleoNFC01A1::instance(*i2c, NULL, XNucleoNFC01A1::DEFAULT_GPO_PIN, XNucleoNFC01A1::DEFAULT_RF_DISABLE_PIN, NC,NC,NC);  
   NDefLib::NDefNfcTag& tag = nfcNucleo->get_M24SR().get_NDef_tag();
   printf("NFC Init done: !\r\n");
   //open the i2c session with the nfc chip
   NFCReadRecordWIFI (nfcNucleo);
   NetworkInterface* network = easy_connect(true, ssid, seckey);  // Wifi SSID and passw from NFC tag
#else
   NetworkInterface* network = easy_connect(true);   // // Wifi SSID and passw from mbed_app.json
#endif    
   if (!network) {
       printf ("Error easy_connect\n\r");
       return -1;
   }  
//=================  TODO Set System Time ideally from NTP srv or from shell
#if 0
    time_t ctTime;
    ctTime = time(NULL);             
    printf ("Start Secure Socket connection with one way server autentication test\n\r");                
    printf("Initial System Time is: %s\r\n", ctime(&ctTime));   
    printf("Need to adjust time? if yes enter time in seconds elapsed since Epoch (cmd: date +'%%s'), otherwise enter 0 ");                
    int t=0;
    scanf("%d",&t);
    printf ("entered time is: %d \n\r", t);
    if (t != 0) { time_t txTm = t; set_time(txTm); }  // set Nucleo system time
    ctTime = time(NULL);    
    printf ("The current system time is: %s", ctime (&ctTime));  // set WiFi module systm time
    WiFiInterface* wifi = easy_get_wifi(0);
    if (!((SpwfSAInterface*)wifi)->set_time(ctTime)) printf ("ERROR set_time\n\r");  
#endif    
//=================   
   MQTTNetwork mqttNetwork(network);	
   MQTT::Client<MQTTNetwork, Countdown, MQTT_MAX_PACKET_SIZE> client(mqttNetwork);
		
   if (quickstartMode){
       char mac[50];  // remove all : from mac
       char *digit=NULL;
       sprintf (id,"%s", "");                
       sprintf (mac,"%s",network->get_mac_address()); 
       digit = strtok (mac,":");
       while (digit != NULL)
       {
           strcat (id, digit);
           digit = strtok (NULL, ":");
       }     
   }
   printf ("ATTEMPT CONNECT\n\r");
   attemptConnect(&client, &mqttNetwork, network);
   if (connack_rc == MQTT_NOT_AUTHORIZED || connack_rc == MQTT_BAD_USERNAME_OR_PASSWORD)    
   {
       printf ("---ERROR line : %d\n\r", __LINE__);
       while (true)
       wait(1.0); // Permanent failures - don't retry
   }

#ifdef X_NUCLEO_NFC01A1_PRESENT 
   NFCWriteRecordURI (nfcNucleo);
#endif   
   myled=1;         
   int count = 0;    
   while (true)
   {
		 if (BlueButtonToggle == false && connected == true) {
       if (++count == 6)
       {   
				 // Publish a message every 3 second
           if (publish(&client) != MQTT::SUCCESS) { 
               myled=0;
			         count=0;
               client.disconnect();
			         mqttNetwork.disconnect();			   
               attemptConnect(&client, &mqttNetwork, network);   // if we have lost the connection                
           } else {					 
			        myled=1;
              count=0;
					 }
       }        
       if (client.isConnected()) client.yield(500);  // allow the MQTT client to receive subscribe messages and manage keep alive
		 } else if (BlueButtonToggle == true && connected == true){   // disconnect MQTT
			 printf ("--->> MQTT Disconnect\n\r");
       connected = false;
			 myled=0;
			 count = 0;
			 BlueButtonToggle = false;
#ifdef SUBSCRIBE			 
//			 unsubscribe(const char* topicFilter);   // unsubscribe if subscribed
#endif			 
			 client.disconnect();
       printf ("--->> TCP Disconnect\n\r");
			 mqttNetwork.disconnect();
		 } else if (BlueButtonToggle == true && connected == false) {
			 attemptConnect(&client, &mqttNetwork, network); 
			 connected = true;
			 BlueButtonToggle = false;
	 } else wait (0.5);
   }
}
    