/**
  ******************************************************************************
  * @file    CustomConfigService.h
  * @author  Central Labs / AST
  * @version V0.9.0
  * @date    23-Dec-2015
  * @brief   BLE calibration service
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */

#ifndef __CUSTOM_BLE_CONFIG_SERVICE_H__
#define __CUSTOM_BLE_CONFIG_SERVICE_H__
#include "BLE.h"
#include "UUID.h"
                              
#define STORE_BE_32(buf, val)    ( ((buf)[3] =  (uint8_t) (val)     ) , \
                                   ((buf)[2] =  (uint8_t) (val>>8)  ) , \
                                   ((buf)[1] =  (uint8_t) (val>>16) ) , \
                                   ((buf)[0] =  (uint8_t) (val>>24) ) )                         
                            
#define SIZEOF_CONFIG_DATA_LEN  2+4+1+1

const LongUUIDBytes_t CONFIG_SERVICE_UUID_128 = {0x00,0x00,0x00,0x00,0x00,0x0F,0x11,0xe1,0x9a,0xb4,0x00,0x02,0xa5,0xd5,0xc5,0x1b};
const LongUUIDBytes_t CONFIG_W2ST_CHAR_UUID   = {0x00,0x00,0x00,0x02,0x00,0x0F,0x11,0xe1,0xac,0x36,0x00,0x02,0xa5,0xd5,0xc5,0x1b};

/* Custom Config Service */
class CustomConfigService  {
public:
    CustomConfigService(BLE &_ble) :
        ble(_ble),
                configw2stCharacteristic(CONFIG_W2ST_CHAR_UUID, configData, SIZEOF_CONFIG_DATA_LEN, SIZEOF_CONFIG_DATA_LEN,
                                                                    GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY |
                                                                    GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE_WITHOUT_RESPONSE
) {  

        static bool serviceAdded = false; /* We should only ever need to add the heart rate service once. */
        if (serviceAdded) {
            return;
        }

        GattCharacteristic *charTable[] = {&configw2stCharacteristic};

        GattService   configService(CONFIG_SERVICE_UUID_128, charTable, sizeof(charTable) / sizeof(GattCharacteristic *));              
                
        ble.gattServer().addService(configService);                              
                
        isEnabledConfNotify = false;
        memset (configData, 0, SIZEOF_CONFIG_DATA_LEN); 
        serviceAdded = true;
    }

    
    uint32_t sendConfigState(uint32_t Feature, uint8_t Command, uint8_t val, uint16_t TimeStamp) {      
            STORE_LE_16(configData ,TimeStamp); 
            STORE_BE_32(configData+2,Feature);          
            configData[6] = Command;
            configData[7] = (val==0x01) ? 100: val;                     
            return ble.gattServer().write(configw2stCharacteristic.getValueAttribute().getHandle(), configData, SIZEOF_CONFIG_DATA_LEN, 0);
    }
    
  uint32_t updateConfigState(uint32_t Feature, uint8_t Command, uint8_t val, uint16_t TimeStamp) {
        if (ble.getGapState().connected && isEnabledConfNotify ) {  
            return sendConfigState(Feature, Command, val, TimeStamp);
        }
        return 0;
  }

    void updateConnectionStatus(ConnectionStatus_t status) {            
            isEnabledConfNotify = false;
            memset (configData, 0, SIZEOF_CONFIG_DATA_LEN);             
        }   
    
    bool isConfHandle (Gap::Handle_t handle) {
            if (handle == configw2stCharacteristic.getValueAttribute().getHandle() - BLE_HANDLE_VALUE_OFFSET) return true;
            return false;
    }           

    void enNotify (Gap::Handle_t handle) {
            if (isConfHandle(handle - BLE_HANDLE_EN_DIS_OFFSET)) { 
                PRINTF("enNotify! %d\n\r", handle); isEnabledConfNotify = true; return; }    
    }
    
    void disNotify (Gap::Handle_t handle) {
            if (isConfHandle(handle - BLE_HANDLE_EN_DIS_OFFSET)) { 
                            isEnabledConfNotify = false; return; }    
    }   
        
private:
        BLE                              &ble;
        uint8_t                          configData[SIZEOF_CONFIG_DATA_LEN];                
        uint8_t                          configState;
        GattCharacteristic               configw2stCharacteristic;  
        bool                             isEnabledConfNotify;

};

#endif /* #ifndef __CUSTOM_BLE_CONFIG_SERVICE_H__*/
