/* mbed Microcontroller Library
 * Copyright (c) 2006-2015 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "BlueNRGDevice.h"

#define BLUENRG_MS_HW_VERSION (0x31)

// This file defines the array used to store the firmware
#include "img73.c"

DigitalOut led1(LED1);


void error_loop(void)
{
    // Blink fast to signal error condition
    while (1) {
        led1 = !led1;
        wait_ms(80);
    }
}

// This program updates the firmware of the
// BlueNRG-MS chip of the X-NUCLEO-IDB05A1 board.
// The target firmware version is 7.2a
// If the update is successful, LED1 blinks slowly;
// on error, LED1 blinks fast.
int main(void)
{
    uint8_t hw_version;
    uint8_t get_hw_version_status;
    int update_status;
    BlueNRGDevice* bleDevice;

    // Keep LED1 turned on to show that update is in progress
    led1 = 1;

    bleDevice = (BlueNRGDevice*) createBLEInstance();
    get_hw_version_status = bleDevice->getUpdaterHardwareVersion(&hw_version);
    
    if (get_hw_version_status != BLE_STATUS_SUCCESS) {
        printf("ERROR: Unable to get HW version\n\r");
        error_loop();
    }
    
    if ( (hw_version < BLUENRG_MS_HW_VERSION) || (hw_version > BLUENRG_MS_HW_VERSION) ) {
        printf("Sorry, you're HW version is not supported\n\r");
        error_loop();
    }

    printf("Start updating BlueNRG-MS firmware...\n\r");
    
    update_status = bleDevice->updateFirmware(fw_image, sizeof(fw_image));

    if (update_status != BLE_STATUS_SUCCESS) {
        printf("ERROR: Unable to update firmware\n\r");
        error_loop();
    }
    
    printf("Firmware updated successfully!\n\r");
    // Blink slowly to signal successful firmware update
    while (1) {
        led1 = !led1;
        wait(1);
    }
}
