/**
  ******************************************************************************
  * File Name          : main.c
  * Date               : 21/10/2014 09:21:14
  * Description        : Main program body
  ******************************************************************************
  *
  * COPYRIGHT(c) 2014 STMicroelectronics
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  
  This program controls a VL6180 ToF sensor running on an STM32F401 card with the
  VL6180 shield.  This program works in single shot mode where the program waits for
  the measurement finishing. This does not allow for the display to be updated while the
  measurement occurs.
  
  The display is very crude with no storing of the last digits, each digit is written
  and then a wait occurs and then the next digit is written. This means that a lot of time is 
  taken writing the display and any long period when the display is not serviced 
  will cause the display to flicker.
  
  In this program access to the VL6180 api is through wrapper functions in
  vl6180_class.cpp. It is also possible to access the api directly. E.G both the lines below
  do the same thing. 

  */

/* Includes ------------------------------------------------------------------*/
//#include "stm32xxx_hal.h"

/* Private variables ---------------------------------------------------------*/

#include <stdio.h>

#include "mbed.h"
#include "XNucleo6810.h"
#include <time.h>

#include "spi_interface.h"


//VL6180_SINGLE_DEVICE_DRIVER

// i2c comms port pins
#define VL6180_I2C_SDA   D14 
#define VL6180_I2C_SCL   D15 

/* USER CODE BEGIN 0 */
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include "vl6180_api.h"
#include "6180a1.h"

#define DigitDisplay_ms     1 /* ms each digit is kept on */

#define DMaxDispTime     0 // don't calculate dmax to save time

// Define interrupt pins
// The interrupt pins depend on solder blobs on the back of the shield.
// Each interupt can have two possible pins, this allows all interrupts to be on the same gpio or different gpio
// The first values are those an unmodified board has
// see ST document UM2657 for more details
PinName CentreIntPin = A3;
PinName LeftIntPin = D13;
PinName RightIntPin = D2;
PinName BottomIntPin = A2;
// alternate set
//PinName CentreIntPin = A5;
//PinName LeftIntPin = D8;
//PinName RightIntPin = D4;
//PinName BottomIntPin = A4;


    

MyVL6180Dev_t                   devCentre;  //data  for each of the vl6180
MyVL6180Dev_t                   devLeft;
MyVL6180Dev_t                   devRight;
MyVL6180Dev_t                   devBottom;
VL6180Dev_t                     Dev = &devCentre; // the currently used vl6180

volatile int IntrFired=0;

VL6180 *Sensor;  

static XNucleo53L1A1 *board=NULL;

//create serial port
// MBed V6.4 has renamed wait_ms and UnbufferedSerial replaces Serial
#if (MBED_VERSION  > 60300) 
UnbufferedSerial  pc(SERIAL_TX, SERIAL_RX); 
extern "C" void wait_ms(int ms);
#else
Serial pc(SERIAL_TX, SERIAL_RX); 
#endif

//create i2c bus
vl6180_DevI2C *dev_I2C = new vl6180_DevI2C(VL6180_I2C_SDA, VL6180_I2C_SCL);

/**
 * VL6180x i2c porting implementation
 */

#define theVL6180Dev   0x52    // what we use as "API device

#define i2c_bus      (&hi2c1)
#define def_i2c_time_out 100


void XNUCLEO6180XA1_WaitMilliSec(int n){
    wait_ms(n);
}

static char DISP_CurString[10];

char buffer[10];

VL6180_RangeData_t Range;  /* Range measurmeent  */
uint16_t range;             /* range average distance */


int main(void)
{
    pc.baud(115200);  // baud rate is important as printf statements take a lot of time
    printf("VL6180_ss_and_wait_ranging \n");
    devCentre.i2c_addr = theVL6180Dev;  // set to default adress for now

    int status;
    
    // create instances for the sensors
    board = XNucleo53L1A1::instance(dev_I2C, CentreIntPin, LeftIntPin, RightIntPin ,BottomIntPin);  
    
        // find the sensors we have and initialise
    status = board->init_board();
    if (status) {
        printf("Failed to init board!\r\n");
        return 0;
    }
    
    //select centre sensor
    devCentre.i2c_addr = NEW_SENSOR_CENTRE_ADDRESS;
    Dev = &devCentre;
    Sensor=board->sensor_centre;
    if ( board->sensor_centre ==NULL)
        printf("board->sensor_centre = Null\n");

    /* Note that as we waited  1msec we could bypass VL6180_WaitDeviceBooted(&Dev); */
    Sensor->vl6180_WaitDeviceBooted(Dev);
    Sensor->vl6180_InitData(Dev);

    printf("started interrupt centre %d\n",status);
    
    /* Enable Dmax calculation only if value is displayed (to save computation power) */
    Sensor->vl6180_DMaxSetState(Dev, DMaxDispTime>0);
    status = Sensor->vl6180_Prepare(Dev);
    
    printf("start loop \n");
    /* Infinite loop */
    while (1) {
      VL6180_RangeData_t Range;
      status =0;
       status = Sensor->vl6180_RangePollMeasurement(Dev, &Range); /* start measurement and wait for completion*/    
       if (status) { 
           printf("VL6180_RangePollMeasurement %d\n",status);
       }

       if((Range.errorStatus == 0) && (status == 0))
       {
  //           printf("range = %d\n",Range.range_mm);// print out resdult if you want
             sprintf(DISP_CurString, " %d", (int)Range.range_mm);
       } 
       else
       {
             sprintf(DISP_CurString, " %d", 0);
       }
       
       // the display is very simple and requires written to frequently so
       // we are writing to the display when we would normally sleep.
       // when we are not writing to the display it is blank
       for (int i = 0 ;i < 50;i++)
       {
            XNUCLEO6180XA1_DisplayString(DISP_CurString, DigitDisplay_ms* 5);
       }
    }

}


 // wait_ms was removed in MBed V6.4          
#if (MBED_VERSION  > 60300) 
void wait_ms(int ms)
 {
    thread_sleep_for(ms);
 }
#endif
/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/
