/*
 * This VL53L3 Expansion board test application performs range measurements
 * using the onboard embedded centre sensor, in singleshot, polling mode.
 * Measured ranges are ouput on the Serial Port, running at 115200 baud.
 *
 * The User Blue button stops the current measurement and entire program,
 * releasing all resources.
 *
 * The Reset button can be used to restart the program.
 *
 * *** Note :
 * Default Mbed build system settings disable print floating-point support.
 * Offline builds can enable this, again.
 * https://github.com/ARMmbed/mbed-os/blob/master/platform/source/minimal-printf/README.md
 * .\mbed-os\platform\mbed_lib.json
 *
 */

#include <stdio.h>
#include <time.h>

#include "mbed.h"

#include "XNucleo53L3A2.h"
#include "vl53L3_I2c.h"


// define i2c mcu pins
#define I2C_SDA   D14
#define I2C_SCL   D15

static XNucleo53L3A2 *board=NULL;
#if (MBED_VERSION  > 60300)
UnbufferedSerial  pc(USBTX, USBRX);
#else
Serial pc(SERIAL_TX, SERIAL_RX);
#endif



VL53LX_Dev_t                   devCentre;
VL53LX_Dev_t                   devLeft;
VL53LX_Dev_t                   devRight;

VL53LX_DEV                     Dev = &devCentre;



/*=================================== Main ==================================
=============================================================================*/
int main()
{
    int status;
    VL53L3 * Sensor;
    uint16_t wordData;

    static VL53LX_MultiRangingData_t RangingData;


    pc.baud(115200);  // baud rate is important as printf statements take a lot of time
    printf("Hello world!\r\n");

    vl53L3_DevI2C *dev_I2C = new vl53L3_DevI2C(I2C_SDA, I2C_SCL);

    /* creates the 53L1A1 expansion board singleton obj */
    board = XNucleo53L3A2::instance(dev_I2C, A2, D8, D2);
    printf("board created!\r\n");


    /* init the 53L1A1 expansion board with default values */
    status = board->init_board();
    if (status) {
        printf("Failed to init board!\r\n");
        return 0;
    }


    printf("board initiated! \n");

    printf("configuring centre channel \n");
    Dev=&devCentre;
    Sensor=board->sensor_centre;
    Dev->I2cDevAddr = NEW_SENSOR_CENTRE_ADDRESS;
    printf("configured centre channel \n");


    // configure the i2c connection
    Dev->comms_speed_khz = 400;
    Dev->comms_type = 1;

    /* Device Initialization and setting */

    status = Sensor->VL53LX_DataInit();
    uint8_t NewDataReady=0;


    status = Sensor->VL53LX_StartMeasurement();
    printf("VL53LX_StartMeasurement %d \n",status);

    while(1) {
        status = Sensor->VL53LX_WaitMeasurementDataReady();

        if(!status) {

            status = Sensor->VL53LX_GetMultiRangingData( &RangingData);

            if ( status == 0) {
                int no_of_object_found=RangingData.NumberOfObjectsFound;
                if ( no_of_object_found < 10 ) {
                    for(int j=0; j<no_of_object_found; j++) {
                        if ((RangingData.RangeData[j].RangeStatus == VL53LX_RANGESTATUS_RANGE_VALID) ||
                                (RangingData.RangeData[j].RangeStatus == VL53LX_RANGESTATUS_RANGE_VALID_NO_WRAP_CHECK_FAIL)) {
                            // print data
                            printf("centre \t object %d  \t status=%d, \t D=%5dmm, \t Signal=%2.2f Mcps, \t Ambient=%2.2f Mcps \n",
                                   j,
                                   RangingData.RangeData[j].RangeStatus,
                                   RangingData.RangeData[j].RangeMilliMeter,
                                   (RangingData.RangeData[j].SignalRateRtnMegaCps/65535.0),
                                   (RangingData.RangeData[j].AmbientRateRtnMegaCps/65535.0));
                        } //if
                    } //for
                } // if  ( no_of_object_found < 10 )
            }   // if status VL53LX_GetMultiRangingData

        } // if !status VL53LX_WaitMeasurementDataReady
        else {
            printf("VL53L1_WaitMeasurementDataReady failed %d \n",status);
        }

        status = Sensor->VL53LX_ClearInterruptAndStartMeasurement();

    } // while(1)

    // *** will not reach, after infinite loop.
    status = Sensor->VL53LX_StopMeasurement();

}

