/*
 * This VL53L3 Expansion board test application performs range measurements
 * using the onboard embedded sensor, and satellite boards, in interrupt mode.
 * Measured ranges are output on the Serial Port, running at 115200 baud.
 *
 * The Reset button can be used to restart the program.
 *
 * *** Note :
 * Default Mbed build system settings disable printf floating-point support.
 * Online builds seem unable to configure this.
 * Offline builds can enable printf floating-point support.
 * https://github.com/ARMmbed/mbed-os/blob/master/platform/source/minimal-printf/README.md
 * .\mbed-os\platform\mbed_lib.json
 *
 */

#include <stdio.h>
#include <time.h>

#include "mbed.h"

#include "XNucleo53L3A2.h"
#include "vl53L3_I2c.h"


#define I2C_SDA   D14
#define I2C_SCL   D15

#define MEASUREMENTTIMING  2 // 5  // 10 // 20 // 30 // 55

static XNucleo53L3A2 *board=NULL;
#if (MBED_VERSION  > 60300)
UnbufferedSerial  pc(USBTX, USBRX);
extern "C" void wait_ms(int ms);
#else
Serial pc(SERIAL_TX, SERIAL_RX);
#endif

static int int_centre_result = 0;
static int int_left_result = 0;
static int int_right_result = 0;


class WaitForMeasurement
{

public:


// this class services the interrupts from the ToF sensors.
// There is a limited amount you can do in an interrupt routine; printfs,mutexes break them among other things.
// We keep things simple by only raising a flag so all the real work is done outside the interrupt.
// This is designed around MBED V2 which doesn't have the RTOS features that would make this work nicely e.g. semaphores/queues.
    WaitForMeasurement(): _interrupt(A1)
    {
    }


    // constructor - Sensor is not used and can be removed
    WaitForMeasurement(PinName pin,VL53LX_DEV Dev) : _interrupt(pin)          // create the InterruptIn on the pin specified to Counter
    {
        Devlocal = Dev;
        _interrupt.rise(callback(this, &WaitForMeasurement::got_interrupt)); // attach increment function of this counter instance

    }

    void process_right_interrupt()
    {
        printf("processing right interrupt\n");
    }

    // function is called every time an interupt is seen. A flag is raised which allows the main routine to service the interupt.
    void got_interrupt()
    {
        _count++;

        if (Devlocal->I2cDevAddr == NEW_SENSOR_CENTRE_ADDRESS)
            int_centre_result = 1;  //flag to main that interrupt happened
        if (Devlocal->I2cDevAddr == NEW_SENSOR_LEFT_ADDRESS)
            int_left_result = 1;   //flag to main that interrupt happened
        if (Devlocal->I2cDevAddr == NEW_SENSOR_RIGHT_ADDRESS)
            int_right_result = 1;  //flag to main that interrupt happened
    }


    //destructor
    ~WaitForMeasurement()
    {
        printf("destruction \n");
    }

private:

    InterruptIn _interrupt;
    volatile int _count;
    VL53LX_DEV Devlocal;
    int status;
};



VL53LX_Dev_t devCentre;
VL53LX_Dev_t devLeft;
VL53LX_Dev_t devRight;
VL53LX_DEV Dev = &devCentre;





/*=================================== Main ==================================
=============================================================================*/
int main()
{
    int status;
    VL53L3 *Sensor;
    uint8_t ToFSensor = 1; // 0=Left, 1=Center(default), 2=Right

    WaitForMeasurement *int2;
    WaitForMeasurement *int1;
    WaitForMeasurement *int3;


    vl53L3_DevI2C *dev_I2C = new vl53L3_DevI2C(I2C_SDA, I2C_SCL);
    /* creates the 53L1A1 expansion board singleton obj */
    board = XNucleo53L3A2::instance(dev_I2C, A2, D8, D2);
    printf("board created!\r\n");

    pc.baud(115200);  // baud rate is important as printf statements take a lot of time
    
    /* init the 53L1A1 expansion board with default values */
    status = board->init_board();
    if (status) {
        printf("Failed to init board!\r\n");
        return 0;
    }

    printf("board initiated! - %d\r\n", status);

    for (ToFSensor = 0; ToFSensor < 3; ToFSensor++) {
        wait_ms(15);
        switch(ToFSensor) {
            case 0:
                if (board->sensor_centre == NULL ) continue;
                Dev=&devCentre;
                Dev->I2cDevAddr = NEW_SENSOR_CENTRE_ADDRESS;
                Sensor = board->sensor_centre;
                printf("configuring centre channel \n");
                break;
            case 1:
                if (board->sensor_left == NULL ) continue;
                Dev=&devLeft;
                Dev->I2cDevAddr = NEW_SENSOR_LEFT_ADDRESS;
                Sensor = board->sensor_left;
                printf("configuring left channel \n");
                break;
            case 2:
                if (board->sensor_right == NULL ) continue;
                Dev=&devRight;
                Dev->I2cDevAddr = NEW_SENSOR_RIGHT_ADDRESS;
                Sensor = board->sensor_right;
                printf("configuring right channel \n");
                break;
            default:
                printf(" error in switch, invalid ToF sensor \n");
        }

// configure the sensors
        Dev->comms_speed_khz = 400;
        Dev->comms_type = 1;

        /* Device Initialization and setting */
        status = Sensor->VL53LX_DataInit();
        status = Sensor->VL53LX_SetDistanceMode(VL53LX_DISTANCEMODE_LONG);
        status = Sensor->VL53LX_SetMeasurementTimingBudgetMicroSeconds( MEASUREMENTTIMING * 1000);
        status = Sensor->VL53LX_SmudgeCorrectionEnable(VL53LX_SMUDGE_CORRECTION_SINGLE);
        status = Sensor->VL53LX_SetXTalkCompensationEnable(1);
    }


    if (board->sensor_centre != NULL ) {
        printf("starting interrupt centre\n");
        Sensor = board->sensor_centre;
        devCentre.I2cDevAddr = NEW_SENSOR_CENTRE_ADDRESS;
        int1 = new WaitForMeasurement(A2,&devCentre);
        status = Sensor->VL53LX_StartMeasurement();
        printf("VL53L1_StartMeasurement %d \n",status);
        status = Sensor->VL53LX_ClearInterruptAndStartMeasurement();
    }


    if (board->sensor_left != NULL ) {
        printf("starting interrupt left\n");
        Sensor = board->sensor_left;
        devLeft.I2cDevAddr = NEW_SENSOR_LEFT_ADDRESS;
        int2 = new WaitForMeasurement(D8,&devLeft);
        status = Sensor->VL53LX_StartMeasurement();
        printf("VL53L1_StartMeasurement %d \n",status);
        status = Sensor->VL53LX_ClearInterruptAndStartMeasurement();
    }

    if (board->sensor_right != NULL ) {
        printf("starting interrupt right\n");
        Sensor = board->sensor_right;
        devRight.I2cDevAddr = NEW_SENSOR_RIGHT_ADDRESS;
        int3 = new WaitForMeasurement(D2,&devRight);
        status = Sensor->VL53LX_StartMeasurement();
        printf("VL53L1_StartMeasurement %d \n",status);
        status = Sensor->VL53LX_ClearInterruptAndStartMeasurement();
    }


    VL53LX_MultiRangingData_t MultiRangingData;
    VL53LX_MultiRangingData_t *pMultiRangingData;
    
    // loop waiting for interrupts to happen. This is signaled by   int_centre_result,int_left_result or int_right_result
    // being non zero. The are set back to zero when processing is completed
    while (1) {
        pMultiRangingData = &MultiRangingData;

        wait_ms(5); // 10

        if (int_centre_result != 0) {
            status = board->sensor_centre->VL53LX_GetMultiRangingData(pMultiRangingData);
            int no_of_object_found = pMultiRangingData->NumberOfObjectsFound;
            if ((no_of_object_found < 10) && (no_of_object_found != 0)) {
                for(int j = 0; j < no_of_object_found; j++) {
                    if (pMultiRangingData->RangeData[j].RangeStatus == 0) {
                        printf("centre\t status=%d, \t D=%5dmm, \t Signal=%2.2f Mcps, \t Ambient=%2.2f Mcps \n",
                               pMultiRangingData->RangeData[j].RangeStatus,
                               pMultiRangingData->RangeData[j].RangeMilliMeter,
                               (pMultiRangingData->RangeData[j].SignalRateRtnMegaCps / 65535.0),
                               (pMultiRangingData->RangeData[j].AmbientRateRtnMegaCps / 65535.0));
                    }
                }
            }
            // clear the interrupt and wait for another result
            int_centre_result = 0;
            wait_ms( MEASUREMENTTIMING );
            status = board->sensor_centre->VL53LX_ClearInterruptAndStartMeasurement();
        }


        if (int_left_result != 0) {
            status = board->sensor_left->VL53LX_GetMultiRangingData(pMultiRangingData);
            if (status == 0) {
                int no_of_object_found = pMultiRangingData->NumberOfObjectsFound;
                if ((no_of_object_found < 10) && (no_of_object_found != 0)) {
                    for(int j = 0; j < no_of_object_found; j++) {
                        if (pMultiRangingData->RangeData[j].RangeStatus == 0) {
                            printf("left  \t status=%d, \t D=%5dmm, \t Signal=%2.2f Mcps, \t Ambient=%2.2f Mcps \n",
                                   pMultiRangingData->RangeData[j].RangeStatus,
                                   pMultiRangingData->RangeData[j].RangeMilliMeter,
                                   (pMultiRangingData->RangeData[j].SignalRateRtnMegaCps / 65535.0),
                                   (pMultiRangingData->RangeData[j].AmbientRateRtnMegaCps / 65535.0) );
                        }
                    }
                }
            }
            // clear the interrupt and wait for another result
            int_left_result = 0;
            wait_ms( MEASUREMENTTIMING );
            status = board->sensor_left->VL53LX_ClearInterruptAndStartMeasurement();
        }


        if (int_right_result != 0) {
            status = board->sensor_right->VL53LX_GetMultiRangingData(pMultiRangingData);
            if (status == 0) {
                // if valid result print it
                int no_of_object_found = pMultiRangingData->NumberOfObjectsFound;
                if ((no_of_object_found < 10) && (no_of_object_found != 0)) {
                    for(int j = 0; j < no_of_object_found; j++) {
                        if (pMultiRangingData->RangeData[j].RangeStatus == 0) {
                            printf("right \t status=%d, \t D=%5dmm, \t Signal=%2.2f Mcps, \t Ambient=%2.2f Mcps \n",
                                   pMultiRangingData->RangeData[j].RangeStatus,
                                   pMultiRangingData->RangeData[j].RangeMilliMeter,
                                   (pMultiRangingData->RangeData[j].SignalRateRtnMegaCps / 65535.0),
                                   (pMultiRangingData->RangeData[j].AmbientRateRtnMegaCps / 65535.0) );
                        }
                    }
                }
            }
            // clear the interrupt and wait for another result
            int_right_result = 0;
            wait_ms( MEASUREMENTTIMING );
            status = board->sensor_right->VL53LX_ClearInterruptAndStartMeasurement();
        }
    }
    printf("terminated");
}

#if (MBED_VERSION  > 60300)
extern "C" void wait_ms(int ms)
{
    thread_sleep_for(ms);
}
#endif
  