/*
 * This VL53L1X Expansion board sample application performs range measurements
 * with interrupts enabled to generate a hardware interrupt each time a new
 * measurement is ready to be read.
 *
 * Measured ranges are output on the Serial Port, running at 115200 baud.
 *
 * On STM32-Nucleo boards :
 *     The User Blue button stops the current measurement and entire program,
 *     releasing all resources.
 *
 *     The Black Reset button is used to restart the program.
 *
 *  *** NOTE : By default hardlinks U10, U11, U15 & U18, on the underside of
 *            the X-NUCELO-53L1A1 expansion board are not made/OFF.
 *            These links must be made to allow interrupts from the Satellite boards
 *            to be received.
 *            U11 and U18 must be made/ON to allow interrupts to be received from the
 *            INT_L & INT_R positions; or
 *            U10 and U15 must be made/ON to allow interrupts to be received from the
 *            Alternate INT_L & INT_R positions.
 *            The X_NUCLEO_53L1A1 firmware library defaults to use the INT_L/INT_R
 *            positions.
 *            INT_L is available on expansion board Arduino Connector CN5, pin 1 as D9.
 *            Alternate INT_L is on CN5 Connector pin 2 as D8.
 *            INT_R is available on expansion board Arduino Connector CN9, pin 3 as D4.
 *            Alternate INT_R is on CN9 Connector pin 5 as D2.
 *            The pinouts are shown here : https://developer.mbed.org/components/X-NUCLEO-53L1A1/
 */

#include <stdio.h>

#include "mbed.h"
#include "XNucleo53L1A1.h"
#include "VL53L1X_I2C.h"

#define VL53L1_I2C_SDA   D14
#define VL53L1_I2C_SCL   D15

#if TARGET_STM  // we are cross compiling for an STM32-Nucleo
    InterruptIn stop_button(USER_BUTTON);
#endif
#if TARGET_Freescale // we are cross-compiling for NXP FRDM boards.
    InterruptIn stop_button(SW2);
#endif


static XNucleo53L1A1 *board=NULL;

/* flags that handle interrupt request for sensor and user blue button*/
volatile bool int_sensor = false;
volatile bool int_stop = false;

/* ISR callback function of the centre sensor */
void sensor_irq(void)
{
    int_sensor = true;
    board->sensor_centre->disable_interrupt_measure_detection_irq();
}

/* Start the sensor ranging */
int init_sensor()
{
    int status = 0;
    /* start the measure on the center sensor */
    if (NULL != board->sensor_centre) {
        status = board->sensor_centre->stop_measurement();
        if (status != 0) {
                return status;
        }

        status = board->sensor_centre->start_measurement(&sensor_irq);
        if (status != 0) {
            return status;
        }
    }
    return status;
}

/* ISR callback function of the user blue button to switch measuring sensor. */
void measuring_stop_irq(void)
{
    int_stop = true;
}

/*
 * Main ranging function
 */
int range_measure(VL53L1X_DevI2C *device_i2c)
{
    int status = 0;
    uint16_t distance = 0;

    /* creates the 53L1A1 expansion board singleton obj */
    board = XNucleo53L1A1::instance(device_i2c, A2, D9, D2);

    /* init the 53L1A1 expansion board with default values */
    status = board->init_board();
    if (status != 0) {
        printf("Failed to init board!\r\n");
        return status;
    }

    /* init an array with chars to id the sensors */
    status = init_sensor();
    if (status != 0) {
        printf("Failed to init sensors!\r\n");
        return status;
    }

    printf("Entering loop mode\r\n");
    /* Main ranging interrupt loop */
    while (true) {
        if (int_sensor) {
            int_sensor = false;
            status = board->sensor_centre->handle_irq(&distance);
            printf("distance: %d\r\n", distance);
        }

        if (int_stop) {
            printf("\r\nEnding loop mode \r\n");
            break;
        }
    }

    return status;

}

/*=================================== Main ==================================
=============================================================================*/
int main()
{
    stop_button.rise(&measuring_stop_irq);

    VL53L1X_DevI2C *device_i2c = new VL53L1X_DevI2C(VL53L1_I2C_SDA, VL53L1_I2C_SCL);
    range_measure(device_i2c);  // start continuous measures
}
