/*
 * This VL53L1X Expansion board sample application performs range measurements
 * with interrupts enabled to generate a hardware interrupt each time a new
 * measurement is ready to be read.
 *
 * Measured ranges are output on the Serial Port, running at 115200 baud.
 *
 * The application supports the centre, on-board, sensor and up to two satellite boards.
 *
 * On STM32-Nucleo boards :
 *     The User Blue button stops the current measurement and entire program,
 *     releasing all resources.
 *
 *     The Black Reset button is used to restart the program.
 *
 * *** NOTE : By default hardlinks U10, U11, U15 & U18, on the underside of
 *            the X-NUCELO-53L0A1 expansion board are not made/OFF.
 *            These links must be made to allow interrupts from the Satellite boards
 *            to be received.
 *            U11 and U18 must be made/ON to allow interrupts to be received from the
 *            INT_L & INT_R positions; or
 *            U10 and U15 must be made/ON to allow interrupts to be received from the
 *            Alternate INT_L & INT_R positions.
 *            The X_NUCLEO_53L1A1 firmware library defaults to use the INT_L/INT_R
 *            positions.
 *            INT_L is available on expansion board Arduino Connector CN5, pin 1 as D9.
 *            Alternate INT_L is on CN5 Connector pin 2 as D8.
 *            INT_R is available on expansion board Arduino Connector CN9, pin 3 as D4.
 *            Alternate INT_R is on CN9 Connector pin 5 as D2.
 *            The pinouts are shown here : https://developer.mbed.org/components/X-NUCLEO-53L1A1/
 */
 
#include <stdio.h>

#include "mbed.h"
#include "XNucleo53L1A1.h"
#include "VL53L1X_I2C.h"

#define VL53L1_I2C_SDA   D14 
#define VL53L1_I2C_SCL   D15 

#if TARGET_STM  // we are cross compiling for an STM32-Nucleo
    InterruptIn stop_button(USER_BUTTON);
#endif
#if TARGET_Freescale // we are cross-compiling for NXP FRDM boards.
    InterruptIn stop_button(SW2);
#endif

/* Installed sensors count */
int sensorCnt = 0;

/* installed sensors prefixes */
char installedSensors[3];

static XNucleo53L1A1 *board=NULL;

/* interrupt requests */
volatile bool centerSensor = false;
volatile bool leftSensor = false;
volatile bool rightSensor = false;
volatile bool int_measuring_stop = false;
/* Current sensor number*/
volatile int currentSensor = 0;

/* current displayed sensor change IRQ */
volatile bool switchChanged = false;

/* ISR callback function of the centre sensor */
void sensor_centre_irq(void)
{
    centerSensor = true;
    board->sensor_centre->disable_interrupt_measure_detection_irq();
}

/* ISR callback function of the left sensor */
void sensor_left_irq(void)
{
    leftSensor = true;
    board->sensor_left->disable_interrupt_measure_detection_irq();
}

/* ISR callback function of the right sensor */
void sensor_right_irq(void)
{
    rightSensor = true;
    board->sensor_right->disable_interrupt_measure_detection_irq();
}

/* ISR callback function of the user blue button to switch measuring sensor. */
void switch_measuring_sensor_irq(void)
{
    stop_button.disable_irq();
    switchChanged = true;
}

/*
 * This function calls the interrupt handler for each sensor
 * and outputs the range
 */
inline void measure_sensors()
{
    bool current = false;
    uint16_t distance = 0;

    /* Handle the interrupt and output the range from the centre sensor */
    if (centerSensor) {
        centerSensor = false;
        board->sensor_centre->handle_irq(&distance);
        current = (currentSensor == 0);
        if (current) {
            printf("Centre: %d\r\n", distance);
        }
    }

    /* Handle the interrupt and output the range from the left sensor */
    if (leftSensor) {
        leftSensor = false;
        board->sensor_left->handle_irq(&distance);
        current = (installedSensors[currentSensor] == 'L');
        if (current) {
            printf("Left: %d\r\n", distance);
        }
    }

    /* Handle the interrupt and output the range from the right sensor */
    if (rightSensor) {
        rightSensor = false;
        board->sensor_right->handle_irq(&distance);
        current = (installedSensors[currentSensor] == 'R');
        if (current) {
            printf("Right: %d\r\n", distance);
        }
    }
}

/*
 * Add to an array a character that represents the sensor and start ranging
 */
int init_sensors_array()
{
    int status = 0;
    sensorCnt = 0;
    /* start the measure on the center sensor */
    if (NULL != board->sensor_centre) {
        installedSensors[sensorCnt] = 'C';
        status = board->sensor_centre->stop_measurement();
        if (status != 0) {
            return status;
        }
        status = board->sensor_centre->start_measurement(&sensor_centre_irq);
        if (status != 0) {
            return status;
        }
        ++sensorCnt;
    }
    /* start the measure on the left sensor */
    if (NULL != board->sensor_left) {
        installedSensors[sensorCnt] = 'L';
        status = board->sensor_left->stop_measurement();
        if (status != 0) {
            return status;
        }
        status = board->sensor_left->start_measurement(&sensor_left_irq);
        if (status != 0) {
            return status;
        }
        ++sensorCnt;
    }
    /* start the measure on the right sensor */
    if (NULL != board->sensor_right) {
        installedSensors[sensorCnt] = 'R';
        status = board->sensor_right->stop_measurement();
        if (status != 0) {
            return status;
        }
        status = board->sensor_right->start_measurement(&sensor_right_irq);
        if (status != 0) {
            return status;
        }
        ++sensorCnt;
    }
    currentSensor = 0;
    return status;
}

/*
 * Main ranging function
 */
int range_measure(VL53L1X_DevI2C *device_i2c)
{
    int status = 0;

    /* creates the 53L1A1 expansion board singleton obj */
    board = XNucleo53L1A1::instance(device_i2c, A2, D9, D2);

    /* init the 53L1A1 expansion board with default values */
    status = board->init_board();
    if (status != 0) {
        printf("Failed to init board!\r\n");
        return status;
    }

    /* init an array with chars to id the sensors */
    status = init_sensors_array();
    if (status != 0) {
        printf("Failed to init sensors!\r\n");
        return status;
    }

    printf("Entering loop mode\r\n");

    /* Main ranging interrupt loop */
    while (true) {
        measure_sensors();
        if (switchChanged) {
            ++currentSensor;
            if (currentSensor == sensorCnt)
                currentSensor = 0;
            printf("Sensor changed to %c\r\n", installedSensors[currentSensor]);
            switchChanged = false;
            stop_button.enable_irq();
        }
    }
    delete board;
    return status;
}

/*=================================== Main ==================================
=============================================================================*/
int main()
{
    stop_button.rise(&switch_measuring_sensor_irq);
    stop_button.enable_irq();
    
    VL53L1X_DevI2C *dev_I2C = new VL53L1X_DevI2C(VL53L1_I2C_SDA, VL53L1_I2C_SCL);
    range_measure(dev_I2C);  // start continuous measures
    
    return 0;
}

