/*
 *  This VL53L1X Expansion board test application performs range measurements
 *  using the onboard embedded centre sensor, in singleshot, polling mode.
 *  Measured ranges are ouput on the Serial Port, running at 115200 baud.
 *
 *
 * This is designed to work with MBed V2 , MBed V5 and MBed V6.
 *
 *
 *  The Reset button can be used to restart the program.
 */
 
#include <stdio.h>

#include "mbed.h"
#include "XNucleo53L1A2.h"
#include "ToF_I2C.h"
#include <time.h>



// define the i2c comms pins
#define I2C_SDA   D14 
#define I2C_SCL   D15 

#define NUM_SENSORS 3

// define interrupt pins
PinName CentreIntPin = A2;
// the satellite pins depend on solder blobs on the back of the shield.
// they may not exist or may be one of two sets.
// the centre pin always exists
PinName LeftIntPin = D9;
PinName RightIntPin = D4;
// alternate set
//PinName LeftIntPin = D8;
//PinName RightIntPin = D2;


static XNucleo53L1A2 *board=NULL;

// MBed V6.4 has renamed wait_ms and UnbufferedSerial replaces Serial
#if (MBED_VERSION  > 60300) 
UnbufferedSerial  pc(USBTX, USBRX); 
extern "C" void wait_ms(int ms);
#else
Serial pc(SERIAL_TX, SERIAL_RX); 
#endif



VL53L1_Dev_t                   devCentre;
VL53L1_Dev_t                   devLeft;
VL53L1_Dev_t                   devRight;
VL53L1_DEV                     Dev = &devCentre;
 
/*=================================== Main ==================================
=============================================================================*/
int main()
{   
    int status;
    VL53L1CB * Sensor;
    uint16_t wordData;
    uint8_t ToFSensor = 1; // 0=Left, 1=Center(default), 2=Right
    static VL53L1_RangingMeasurementData_t RangingData;
    uint32_t  polling_time_ms;
    uint32_t  start_time_ms;
    uint32_t  current_time_ms;
    
    pc.baud(115200);  // baud rate is important as printf statements take a lot of time

    printf("Polling Lite-Ranging mbed = %d \r\n",MBED_VERSION);
// create i2c interface
    ToF_DevI2C *dev_I2C = new ToF_DevI2C(I2C_SDA, I2C_SCL);

    dev_I2C->frequency(400000); //also needs doing in spi_interface.c
    
    /* creates the 53L1A2 expansion board singleton obj */
    board = XNucleo53L1A2::instance(dev_I2C, CentreIntPin, LeftIntPin, RightIntPin);
    printf("board created!\r\n");

    /* init the 53L1A1 expansion board with default values */
    status = board->init_board();
    if (status) {
        printf("Failed to init board!\r\n");
        return 0;
    }
       
        
    printf("board initiated! - %d\r\n", status);
    
   // for each sensor, if it exists, initialise and configure                                             
        for (ToFSensor=0;ToFSensor< NUM_SENSORS;ToFSensor++){
        switch(ToFSensor){
            case 0:
                if (board->sensor_centre== NULL ) continue;
                Dev=&devCentre;
                Sensor=board->sensor_centre;
                Dev->i2c_slave_address = NEW_SENSOR_CENTRE_ADDRESS;
                printf("configuring centre channel \n");
                break;
            case 1:
                if (board->sensor_left== NULL ) continue;
                Dev=&devLeft; 
                Sensor=board->sensor_left;
                Dev->i2c_slave_address = NEW_SENSOR_LEFT_ADDRESS;
                printf("configuring left channel \n");
                break;
            case 2:
                if (board->sensor_right== NULL ) continue;
                Dev=&devRight;  
                Sensor=board->sensor_right;
                Dev->i2c_slave_address = NEW_SENSOR_RIGHT_ADDRESS;
                printf("configuring right channel \n");
                break;      
        }
            
            Dev->comms_speed_khz = 400;
            Dev->comms_type = 1;
    
            Sensor->VL53L1CB_RdWord(Dev, 0x01, &wordData);
            printf("VL53L1X: %02X   %d\n\r", wordData,Dev->i2c_slave_address);
    /* Device Initialization and setting */
    
            status = Sensor->VL53L1CB_DataInit();
            status = Sensor->VL53L1CB_StaticInit();
            status = Sensor->VL53L1CB_SetPresetMode( VL53L1_PRESETMODE_LITE_RANGING);
            status = Sensor->VL53L1CB_SetDistanceMode( VL53L1_DISTANCEMODE_LONG);
            status = Sensor->VL53L1CB_SetMeasurementTimingBudgetMicroSeconds( 50000);
            status = Sensor->VL53L1CB_SetInterMeasurementPeriodMilliSeconds( 100);
        }
        
        if (board->sensor_centre!= NULL ) 
        {
            Dev=&devCentre;
            Sensor=board->sensor_centre;
            Dev->i2c_slave_address = NEW_SENSOR_CENTRE_ADDRESS;
        }
        
        if (board->sensor_left!= NULL ) 
        {
            Dev=&devLeft;
            Sensor=board->sensor_left;
            Dev->i2c_slave_address = NEW_SENSOR_LEFT_ADDRESS;
        }
        
        
        if (board->sensor_right!= NULL ) 
        {
            Dev=&devRight;
            Sensor=board->sensor_right;
            Dev->i2c_slave_address = NEW_SENSOR_RIGHT_ADDRESS;
        }
        
     while(1) {     
        for (ToFSensor=0;ToFSensor< NUM_SENSORS;ToFSensor++){
            switch(ToFSensor){
                case 0:
                    if (board->sensor_centre== NULL ) continue;
                    Dev=&devCentre;
                    Sensor=board->sensor_centre;
                    Dev->i2c_slave_address = NEW_SENSOR_CENTRE_ADDRESS;
                    break;
                case 1:
                    if (board->sensor_left== NULL ) continue;
                    Dev=&devLeft; 
                    Sensor=board->sensor_left;
                    Dev->i2c_slave_address = NEW_SENSOR_LEFT_ADDRESS;
                    break;
                case 2:
                    if (board->sensor_right== NULL ) continue;
                    Dev=&devRight;
                    Sensor=board->sensor_right;
                    Dev->i2c_slave_address = NEW_SENSOR_RIGHT_ADDRESS;
                    break;   
                default:   
                    printf("default \n");
                    break;
            }  // switch
        
            start_time_ms = us_ticker_read() / 1000;
            status = Sensor->VL53L1CB_StartMeasurement();  

            status = Sensor->VL53L1CB_WaitMeasurementDataReady();

            if(!status)
            {
                 status = Sensor->VL53L1CB_GetRangingMeasurementData( &RangingData);
                                                 current_time_ms = us_ticker_read() / 1000;

                 if((status==0) &&  (RangingData.RangeStatus != 10)) {
                     printf("data %d,%d,%d,%.2f,%.2f\n", ToFSensor,RangingData.RangeStatus,RangingData.RangeMilliMeter,
                           (RangingData.SignalRateRtnMegaCps/65536.0),RangingData.AmbientRateRtnMegaCps/65336.0);

                 }
            } // if status
            else
            {
                printf("VL53L1_WaitMeasurementDataReady failed %d \n",status);
            }

            status = Sensor->VL53L1CB_ClearInterruptAndStartMeasurement();                                     
            status = Sensor->VL53L1CB_WaitMeasurementDataReady();
            if(!status)
            {
                status = Sensor->VL53L1CB_GetRangingMeasurementData( &RangingData);

                polling_time_ms = current_time_ms - start_time_ms;
                if((status==0) &&  (RangingData.RangeStatus != 10)) {
                    printf("data %d,%d,%d,%.2f,%.2f  %d\n", ToFSensor,RangingData.RangeStatus,RangingData.RangeMilliMeter,
                          (RangingData.SignalRateRtnMegaCps/65536.0),RangingData.AmbientRateRtnMegaCps/65336.0,polling_time_ms);
                }
            }

            status = Sensor->VL53L1CB_ClearInterruptAndStartMeasurement();
        

            status = Sensor->VL53L1CB_StopMeasurement();  
        } // endwhile(1)
        
    }
    

}


#if (MBED_VERSION  > 60300) 
void wait_ms(int ms)
 {
    thread_sleep_for(ms);
 }
#endif
  
