/*
 *  This VL53L1X Expansion board test application performs range measurements
 *  using the onboard embedded centre sensor and two satelites, in Lite-ranging, interrupt mode.
 *  Measured ranges are ouput on the Serial Port, running at 115200 baud.
 *
 *
 * This is designed to work with MBed V2 , MBed V5 and MBed V6.
 *
 *
 *  The Reset button can be used to restart the program.
 */
 // changed pins
#include <stdio.h>

#include "mbed.h"
#include "XNucleo53L1A2.h"
#include "ToF_I2C.h"
#include <time.h>


// define the i2c comms pins
#define I2C_SDA   D14 
#define I2C_SCL   D15 

#define NUM_SENSORS 3

PinName CentreIntPin = A2;
// the satellite pins depend on solder blobs on the back of the shield.
// they may not exist or may be one of two sets.
// the centre pin always exists
PinName LeftIntPin = D9;
PinName RightIntPin = D4;
// alternate set
//PinName LeftIntPin = D8;
//PinName RightIntPin = D2;



static XNucleo53L1A2 *board=NULL;

// MBed V6.4 has renamed wait_ms and UnbufferedSerial replaces Serial
#if (MBED_VERSION  > 60300) 
UnbufferedSerial  pc(USBTX, USBRX); 
extern "C" void wait_ms(int ms);
#else
Serial pc(SERIAL_TX, SERIAL_RX); 
#endif

static int int_centre_result = 0;
static int int_left_result = 0;
static int int_right_result = 0;


class WaitForMeasurement {
public:


// this class services the interrupts from the ToF sensors.
// There is a limited amount you can do in an interrupt routine; printfs,mutexes break them among other things.
// We keep things simple by only raising a flag so all the real work is done outside the interrupt.
// This is designed around MBED V2 which doesn't have the RTOS features that would make this work nicely e.g. semaphores/queues.
WaitForMeasurement(): _interrupt(A1)
{
}


    // constructor - Sensor is not used and can be removed
    WaitForMeasurement(PinName pin,VL53L1_DEV Dev) : _interrupt(pin)          // create the InterruptIn on the pin specified to Counter
    {
         Devlocal = Dev;
        _interrupt.rise(callback(this, &WaitForMeasurement::got_interrupt)); // attach increment function of this counter instance
        
    }
    
    void process_right_interrupt()
    {
          printf("processing right interrupt\n");
    }

  // function is called every time an interupt is seen. A flag is raised which allows the main routine to service the interupt.
    void got_interrupt()
    {
    
        _count++;

        if (Devlocal->i2c_slave_address == NEW_SENSOR_CENTRE_ADDRESS)
            int_centre_result = 1;  //flag to main that interrupt happened
        if (Devlocal->i2c_slave_address == NEW_SENSOR_LEFT_ADDRESS)
            int_left_result = 1;   //flag to main that interrupt happened
        if (Devlocal->i2c_slave_address == NEW_SENSOR_RIGHT_ADDRESS)
        {
            int_right_result = 1;  //flag to main that interrupt happened
        }
    }
    
    //destructor
    ~WaitForMeasurement()
    {
        printf("destruction \n");
    }

private:
    InterruptIn _interrupt;
    volatile int _count;
    VL53L1_DEV Devlocal;
    int status;
    
};



VL53L1_Dev_t                   devCentre;
VL53L1_Dev_t                   devLeft;
VL53L1_Dev_t                   devRight;
VL53L1_DEV                     Dev = &devCentre;

 
/*=================================== Main ==================================
=============================================================================*/
int main()
{   
    int status;
    VL53L1CB * Sensor;
    uint16_t wordData;
    uint8_t ToFSensor = 1;   // loop count
  
    
    WaitForMeasurement* int2;
    WaitForMeasurement* int1;
    WaitForMeasurement* int3;

    pc.baud(115200);  // baud rate is important as printf statements take a lot of time

    printf("Interrupt Lite-ranging mbed = %d \r\n",MBED_VERSION);

// create i2c interface
    ToF_DevI2C *dev_I2C = new ToF_DevI2C(I2C_SDA, I2C_SCL);
    
    dev_I2C->frequency(400000); //also needs doing in spi_interface.c
    
    /* creates the 53L1A2 expansion board singleton obj */
    board = XNucleo53L1A2::instance(dev_I2C, CentreIntPin, LeftIntPin, RightIntPin);
    printf("board created!\r\n");
                                                                                     
    /* init the 53L1A1 expansion board with default values */
    status = board->init_board();
    if (status) { 
        return 0;
    }
       
   
    printf("board initiated! - %d\r\n", status);

    
    // create sensor controller classes for each vl53l1 and configure each vl53l1                                                  
        for (ToFSensor=0;ToFSensor<NUM_SENSORS ;ToFSensor++)
        {
            switch(ToFSensor){
                case 1:
                    if (board->sensor_centre== NULL ) continue;  // don't create if sensor not detected
                    Dev=&devCentre;
                    Sensor=board->sensor_centre;
                    Dev->i2c_slave_address = NEW_SENSOR_CENTRE_ADDRESS;
                    printf("configuring centre channel \n");
                    break;
                case 0:
                    if (board->sensor_left== NULL ) continue;
                    Dev=&devLeft; 
                    Sensor=board->sensor_left;
                    Dev->i2c_slave_address = NEW_SENSOR_LEFT_ADDRESS;
                    printf("configuring left channel \n");
                    break;
                case 2:
                    if (board->sensor_right== NULL ) continue;
                    Dev=&devRight;  
                    Sensor=board->sensor_right;
                    Dev->i2c_slave_address = NEW_SENSOR_RIGHT_ADDRESS;
                    printf("configuring right channel \n");
                    break;      
                default:
                   printf(" error in switch, invalid ToF sensor \n");
            } // end of switch
        
// configure the sensors
            Dev->comms_speed_khz = 400;
            Dev->comms_type = 1;
    
    /* Device Initialization and setting */  
            status = Sensor->VL53L1CB_DataInit();
            status = Sensor->VL53L1CB_StaticInit();
            status = Sensor->VL53L1CB_SetPresetMode( VL53L1_PRESETMODE_LITE_RANGING);
            status = Sensor->VL53L1CB_SetDistanceMode( VL53L1_DISTANCEMODE_LONG);
   //         status = Sensor->vl53L1_SetMeasurementTimingBudgetMicroSeconds( 50000);
    //        status = Sensor->vl53L1_SetInterMeasurementPeriodMilliSeconds( 100);
        }  // end of sensor loop
        printf("starting sensors \n");
        
        // initialise sensor interrupts
        if (board->sensor_centre!= NULL )
        {
            Sensor=board->sensor_centre;
            devCentre.i2c_slave_address = NEW_SENSOR_CENTRE_ADDRESS;
            int1 =  new WaitForMeasurement(CentreIntPin,&devCentre);  // initialise sensor interrupts
            status = Sensor->VL53L1CB_StartMeasurement();
            printf("started interrupt centre %d\n",status);
        }
        

        if (board->sensor_left!= NULL )
        {
            Sensor=board->sensor_left;
            devLeft.i2c_slave_address = NEW_SENSOR_LEFT_ADDRESS;
            int2 = new WaitForMeasurement(LeftIntPin,&devLeft);
            status = Sensor->VL53L1CB_StartMeasurement();
            printf("started interrupt left %d\n",status);
        }

        if (board->sensor_right!= NULL )
        {
            Sensor=board->sensor_right;
            devRight.i2c_slave_address = NEW_SENSOR_RIGHT_ADDRESS;
            int3 = new WaitForMeasurement(RightIntPin,&devRight);
            status = board->sensor_right->VL53L1CB_StartMeasurement();
            printf("started interrupt sensor_right  %d\n",status);
        }
        

    
       // loop waiting for interrupts to happen. This is signaled by   int_centre_result,int_left_result or int_right_result
       // being non zero. The are set back to zero when processing is completed
        while (1)
        {   
            static VL53L1_RangingMeasurementData_t RangingData;  
                            
            wait_ms( 1 * 30);
            if (int_centre_result != 0)
            {
                // read result
                status = board->sensor_centre->VL53L1CB_GetRangingMeasurementData( &RangingData);
                // if valid result print it
                if (RangingData.RangeStatus != 255)
                    printf("c %d,%d \n",RangingData.RangeStatus,RangingData.RangeMilliMeter);
                // clear interrupt flag
                int_centre_result = 0;
                // clear theinterrupt and wait for another result
                status = board->sensor_centre->VL53L1CB_ClearInterruptAndStartMeasurement();
            }


            if (int_left_result != 0)
            {
                // read result
                status = board->sensor_left->VL53L1CB_GetRangingMeasurementData( &RangingData);
                // if valid result print it 
                if (RangingData.RangeStatus != 255)
                    printf("l %d,%d\n",RangingData.RangeStatus,RangingData.RangeMilliMeter);
                // clear interrupt flag                
                int_left_result = 0;
                // clear theinterrupt and wait for another result
                status = board->sensor_left->VL53L1CB_ClearInterruptAndStartMeasurement();
            }
            
            
            if (int_right_result != 0)
            {   
                // read result
                status = board->sensor_right->VL53L1CB_GetRangingMeasurementData( &RangingData);
                // if valid result print it
                if (RangingData.RangeStatus != 255)
                    printf("r %d,%d\n",RangingData.RangeStatus,RangingData.RangeMilliMeter);
                // clear interrupt flag
                int_right_result = 0;
                // clear theinterrupt and wait for another result
                status = board->sensor_right->VL53L1CB_ClearInterruptAndStartMeasurement();
            }
            
        }
        printf("terminated");
    }
  
  
  
          
#if (MBED_VERSION  > 60300) 
void wait_ms(int ms)
 {
    thread_sleep_for(ms);
 }
#endif
  