/*
 * This VL53L1CB Expansion board test application performs range measurements
 * using the onboard embedded sensor, in polling mode.
 * Measured ranges are output on the Serial Port, running at 115200 baud.
 *
 * This is designed to work with MBed v2.x, & MBedOS v5.x / v6.x.
 *
 * The Reset button can be used to restart the program.
 *
 * *** NOTE :
 *     Default Mbed build system settings disable printf() floating-point support.
 *     Offline builds can enable this, again.
 *     https://github.com/ARMmbed/mbed-os/blob/master/platform/source/minimal-printf/README.md
 *     .\mbed-os\platform\mbed_lib.json
 *
 * *** NOTE : By default hardlinks U10, U11, U15 & U18, on the underside of
 *            the X-NUCELO-53L1A1 expansion board are not made/OFF.
 *            These links must be made to allow interrupts from the Satellite boards
 *            to be received.
 *            U11 and U18 must be made/ON to allow interrupts to be received from the
 *            INT_L & INT_R positions; or
 *            U10 and U15 must be made/ON to allow interrupts to be received from the
 *            Alternate INT_L & INT_R positions.
 *            The X_NUCLEO_53L1A2 library defaults to use the INT_L/INT_R positions.
 *            INT_L is available on expansion board Arduino Connector CN5, pin 1 as D8.
 *            Alternate INT_L is on CN5 Connector pin 2 as D9.
 *            INT_R is available on expansion board Arduino Connector CN9, pin 3 as D2.
 *            Alternate INT_R is on CN9 Connector pin 5 as D4.
 *            The pinouts are shown here : https://developer.mbed.org/components/X-NUCLEO-53L1A2/
 *
 */

#include <stdio.h>
#include <time.h>

#include "mbed.h"

#include "XNucleo53L1A2.h"
#include "ToF_I2C.h"

// i2c comms port pins
#define I2C_SDA   D14
#define I2C_SCL   D15


#define NUM_SENSORS 3

// define interrupt pins
PinName CentreIntPin = A2;
// the satellite pins depend on solder blobs on the back of the shield.
// they may not exist or may be one of two sets.
// the centre pin always exists
//PinName LeftIntPin = D8;
PinName RightIntPin = D2;
// alternate set
PinName LeftIntPin = D9;
//PinName RightIntPin = D4;

static XNucleo53L1A2 *board=NULL;

#if (MBED_VERSION  > 60300)
UnbufferedSerial  pc(USBTX, USBRX);
#else
Serial pc(SERIAL_TX, SERIAL_RX);
#endif

void print_results(int devNumber, VL53L1_MultiRangingData_t *pMultiRangingData );


VL53L1_Dev_t devCentre;
VL53L1_DEV Dev = &devCentre;

VL53L1CB *Sensor;



/* flags that handle interrupt request for sensor and user blue button*/
volatile bool int_sensor = false;
volatile bool int_stop = false;

/* ISR callback function of the centre sensor */
void sensor_irq(void)
{
    int_sensor = true;
    board->sensor_centre->disable_interrupt_measure_detection_irq();
}

/* Start the sensor ranging */
int configure_sensor()
{
    int status = 0;
    VL53L1_DeviceInfo_t device_info;
    VL53L1_RoiConfig_t RoiConfig;

    Dev = &devCentre;
    Sensor = board->sensor_centre;

    if (Sensor == NULL)
        return -1;

    printf("configuring centre channel \n");

    status = Sensor->VL53L1CB_GetDeviceInfo(&device_info);
    if (status != 0) {
        return status;
    }
    printf("device name %s \n",device_info.Name);
    printf("device type %s \n",device_info.Type);
    printf("device productID %s \n",device_info.ProductId);
    printf("device productType %x \n",device_info.ProductType);

    RoiConfig.NumberOfRoi = 3;

    RoiConfig.UserRois[0].TopLeftX = 3;
    RoiConfig.UserRois[0].TopLeftY = 10;
    RoiConfig.UserRois[0].BotRightX = 10;
    RoiConfig.UserRois[0].BotRightY = 3;

    RoiConfig.UserRois[1].TopLeftX = 5;
    RoiConfig.UserRois[1].TopLeftY = 12;
    RoiConfig.UserRois[1].BotRightX = 12;
    RoiConfig.UserRois[1].BotRightY = 5;

    RoiConfig.UserRois[2].TopLeftX = 6;
    RoiConfig.UserRois[2].TopLeftY = 13;
    RoiConfig.UserRois[2].BotRightX = 13;
    RoiConfig.UserRois[2].BotRightY = 6;

    status = Sensor->VL53L1CB_SetPresetMode(VL53L1_PRESETMODE_MULTIZONES_SCANNING);
    if (status != 0) {
        return status;
    }

    status = Sensor->VL53L1CB_SetDistanceMode(VL53L1_DISTANCEMODE_LONG);
    if (status != 0) {
        return status;
    }

    status = Sensor->VL53L1CB_SetROI(&RoiConfig);
    if (status != 0) {
        return status;
    }

    status = Sensor->VL53L1CB_SetMeasurementTimingBudgetMicroSeconds(60000);
    if (status != 0) {
        return status;
    }

    return status;
}

/* ISR callback function of the user blue button to switch measuring sensor. */
void measuring_stop_irq(void)
{
    int_stop = true;
}

/*=================================== Main ==================================
=============================================================================*/
int main()
{
    int status;

    pc.baud(115200);  // baud rate is important as printf statements take a lot of time

    printf("mbed version : %d \r\n",MBED_VERSION);

// create i2c interface
    ToF_DevI2C *dev_I2C = new ToF_DevI2C(I2C_SDA, I2C_SCL);
    /* creates the 53L1A2 expansion board singleton obj */
    board = XNucleo53L1A2::instance(dev_I2C, CentreIntPin, LeftIntPin, RightIntPin);

    printf("board created!\r\n");

    /* init the 53L1A1 expansion board with default values */
    status = board->init_board();
    if (status) {
        printf("Failed to init board!\r\n");
        return status;
    }

    printf("board initiated! - %d\r\n", status);

    /* init an array with chars to id the sensors */
    status = configure_sensor();
    if (status != 0) {
        printf("Failed to init sensors!\r\n");
        return status;
    }

    // start measurements
    status = board->sensor_centre->VL53L1CB_StartMeasurement();
    if (status != 0) {
        return status;
    }

    printf("loop forever\n");

    VL53L1_MultiRangingData_t MultiRangingData;
    VL53L1_MultiRangingData_t *pMultiRangingData = &MultiRangingData;

    while (true) {

        status = board->sensor_centre->VL53L1CB_WaitMeasurementDataReady();
        status = board->sensor_centre->VL53L1CB_GetMultiRangingData( pMultiRangingData);

        print_results( devCentre.i2c_slave_address, pMultiRangingData );

        status = board->sensor_centre->VL53L1CB_ClearInterruptAndStartMeasurement();
    }

    printf("Terminating.\n");
}


// print what ever results are required
void print_results( int devNumber, VL53L1_MultiRangingData_t *pMultiRangingData )
{
    int no_of_object_found = pMultiRangingData->NumberOfObjectsFound;
    int signal_kcps = 0;
    int ambient_kcps = 0;

    int RoiNumber = pMultiRangingData->RoiNumber;

    if (no_of_object_found <= 1)
        no_of_object_found = 1;
    printf("i2cAddr=%d\tRoiNumber=%d",  devNumber, RoiNumber);
    for(int j=0; j<no_of_object_found; j++) {
        signal_kcps = 1000*(pMultiRangingData->RangeData[j].SignalRateRtnMegaCps) / 65536;
        ambient_kcps = 1000*(pMultiRangingData->RangeData[j].AmbientRateRtnMegaCps) / 65536;
        if (j > 0)
            printf("\t\t\t");
        printf("\trange[%d] status=%d, \t D=%5dmm, \t Signal=%d Kcps, \t Ambient=%d Kcps \n",
               j,
               pMultiRangingData->RangeData[j].RangeStatus,
               pMultiRangingData->RangeData[j].RangeMilliMeter,
               signal_kcps,
               ambient_kcps);
    }
}

