/*
 * This VL53L1CB test application performs range measurements, using a satellite 
 * board plugged directly in to the motherboard.
 * Measured ranges are ouput on the Serial Port, running at 115200 baud.
 *
 * This is designed to work with MBed v2.x, & MBedOS v5.x / v6.x.
 *
 * The satellite board is expected to be connected as follows :
 *
 * I2C_SDA   			D14 / Arduino Connector CN5, pin 9
 * I2C_SCL   			D15 / Arduino Connector CN5, pin 10
 * Sensor xShutdown		D07 / Arduino Connector CN9, pin 8
 * Sensor Interrupt		A02 / Arduino Connector CN8, pin 3
 *
 * The Reset button can be used to restart the program.
 *
 * *** NOTE :
 *     Default Mbed build system settings disable printf() floating-point support.
 *     Offline builds can enable this, again.
 *     https://github.com/ARMmbed/mbed-os/blob/master/platform/source/minimal-printf/README.md
 *     .\mbed-os\platform\mbed_lib.json
 *
 * *** NOTE : By default hardlinks U10, U11, U15 & U18, on the underside of
 *            the X-NUCELO-53L1A1 expansion board are not made/OFF.
 *            These links must be made to allow interrupts from the Satellite boards
 *            to be received.
 *            U11 and U18 must be made/ON to allow interrupts to be received from the
 *            INT_L & INT_R positions; or
 *            U10 and U15 must be made/ON to allow interrupts to be received from the
 *            Alternate INT_L & INT_R positions.
 *            The X_NUCLEO_53L1A2 library defaults to use the INT_L/INT_R positions.
 *            INT_L is available on expansion board Arduino Connector CN5, pin 1 as D8.
 *            Alternate INT_L is on CN5 Connector pin 2 as D9.
 *            INT_R is available on expansion board Arduino Connector CN9, pin 3 as D2.
 *            Alternate INT_R is on CN9 Connector pin 5 as D4.
 *            The pinouts are shown here : https://developer.mbed.org/components/X-NUCLEO-53L1A2/
 *
 */

#include <stdio.h>
#include <time.h>

#include "mbed.h"

#include "VL53L1CB.h"
#include "ToF_I2C.h"

// i2c comms port pins
#define I2C_SDA   D14
#define I2C_SCL   D15


#define NEW_SENSOR_CENTRE_ADDRESS   0x52


// define interrupt pins
PinName CentreIntPin = A2;
// the satellite pins depend on solder blobs on the back of the shield.
// they may not exist or may be one of two sets.
// the centre pin always exists
//PinName LeftIntPin = D8;
PinName RightIntPin = D2;
// alternate set
PinName LeftIntPin = D9;
//PinName RightIntPin = D4;


#if (MBED_VERSION  > 60300)
UnbufferedSerial  pc(USBTX, USBRX);
extern "C" void wait_ms(int ms);
#else
Serial pc(SERIAL_TX, SERIAL_RX);
#endif

void print_results(int devNumber, VL53L1_MultiRangingData_t *pMultiRangingData );


VL53L1CB *Sensor;


/* Start the sensor ranging */
int init_sensor()
{
    int status = 0;

    printf("configuring sensor\n");

    /* Device Initialization and setting */
    status = Sensor->VL53L1CB_DataInit();
    status = Sensor->VL53L1CB_StaticInit();

    status = Sensor->VL53L1CB_SetPresetMode(VL53L1_PRESETMODE_AUTONOMOUS);
    status = Sensor->VL53L1CB_SetDistanceMode(VL53L1_DISTANCEMODE_LONG);

    status = Sensor->VL53L1CB_SetInterMeasurementPeriodMilliSeconds(500);

    status = Sensor->VL53L1CB_SetMeasurementTimingBudgetMicroSeconds(45000);

    status = Sensor->VL53L1CB_SetSequenceStepEnable(VL53L1_SEQUENCESTEP_MM1, 0);
    status = Sensor->VL53L1CB_SetSequenceStepEnable(VL53L1_SEQUENCESTEP_MM2, 0);


    // create interrupt handler and start measurements
	if (Sensor != NULL) {
        status = Sensor->stop_measurement();
        if (status != 0) {
            return status;
        }

        status = Sensor->VL53L1CB_StartMeasurement();
        if (status != 0) {
            return status;
        }
    }
    return status;
}


/*=================================== Main ==================================
=============================================================================*/
int main()
{
    int status;
    DigitalOut xshutdown(D7);


    pc.baud(115200);  // baud rate is important as printf statements take a lot of time

    printf("mbed version : %d \r\n",MBED_VERSION);

// create i2c interface
    ToF_DevI2C *dev_I2C = new ToF_DevI2C(I2C_SDA, I2C_SCL);

    Sensor = new VL53L1CB(dev_I2C, &xshutdown, A2);

    printf("sensor created!\r\n");

    /* init the 53L1A1 expansion board with default values */
    Sensor->VL53L1CB_Off();

    Sensor->VL53L1CB_On();
    status = Sensor->InitSensor(NEW_SENSOR_CENTRE_ADDRESS);
    if (status) {
        delete Sensor;
        Sensor = NULL;
        printf("Sensor not present\n\r");
    } else {
        printf("Sensor present\n\r");
    }

#if (MBED_VERSION  > 60300)
    thread_sleep_for(100);
#else
    wait_ms(100);  // NEEDS A DELAY BETWEEN SENSORS
#endif

    printf("board initialised! - %d\r\n", status);

    /* init an array with chars to id the sensors */
    status = init_sensor();
    if (status != 0) {
        printf("Failed to init sensors!\r\n");
        return status;
    }

    printf("loop forever\n");

    VL53L1_MultiRangingData_t MultiRangingData;
    VL53L1_MultiRangingData_t *pMultiRangingData = NULL;

    while (true) {
        pMultiRangingData = &MultiRangingData;

        status = Sensor->VL53L1CB_WaitMeasurementDataReady();

        status = Sensor->VL53L1CB_GetMultiRangingData( pMultiRangingData);

        print_results(NEW_SENSOR_CENTRE_ADDRESS, pMultiRangingData );

        status = Sensor->VL53L1CB_ClearInterrupt();
    }

    printf("Terminating.\n");
}


// print what ever results are required
void print_results( int devNumber, VL53L1_MultiRangingData_t *pMultiRangingData )
{
    int no_of_object_found = pMultiRangingData->NumberOfObjectsFound;
    int signal_rate = 0;
    int ambient_rate = 0;

    int RoiNumber = pMultiRangingData->RoiNumber;

    if (( no_of_object_found < 10 ) &&  ( no_of_object_found != 0)) {
        for(int j=0; j<no_of_object_found; j++) {
            if ((pMultiRangingData->RangeData[j].RangeStatus == VL53L1_RANGESTATUS_RANGE_VALID) ||
                    (pMultiRangingData->RangeData[j].RangeStatus == VL53L1_RANGESTATUS_RANGE_VALID_NO_WRAP_CHECK_FAIL)) {
                signal_rate = pMultiRangingData->RangeData[j].SignalRateRtnMegaCps / 65535;
                ambient_rate = pMultiRangingData->RangeData[j].AmbientRateRtnMegaCps / 65535;
                printf("\t i2cAddr=%d \t RoiNumber=%d   \t status=%d, \t D=%5dmm, \t Signal=%d Mcps, \t Ambient=%d Mcps \n",
                       devNumber, RoiNumber,
                       pMultiRangingData->RangeData[j].RangeStatus,
                       pMultiRangingData->RangeData[j].RangeMilliMeter,
                       signal_rate,
                       ambient_rate);
/*
// online compiler disables printf() / floating-point support, for code-size reasons.                        
// offline compiler can switch it on.
                printf("\t i2cAddr=%d \t RoiNumber=%d   \t status=%d, \t D=%5dmm, \t Signal=%2.2f Mcps, \t Ambient=%2.2f Mcps \n",
                       devNumber, RoiNumber,
                       pMultiRangingData->RangeData[j].RangeStatus,
                       pMultiRangingData->RangeData[j].RangeMilliMeter,
                       pMultiRangingData->RangeData[j].SignalRateRtnMegaCps / 65535.0,
                       pMultiRangingData->RangeData[j].AmbientRateRtnMegaCps / 65535.0);
*/                       
            }
        }
    } // if (( no_of_object_found < 10 ) &&  ( no_of_object_found != 0))
}


#if (MBED_VERSION  > 60300)
extern "C" void wait_ms(int ms)
{
    thread_sleep_for(ms);
}
#endif


