/*
 * This VL53L1CB Expansion board test application performs range measurements
 * using the onboard embedded (centre) sensor, and Left & Right satellites, 
 * in polling mode.
 * Measured ranges are ouput on the Serial Port, running at 115200 baud.
 *
 * This is designed to work with MBed v2.x, & MBedOS v5.x / v6.x.
 *
 * The Reset button can be used to restart the program.
 *
 * *** NOTE :
 *     Default Mbed build system settings disable printf() floating-point support.
 *     Offline builds can enable this, again.
 *     https://github.com/ARMmbed/mbed-os/blob/master/platform/source/minimal-printf/README.md
 *     .\mbed-os\platform\mbed_lib.json
 *
 * *** NOTE : By default hardlinks U10, U11, U15 & U18, on the underside of
 *            the X-NUCELO-53L1A1 expansion board are not made/OFF.
 *            These links must be made to allow interrupts from the Satellite boards
 *            to be received.
 *            U11 and U18 must be made/ON to allow interrupts to be received from the
 *            INT_L & INT_R positions; or
 *            U10 and U15 must be made/ON to allow interrupts to be received from the
 *            Alternate INT_L & INT_R positions.
 *            The X_NUCLEO_53L1A2 library defaults to use the INT_L/INT_R positions.
 *            INT_L is available on expansion board Arduino Connector CN5, pin 1 as D8.
 *            Alternate INT_L is on CN5 Connector pin 2 as D9.
 *            INT_R is available on expansion board Arduino Connector CN9, pin 3 as D2.
 *            Alternate INT_R is on CN9 Connector pin 5 as D4.
 *            The pinouts are shown here : https://developer.mbed.org/components/X-NUCLEO-53L1A2/
 *
 */

#include <stdio.h>
#include <time.h>

#include "mbed.h"

#include "XNucleo53L1A2.h"
#include "ToF_I2C.h"

// i2c comms port pins
#define I2C_SDA   D14
#define I2C_SCL   D15


#define NUM_SENSORS 3

// define interrupt pins
PinName CentreIntPin = A2;
// the satellite pins depend on solder blobs on the back of the shield.
// they may not exist or may be one of two sets.
// the centre pin always exists
//PinName LeftIntPin = D8;
PinName RightIntPin = D2;
// alternate set
PinName LeftIntPin = D9;
//PinName RightIntPin = D4;

static XNucleo53L1A2 *board=NULL;

#if (MBED_VERSION  > 60300)
UnbufferedSerial  pc(USBTX, USBRX);
extern "C" void wait_ms(int ms);
#else
Serial pc(SERIAL_TX, SERIAL_RX);
#endif

void print_results(int devNumber, VL53L1_MultiRangingData_t *pMultiRangingData );


VL53L1_Dev_t devCentre;
VL53L1_DEV DevC = &devCentre;

VL53L1_Dev_t devLeft;
VL53L1_DEV DevL = &devLeft;

VL53L1_Dev_t devRight;
VL53L1_DEV DevR = &devRight;

VL53L1CB *Sensor;



/* flags that handle interrupt request for sensor and user blue button*/
volatile bool int_sensor = false;
volatile bool int_stop = false;

/* ISR callback function of the centre sensor */
void sensor_irq(void)
{
    int_sensor = true;
    board->sensor_centre->disable_interrupt_measure_detection_irq();
}

/* Start the sensor ranging */
int init_sensor()
{
    int status = 0;

    DevC=&devCentre;
    DevC->comms_speed_khz = 400;
    DevC->comms_type = 1;
    DevC->i2c_slave_address = NEW_SENSOR_CENTRE_ADDRESS;
    devCentre.i2c_slave_address = NEW_SENSOR_CENTRE_ADDRESS;
//    Sensor=board->sensor_centre;

    DevL=&devLeft;
//    Sensor=board->sensor_left;
    // configure the sensors
    DevL->comms_speed_khz = 400;
    DevL->comms_type = 1;
    DevL->i2c_slave_address = NEW_SENSOR_LEFT_ADDRESS;
    devLeft.i2c_slave_address = NEW_SENSOR_LEFT_ADDRESS;

    DevR=&devRight;
    // configure the sensors
    DevR->comms_speed_khz = 400;
    DevR->comms_type = 1;
    DevR->i2c_slave_address = NEW_SENSOR_RIGHT_ADDRESS;
    devRight.i2c_slave_address = NEW_SENSOR_RIGHT_ADDRESS;

    for (int i = 0; i < 3; i++)
    {
        if (i == 0) { Sensor=board->sensor_centre;  }
        if (i == 1) { Sensor=board->sensor_left;  }    
        if (i == 2) { Sensor=board->sensor_right;  }    
        
        if (Sensor != NULL)
        {
            if (i == 0) {  printf("configuring centre channel \n"); }
            if (i == 1) {  printf("configuring left channel \n"); }    
            if (i == 2) {  printf("configuring right channel \n"); }    
            
            /* Device Initialization and setting */
            status = Sensor->VL53L1CB_DataInit();
            status = Sensor->VL53L1CB_StaticInit();

            status = Sensor->VL53L1CB_SetPresetMode(VL53L1_PRESETMODE_AUTONOMOUS);
            status = Sensor->VL53L1CB_SetDistanceMode(VL53L1_DISTANCEMODE_LONG);

            status = Sensor->VL53L1CB_SetInterMeasurementPeriodMilliSeconds(500);

            status = Sensor->VL53L1CB_SetMeasurementTimingBudgetMicroSeconds(45000);

            status = Sensor->VL53L1CB_SetSequenceStepEnable(VL53L1_SEQUENCESTEP_MM1, 0);
            status = Sensor->VL53L1CB_SetSequenceStepEnable(VL53L1_SEQUENCESTEP_MM2, 0);
        }
    }    

    // create interrupt handler and start measurements
    if (board->sensor_centre!= NULL) {
        status = board->sensor_centre->stop_measurement();
        if (status != 0) {
            return status;
        }

        status = board->sensor_centre->VL53L1CB_StartMeasurement();
        if (status != 0) {
            return status;
        }
    }
    
    // create interrupt handler and start measurements
    if (board->sensor_left!= NULL) {
        status = board->sensor_left->stop_measurement();
        if (status != 0) {
            return status;
        }

        status = board->sensor_left->VL53L1CB_StartMeasurement();
        if (status != 0) {
            return status;
        }
    }
    
    // create interrupt handler and start measurements
    if (board->sensor_right!= NULL) {
        status = board->sensor_right->stop_measurement();
        if (status != 0) {
            return status;
        }

        status = board->sensor_right->VL53L1CB_StartMeasurement();
        if (status != 0) {
            return status;
        }
    }
    
    return status;
}

/* ISR callback function of the user blue button to switch measuring sensor. */
void measuring_stop_irq(void)
{
    int_stop = true;
}

/*=================================== Main ==================================
=============================================================================*/
int main()
{
    int status;

    pc.baud(115200);  // baud rate is important as printf statements take a lot of time

    printf("mbed version : %d \r\n",MBED_VERSION);

// create i2c interface
    ToF_DevI2C *dev_I2C = new ToF_DevI2C(I2C_SDA, I2C_SCL);
    /* creates the 53L1A2 expansion board singleton obj */
    board = XNucleo53L1A2::instance(dev_I2C, CentreIntPin, LeftIntPin, RightIntPin);    

    printf("board created!\r\n");

    /* init the 53L1A1 expansion board with default values */
    status = board->init_board();
    if (status) {
        printf("Failed to init board!\r\n");
        return status;
    }

    printf("board initiated! - %d\r\n", status);

    /* init an array with chars to id the sensors */
    status = init_sensor();
    if (status != 0) {
        printf("Failed to init sensors!\r\n");
        return status;
    }

    printf("loop forever\n");

    VL53L1_MultiRangingData_t MultiRangingData;
    VL53L1_MultiRangingData_t *pMultiRangingData = NULL;

    while (true) {
        pMultiRangingData = &MultiRangingData;

        if (board->sensor_centre!= NULL) {
            printf("Range result from Centre sensor\n");
            status = board->sensor_centre->VL53L1CB_WaitMeasurementDataReady();
    
            status = board->sensor_centre->VL53L1CB_GetMultiRangingData( pMultiRangingData);
    
            print_results( devCentre.i2c_slave_address, pMultiRangingData );
    
            status = board->sensor_centre->VL53L1CB_ClearInterrupt();
        }
    
        if (board->sensor_left!= NULL) {
            printf("Range result from Left Satellite\n");
            status = board->sensor_left->VL53L1CB_WaitMeasurementDataReady();
    
            status = board->sensor_left->VL53L1CB_GetMultiRangingData( pMultiRangingData);
    
            print_results( devLeft.i2c_slave_address, pMultiRangingData );
    
            status = board->sensor_left->VL53L1CB_ClearInterrupt();
        }
    
        // create interrupt handler and start measurements
        if (board->sensor_right!= NULL) {
            printf("Range result from Right Satellite\n");
            status = board->sensor_right->VL53L1CB_WaitMeasurementDataReady();
    
            status = board->sensor_right->VL53L1CB_GetMultiRangingData( pMultiRangingData);
    
            print_results( devRight.i2c_slave_address, pMultiRangingData );
    
            status = board->sensor_right->VL53L1CB_ClearInterrupt();
        }    
    }

    printf("Terminating.\n");
}


// print what ever results are required
void print_results( int devNumber, VL53L1_MultiRangingData_t *pMultiRangingData )
{
    int no_of_object_found = pMultiRangingData->NumberOfObjectsFound;
    int signal_rate = 0;
    int ambient_rate = 0;

    int RoiNumber = pMultiRangingData->RoiNumber;
    
 //   printf("no_of_object_found : %d\n", no_of_object_found);

    if (( no_of_object_found < 10 ) &&  ( no_of_object_found != 0)) {
        for(int j=0; j<no_of_object_found; j++) {
            if ((pMultiRangingData->RangeData[j].RangeStatus == VL53L1_RANGESTATUS_RANGE_VALID) ||
                    (pMultiRangingData->RangeData[j].RangeStatus == VL53L1_RANGESTATUS_RANGE_VALID_NO_WRAP_CHECK_FAIL)) {
                signal_rate = pMultiRangingData->RangeData[j].SignalRateRtnMegaCps / 65535;
                ambient_rate = pMultiRangingData->RangeData[j].AmbientRateRtnMegaCps / 65535;
                printf("\t i2cAddr=%d \t RoiNumber=%d   \t status=%d, \t D=%5dmm, \t Signal=%d Mcps, \t Ambient=%d Mcps \n",
                       devNumber, RoiNumber,
                       pMultiRangingData->RangeData[j].RangeStatus,
                       pMultiRangingData->RangeData[j].RangeMilliMeter,
                       signal_rate,
                       ambient_rate);
/*
// online compiler disables printf() / floating-point support, for code-size reasons.                        
// offline compiler can switch it on.
                printf("\t i2cAddr=%d \t RoiNumber=%d   \t status=%d, \t D=%5dmm, \t Signal=%2.2f Mcps, \t Ambient=%2.2f Mcps \n",
                       devNumber, RoiNumber,
                       pMultiRangingData->RangeData[j].RangeStatus,
                       pMultiRangingData->RangeData[j].RangeMilliMeter,
                       pMultiRangingData->RangeData[j].SignalRateRtnMegaCps / 65535.0,
                       pMultiRangingData->RangeData[j].AmbientRateRtnMegaCps / 65535.0);
*/                       
            }
        }
    } // if (( no_of_object_found < 10 ) &&  ( no_of_object_found != 0))
}


#if (MBED_VERSION  > 60300)
extern "C" void wait_ms(int ms)
{
    thread_sleep_for(ms);
}
#endif



