/*******************************************************************************
 * @file    vl53l1cb_class.h
 * @author  JS
 * @version V0.0.1
 * @date    15-January-2019
 * @brief   Header file for VL53L1 sensor component
 ******************************************************************************
 Copyright  2021, STMicroelectronics International N.V.
 All rights reserved.
 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions are met:
 * Redistributions of source code must retain the above copyright
 notice, this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 notice, this list of conditions and the following disclaimer in the
 documentation and/or other materials provided with the distribution.
 * Neither the name of STMicroelectronics nor the
 names of its contributors may be used to endorse or promote products
 derived from this software without specific prior written permission.
 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND
 NON-INFRINGEMENT OF INTELLECTUAL PROPERTY RIGHTS ARE DISCLAIMED.
 IN NO EVENT SHALL STMICROELECTRONICS INTERNATIONAL N.V. BE LIABLE FOR ANY
 DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *****************************************************************************/

#ifndef __VL53L1CB_CLASS_H
#define __VL53L1CB_CLASS_H


#ifdef _MSC_VER
#   ifdef VL53L1CB_API_EXPORTS
#       define VL53L1CB_API  __declspec(dllexport)
#   else
#       define VL53L1CB_API
#   endif
#else
#   define VL53L1CB_API
#endif


/* Includes ------------------------------------------------------------------*/

#include "mbed.h"
#include "PinNames.h"

#include "RangeSensor.h"
#include "Stmpe1600.h"

#include "vl53l1_error_codes.h"
#include "vl53l1_platform_user_data.h"

#include "ToF_I2C.h"
/**********************************************************/
#include "vl53l1_def.h"
/***********************************************************/


#define VL53L1CB_IMPLEMENTATION_VER_MAJOR       1
#define VL53L1CB_IMPLEMENTATION_VER_MINOR       0
#define VL53L1CB_IMPLEMENTATION_VER_SUB         2
#define VL53L1CB_IMPLEMENTATION_VER_REVISION  0000

typedef int8_t VL53L1CB_ERROR;

#define VL53L1_DEFAULT_DEVICE_ADDRESS      0x52
#define VL53L1_REG_IDENTIFICATION_MODEL_ID 0x010F

/* Classes -------------------------------------------------------------------*/
/** Class representing a VL53L1 sensor component
 */
class VL53L1CB : public RangeSensor
{
 public:
 
 #define VL53L1DevStructGetLLDriverHandle(Dev) (&Dev->Data.LLData)
    /** Constructor
     * @param[in] &i2c device I2C to be used for communication
     * @param[in] &pin_gpio1 pin Mbed InterruptIn PinName to be used as component GPIO_1 INT
     * @param[in] DevAddr device address, 0x52 by default
     */

    VL53L1CB(ToF_DevI2C *i2c, DigitalOut *pin, PinName pin_gpio1, uint8_t dev_addr = VL53L1_DEFAULT_DEVICE_ADDRESS) 
    : RangeSensor(), dev_i2c(i2c), _gpio0(pin)
    {
        MyDevice.i2c_slave_address = dev_addr;
        MyDevice.dev_i2c = (void*) i2c;
        Device = &MyDevice;
               
        _expgpio0 = NULL;
        if (pin_gpio1 != NC) {
            _gpio1Int = new InterruptIn(pin_gpio1);
        } else {
            _gpio1Int = NULL;
        }
    }
   
    /** Constructor 2 (STMPE1600DigiOut)
     * @param[in] i2c device I2C to be used for communication
     * @param[in] &pin Gpio Expander STMPE1600DigiOut pin to be used as component GPIO_0 CE
     * @param[in] pin_gpio1 pin Mbed InterruptIn PinName to be used as component GPIO_1 INT
     * @param[in] device address, 0x29 by default
     */
    VL53L1CB(ToF_DevI2C *i2c, Stmpe1600DigiOut *pin, PinName pin_gpio1,
            uint8_t dev_addr = VL53L1_DEFAULT_DEVICE_ADDRESS) 
            : dev_i2c(i2c), _expgpio0(pin)
    {
        MyDevice.i2c_slave_address = dev_addr;
        MyDevice.dev_i2c = (void*) i2c;
        Device = &MyDevice;
               
        _gpio0 = NULL;
        if (pin_gpio1 != NC) {
            _gpio1Int = new InterruptIn(pin_gpio1);
        } else {
            _gpio1Int = NULL;
        }
    }    
    
   /** Destructor
    */
    virtual ~VL53L1CB()
    {        
        if (_gpio1Int != NULL) {
            delete _gpio1Int;
        }
    }
    
        
    
    VL53L1_DEV getDevicePtr() { return Device; }

    
    /* warning: VL53L1 class inherits from GenericSensor, RangeSensor and LightSensor, that haven`t a destructor.
       The warning should request to introduce a virtual destructor to make sure to delete the object */

    /*** Interface Methods ***/
    /*** High level API ***/
    /**
     * @brief       PowerOn the sensor
     * @return      void
     */
    /* turns on the sensor */
    virtual void VL53L1CB_On(void)
    {
        printf("VL53L1_On\r\n");
        if (_gpio0) {
            *_gpio0 = 1;
        } else {
            if (_expgpio0) {
                *_expgpio0 = 1;
            }
        }
#if (MBED_VERSION > 60300)
    thread_sleep_for(100);
#else
    wait_ms(100);  // NEEDS A DELAY BETWEEN SENSORS
#endif
    }

    /**
     * @brief       PowerOff the sensor
     * @return      void
     */
    /* turns off the sensor */
    virtual void VL53L1CB_Off(void)
    {
        printf("VL53L1_Off\r\n");
        if (_gpio0) {
            *_gpio0 = 0;
        } else {
            if (_expgpio0) {
                *_expgpio0 = 0;
            }
        }
#if (MBED_VERSION > 60300)
    thread_sleep_for(100);
#else
    wait_ms(100);  // NEEDS A DELAY BETWEEN SENSORS
#endif
    }
    
    int is_present()
    {
        int status;
        uint8_t id = 0;

        status = read_id(&id);
        if (status) {
            printf("Failed to read ID device. Device not present!\n\r");
        }
        return status;
    }

    /**
     * @brief       Initialize the sensor with default values
     * @return      0 on Success
     */
     
     VL53L1CB_ERROR InitSensor(uint8_t address){
        VL53L1CB_ERROR status = 0;
        VL53L1CB_Off();
        VL53L1CB_On();
        status = VL53L1CB_WaitDeviceBooted();
        if(!status){
            status = VL53L1CB_SetI2CAddress(address);
        }
        if(!status){
            status = VL53L1CB_SensorInit();
        }      
        return status;
     }



/**
 *
 * @brief One time device initialization
 * @param void
 * @return     0 on success,  @a #CALIBRATION_WARNING if failed
 */
    virtual int init(void *init)
    {
         return VL53L1CB_SensorInit();
         return 0;
    }


    /**
     * @brief       Initialize the sensor with default values
     * @return      "0" on success
     */
    int init_sensor(uint8_t new_addr);

    /* Read function of the ID device */
    virtual int read_id(uint8_t *id){
        int status = 0;
        uint16_t rl_id = 0;
    
        uint8_t ExpanderData[2];
    
        ExpanderData[0] = 0;
        ExpanderData[1] = 0;
        rl_id = 0;
        dev_i2c->ToF_i2c_read(&ExpanderData[0], Device->i2c_slave_address, VL53L1_REG_IDENTIFICATION_MODEL_ID, 2);
    
        rl_id = (ExpanderData[0] << 8) + ExpanderData[1];
        printf("Model ID is: %d (%X)  \r\n",rl_id, rl_id);
    
        uint8_t tmp = 0;
        ExpanderData[0] = VL53L1_FIRMWARE__SYSTEM_STATUS >> 8;
        ExpanderData[1] = VL53L1_FIRMWARE__SYSTEM_STATUS & 0x0FF;
        dev_i2c->ToF_i2c_read(&tmp, Device->i2c_slave_address, VL53L1_FIRMWARE__SYSTEM_STATUS, 1);

        printf("Firmware system is: %d\r\n",tmp);
    
        if (rl_id == 0xEACC) {
            printf("Device is present %d:\r\n", rl_id);
            return status;
        }
        return -1;
    }


    /**
     * @brief       Interrupt handling func to be called by user after an INT is occurred
     * @param[out]  Data pointer to the distance to read data in to
     * @return      0 on Success
     */
    int handle_irq(uint16_t *distance);

    /**
     * @brief       Start the measure indicated by operating mode
     * @param[in]   fptr specifies call back function must be !NULL in case of interrupt measure
     * @return      0 on Success
     */
    int start_measurement(void (*fptr)(void));
    /**
     * @brief       Stop the currently running measure indicate by operating_mode
     * @return      0 on Success
     */
    int stop_measurement();
    /**
     * @brief       Get results for the measure
     * @param[out]  Data pointer to the distance_data to read data in to
     * @return      0 on Success
     */
    int get_measurement(uint16_t *distance);
    /**
     * @brief       Enable interrupt measure IRQ
     * @return      0 on Success
     */
    void enable_interrupt_measure_detection_irq(void)
    {
        if (_gpio1Int != NULL)
            _gpio1Int->enable_irq();
    }

    /**
     * @brief       Disable interrupt measure IRQ
     * @return      0 on Success
     */
    void disable_interrupt_measure_detection_irq(void)
    {
        if (_gpio1Int != NULL)
            _gpio1Int->disable_irq();
    }
    /**
     * @brief       Attach a function to call when an interrupt is detected, i.e. measurement is ready
     * @param[in]   fptr pointer to call back function to be called whenever an interrupt occours
     * @return      0 on Success
     */
    void attach_interrupt_measure_detection_irq(void (*fptr)(void))
    {
        if (_gpio1Int != NULL)
            _gpio1Int->rise(fptr);
    }

/**
 * @brief Get ranging result and only that
 * @param pRange_mm  Pointer to range distance
 * @return           0 on success
 */
    virtual int get_distance(uint32_t *piData)
    {
    int status;
    uint16_t distance;
    status = get_measurement(&distance);
    *piData = (uint32_t) distance;
    return status;
    }


/* vl53l1_api.h functions */

    /**
     * @brief This function returns the SW driver version
     */
    VL53L1CB_ERROR VL53L1CB_GetSWVersion(VL53L1_Version_t *pVersion);

    /**
     * @brief This function sets the sensor I2C address used in case multiple devices application, default address 0x52
     */
    VL53L1CB_ERROR VL53L1CB_SetI2CAddress(uint8_t new_address);

    /**
     * @brief This function loads the 135 bytes default values to initialize the sensor.
     * @param dev Device address
     * @return 0:success, != 0:failed
     */
    VL53L1CB_ERROR VL53L1CB_SensorInit();

    /**
     * @brief This function returns the boot state of the device (1:booted, 0:not booted)
     */
    VL53L1CB_ERROR VL53L1CB_BootState(uint8_t *state);

    
    /**************************************************************************/

    VL53L1CB_ERROR VL53L1CB_WaitDeviceBooted();
    
    VL53L1CB_ERROR VL53L1CB_GetCalibrationData(VL53L1_CalibrationData_t  *pCalibrationData);
        
    VL53L1_GPIO_Interrupt_Mode ConvertModeToLLD(VL53L1CB_ERROR *pStatus,
        VL53L1_ThresholdMode CrossMode);
      
    VL53L1CB_ERROR VL53L1CB_GetVersion(VL53L1_Version_t *pVersion);
    
    VL53L1CB_ERROR VL53L1CB_GetProductRevision(
    uint8_t *pProductRevisionMajor, uint8_t *pProductRevisionMinor);
    
    
    VL53L1CB_ERROR VL53L1CB_GetDeviceInfo(
    VL53L1_DeviceInfo_t *pVL53L1_DeviceInfo);
    
    VL53L1CB_ERROR VL53L1CB_GetUID( uint64_t *pUid);
    
    VL53L1CB_ERROR VL53L1CB_GetRangeStatusString(uint8_t RangeStatus,
    char *pRangeStatusString);
    
    
    VL53L1CB_ERROR VL53L1CB_GetPalErrorString(VL53L1CB_ERROR PalErrorCode,
    char *pPalErrorString);
    
    
    VL53L1CB_ERROR VL53L1CB_GetPalStateString(VL53L1_State PalStateCode,
    char *pPalStateString);
    
    
    VL53L1CB_ERROR VL53L1CB_GetPalState(VL53L1_State *pPalState);
    
    VL53L1CB_ERROR VL53L1CB_DataInit();
    
    VL53L1CB_ERROR VL53L1CB_StaticInit();
        
    VL53L1CB_ERROR VL53L1CB_ClearInterruptAndStartMeasurement();
    
    VL53L1CB_ERROR VL53L1CB_GetRangingMeasurementData(
        VL53L1_RangingMeasurementData_t *pRangingMeasurementData);

    VL53L1CB_ERROR VL53L1CB_GetMultiRangingData(
        VL53L1_MultiRangingData_t *pMultiRangingData);   
        
    VL53L1CB_ERROR VL53L1CB_GetAdditionalData(
                VL53L1_AdditionalData_t *pAdditionalData);
                
    VL53L1CB_ERROR VL53L1CB_SetTuningParameter(
        uint16_t TuningParameterId, int32_t TuningParameterValue);
        
    VL53L1CB_ERROR VL53L1CB_GetTuningParameter(
        uint16_t TuningParameterId, int32_t *pTuningParameterValue);
  
    VL53L1CB_ERROR VL53L1CB_SetXTalkCompensationEnable(
         uint8_t XTalkCompensationEnable);
         
         
    VL53L1CB_ERROR VL53L1CB_GetXTalkCompensationEnable(
            uint8_t *pXTalkCompensationEnable);
                       
    VL53L1CB_ERROR VL53L1CB_PerformXTalkCalibration(
    uint8_t CalibrationOption);
    
    VL53L1CB_ERROR VL53L1CB_SetOffsetCalibrationMode(
            VL53L1_OffsetCalibrationModes OffsetCalibrationMode);
            

    VL53L1CB_ERROR VL53L1CB_SetOffsetCorrectionMode(
            VL53L1_OffsetCorrectionModes OffsetCorrectionMode);
        
    VL53L1CB_ERROR VL53L1CB_PerformOffsetCalibration(
            int32_t CalDistanceMilliMeter, FixPoint1616_t CalReflectancePercent);
            
    VL53L1CB_ERROR VL53L1CB_PerformOffsetSimpleCalibration(
             int32_t CalDistanceMilliMeter);
    
    VL53L1CB_ERROR VL53L1CB_PerformOffsetZeroDistanceCalibration();
    
    VL53L1CB_ERROR VL53L1CB_SetCalibrationData(
        VL53L1_CalibrationData_t *pCalibrationData);
        
    VL53L1CB_ERROR VL53L1CB_SetZoneCalibrationData(
        VL53L1_ZoneCalibrationData_t *pZoneCalibrationData);
        
    VL53L1CB_ERROR VL53L1CB_GetZoneCalibrationData(
        VL53L1_ZoneCalibrationData_t  *pZoneCalibrationData);
        
    VL53L1CB_ERROR VL53L1CB_GetOpticalCenter(
        FixPoint1616_t *pOpticalCenterX,
        FixPoint1616_t *pOpticalCenterY);
        
    VL53L1CB_ERROR VL53L1CB_SetThresholdConfig(VL53L1_DetectionConfig_t *pConfig);
    
    
    VL53L1CB_ERROR VL53L1CB_GetLimitCheckStatus( uint16_t LimitCheckId,
                                              uint8_t *pLimitCheckStatus);    
                                              
        
    VL53L1CB_ERROR VL53L1CB_SetLimitCheckEnable( uint16_t LimitCheckId,
                                             uint8_t LimitCheckEnable);
        
    VL53L1CB_ERROR VL53L1CB_GetLimitCheckEnable(uint16_t LimitCheckId,
                                             uint8_t *pLimitCheckEnable);
    
    VL53L1CB_ERROR VL53L1CB_GetThresholdConfig(VL53L1_DetectionConfig_t *pConfig);
        
        
    VL53L1CB_ERROR VL53L1CB_PerformOffsetPerVcselCalibration(int32_t CalDistanceMilliMeter);
  
    VL53L1CB_ERROR VL53L1CB_GetNumberOfLimitCheck(uint16_t *pNumberOfLimitCheck);      
    
    VL53L1CB_ERROR VL53L1CB_GetLimitCheckInfo(uint16_t LimitCheckId,
                                           char *pLimitCheckString);       
        
    VL53L1CB_ERROR VL53L1CB_SetPresetMode(VL53L1_PresetModes PresetMode);
        
    VL53L1CB_ERROR VL53L1CB_GetPresetMode( VL53L1_PresetModes *pPresetMode);
                                    
    VL53L1CB_ERROR VL53L1CB_SetDistanceMode( VL53L1_DistanceModes DistanceMode);
                                                                                    
    VL53L1CB_ERROR VL53L1CB_GetDistanceMode(VL53L1_DistanceModes *pDistanceMode);
                                             
    VL53L1CB_ERROR VL53L1CB_SetOutputMode(VL53L1_OutputModes OutputMode);
                                           
    VL53L1CB_ERROR VL53L1CB_GetOutputMode(VL53L1_OutputModes *pOutputMode);
                                        
    VL53L1CB_ERROR VL53L1CB_SetMeasurementTimingBudgetMicroSeconds(uint32_t MeasurementTimingBudgetMicroSeconds);
                                    
    VL53L1CB_ERROR VL53L1CB_SetInterMeasurementPeriodMilliSeconds(uint32_t InterMeasurementPeriodMilliSeconds);
                                   
    
    VL53L1CB_ERROR VL53L1CB_GetInterMeasurementPeriodMilliSeconds(
            uint32_t *pInterMeasurementPeriodMilliSeconds);
        
        
    VL53L1CB_ERROR VL53L1CB_SetDmaxReflectance( FixPoint1616_t DmaxReflectance);
                                                
    VL53L1CB_ERROR VL53L1CB_GetDmaxReflectance(FixPoint1616_t *pDmaxReflectance);
                                                
                                                
    VL53L1CB_ERROR VL53L1CB_GetDmaxMode( VL53L1_DeviceDmaxModes *pDmaxMode);    
                                    
    VL53L1CB_ERROR VL53L1CB_GetMeasurementTimingBudgetMicroSeconds(
                                 uint32_t *pMeasurementTimingBudgetMicroSeconds);  
                                 
    VL53L1CB_ERROR VL53L1CB_SetDmaxMode(VL53L1_DeviceDmaxModes DmaxMode);    
                                     
    VL53L1CB_ERROR VL53L1CB_SetLimitCheckValue( uint16_t LimitCheckId,
                                            FixPoint1616_t LimitCheckValue);
                                            
    VL53L1CB_ERROR VL53L1CB_GetLimitCheckValue(uint16_t LimitCheckId,
                                            FixPoint1616_t *pLimitCheckValue);   
                                            
    VL53L1CB_ERROR VL53L1CB_GetLimitCheckCurrent( uint16_t LimitCheckId,
                                              FixPoint1616_t *pLimitCheckCurrent);   
                                            
    VL53L1CB_ERROR VL53L1CB_GetMaxNumberOfROI( uint8_t *pMaxNumberOfROI);    
                                           
    VL53L1CB_ERROR VL53L1CB_SetROI(VL53L1_RoiConfig_t *pRoiConfig);
                                
     VL53L1CB_ERROR VL53L1CB_GetROI(VL53L1_RoiConfig_t *pRoiConfig);
                                 
    VL53L1CB_ERROR VL53L1CB_GetNumberOfSequenceSteps(uint8_t *pNumberOfSequenceSteps);          
    
    VL53L1CB_ERROR VL53L1CB_GetSequenceStepsInfo(VL53L1_SequenceStepId SequenceStepId,
                                              char *pSequenceStepsString);   
                                              
                                              
    VL53L1CB_ERROR VL53L1CB_SetSequenceStepEnable(VL53L1_SequenceStepId SequenceStepId, uint8_t SequenceStepEnabled);    
        
        
    VL53L1CB_ERROR VL53L1CB_GetSequenceStepEnable(VL53L1_SequenceStepId SequenceStepId, uint8_t *pSequenceStepEnabled); 
           
    
    VL53L1CB_ERROR VL53L1CB_StartMeasurement();
    
    VL53L1CB_ERROR VL53L1CB_StopMeasurement();
    
    
    VL53L1CB_ERROR VL53L1CB_GetMeasurementDataReady(uint8_t *pMeasurementDataReady); 
                                                 
    VL53L1CB_ERROR VL53L1CB_WaitMeasurementDataReady();        
    
    VL53L1CB_ERROR VL53L1CB_SmudgeCorrectionEnable(VL53L1_SmudgeCorrectionModes Mode);                                                                                                                                                    
    /***************************************************************************/
    

    /* Write and read functions from I2C */

    VL53L1CB_ERROR VL53L1_WrByte(VL53L1_DEV dev, uint16_t index, uint8_t data);  // DON'T CHANGE NAME, SATELLITES STOP WORKING
    VL53L1CB_ERROR VL53L1_WrWord(VL53L1_DEV dev, uint16_t index, uint16_t data); // DON'T CHANGE NAME,  SATELLITES STOP WORKING
    VL53L1CB_ERROR VL53L1_WrDWord(VL53L1_DEV dev, uint16_t index, uint32_t data); // DON'T CHANGE NAME, SATELLITES STOP WORKING
    VL53L1CB_ERROR VL53L1_RdByte(VL53L1_DEV dev, uint16_t index, uint8_t *data); // DON'T CHANGE NAME,  SATELLITES STOP WORKING
    VL53L1CB_ERROR VL53L1_RdWord(VL53L1_DEV dev, uint16_t index, uint16_t *data); // DON'T CHANGE NAME, SATELLITES STOP WORKING
    VL53L1CB_ERROR VL53L1_RdDWord(VL53L1_DEV dev, uint16_t index, uint32_t *data);// DON'T CHANGE NAME,  SATELLITES STOP WORKING
    VL53L1CB_ERROR VL53L1_UpdateByte(VL53L1_DEV dev, uint16_t index, uint8_t AndData, uint8_t OrData); // DON'T CHANGE NAME,  SATELLITES STOP WORKING

    VL53L1CB_ERROR VL53L1CB_WriteMulti(VL53L1_DEV Dev, uint16_t index, uint8_t *pdata, uint32_t count);
    VL53L1CB_ERROR VL53L1CB_ReadMulti(VL53L1_DEV Dev, uint16_t index, uint8_t *pdata, uint32_t count);

    VL53L1CB_ERROR VL53L1CB_I2CWrite(uint8_t dev, uint16_t index, uint8_t *data, uint16_t number_of_bytes);
    VL53L1CB_ERROR VL53L1CB_I2CRead(uint8_t dev, uint16_t index, uint8_t *data, uint16_t number_of_bytes);
    VL53L1CB_ERROR VL53L1CB_GetTickCount(uint32_t *ptick_count_ms);
    VL53L1CB_ERROR VL53L1CB_WaitUs(VL53L1_Dev_t *pdev, int32_t wait_us);
    VL53L1CB_ERROR VL53L1CB_WaitMs(VL53L1_Dev_t *pdev, int32_t wait_ms);
    
    VL53L1CB_ERROR VL53L1CB_WaitValueMaskEx(VL53L1_Dev_t *pdev, uint32_t timeout_ms, uint16_t index, uint8_t value, uint8_t mask, uint32_t poll_delay_ms);
    VL53L1CB_ERROR VL53L1CB_SetDeviceAddress(VL53L1_DEV Dev, uint8_t DeviceAddress);


 protected:

    /* IO Device */
    ToF_DevI2C *dev_i2c;
    
    /* Digital out pin */
    DigitalOut *_gpio0;
    /* GPIO expander */
    Stmpe1600DigiOut *_expgpio0;
    /* Measure detection IRQ */
    InterruptIn *_gpio1Int;
 
    ///* Digital out pin */
    /* Device data */
    VL53L1_Dev_t MyDevice;
    VL53L1_DEV Device;
};


#endif /* _VL53L1CB_CLASS_H_ */
