
// SPDX-License-Identifier: GPL-2.0+ OR BSD-3-Clause
/******************************************************************************
 * Copyright (c) 2020, STMicroelectronics - All Rights Reserved

 This file is part of VL53L1 and is dual licensed,
 either GPL-2.0+
 or 'BSD 3-clause "New" or "Revised" License' , at your option.
 ******************************************************************************
 */

unsigned int i2creadCount = 0;
unsigned int i2cwriteCount = 0;
unsigned char SPI2C_Buffer[256];


#include <mbed_wait_api.h>
#include "vl53l1_platform.h"
#include "vl53l1_platform_log.h"
#include "vl53l1_api.h"
#include "spi_interface.h"
#include <string.h>
#include <time.h>
#include <math.h>



#define I2C_TIME_OUT_BASE   10
#define I2C_TIME_OUT_BYTE   1

#ifdef VL53L1_LOG_ENABLE
#define trace_print(level, ...) VL53L1_trace_print_module_function(VL53L1_TRACE_MODULE_PLATFORM, level, VL53L1_TRACE_FUNCTION_NONE, ##__VA_ARGS__)
#define trace_i2c(...) VL53L1_trace_print_module_function(VL53L1_TRACE_MODULE_NONE, VL53L1_TRACE_LEVEL_NONE, VL53L1_TRACE_FUNCTION_I2C, ##__VA_ARGS__)
#endif

/* when not customized by application define dummy one */
#ifndef VL53L1_GetI2cBus
/** This macro can be overloaded by user to enforce i2c sharing in RTOS context
 */
#   define VL53L1_GetI2cBus(...) (void)0
#endif

#ifndef VL53L1_PutI2cBus
/** This macro can be overloaded by user to enforce i2c sharing in RTOS context
 */
#   define VL53L1_PutI2cBus(...) (void)0
#endif

uint8_t _I2CBuffer[256];


VL53L1_Error VL53L1_WriteMulti(VL53L1_DEV Dev, uint16_t index, uint8_t *pdata, uint32_t count)
{
   int  status;
//   printf("VL53L1_WriteMulti %d %d %d \n",Dev->I2cDevAddr,index,count);
   status = v53l1cb_i2c_write_if(Dev->dev_i2c, pdata,Dev->i2c_slave_address, index,count);
   return status;
}


VL53L1_Error VL53L1_ReadMulti(VL53L1_DEV Dev, uint16_t index, uint8_t *pdata, uint32_t count)
{
    int status;

    status = v53l1cb_i2c_read_if(Dev->dev_i2c, pdata,Dev->i2c_slave_address, index,count);

    return status;
}


VL53L1_Error VL53L1_WrByte(VL53L1_DEV Dev, uint16_t index, uint8_t data)
{
   int  status;

   status = v53l1cb_i2c_write_if(Dev->dev_i2c, &data,Dev->i2c_slave_address, index,1);
   return status;
}


VL53L1_Error VL53L1_WrWord(VL53L1_DEV Dev, uint16_t index, uint16_t data)
{
   int  status;
   uint8_t buffer[2];

   buffer[0] = data >> 8;
   buffer[1] = data & 0x00FF;
   status = v53l1cb_i2c_write_if(Dev->dev_i2c, (uint8_t *)buffer,Dev->i2c_slave_address, index,2);
   return status;
}


VL53L1_Error VL53L1_WrDWord(VL53L1_DEV Dev, uint16_t index, uint32_t data)
{
   int  status;
   uint8_t buffer[4];

    buffer[0] = (data >> 24) & 0xFF;
    buffer[1] = (data >> 16) & 0xFF;
    buffer[2] = (data >>  8) & 0xFF;
    buffer[3] = (data >>  0) & 0xFF;
    status = v53l1cb_i2c_write_if(Dev->dev_i2c, (uint8_t *)buffer,Dev->i2c_slave_address, index,4);
   return status;
}

VL53L1_Error VL53L1_UpdateByte(VL53L1_DEV Dev, uint16_t index, uint8_t AndData, uint8_t OrData)
{
   int  status;
   uint8_t buffer = 0;

   /* read data direct onto buffer */
   status = v53l1cb_i2c_read_if(Dev->dev_i2c, &buffer,Dev->i2c_slave_address, index,1);
   if (!status)
   {
      buffer = (buffer & AndData) | OrData;
      status = v53l1cb_i2c_write_if(Dev->dev_i2c, &buffer,Dev->i2c_slave_address, index,1);
   }
   return status;
}

VL53L1_Error VL53L1_RdByte(VL53L1_DEV Dev, uint16_t index, uint8_t *data)
{
   int  status;

   status = v53l1cb_i2c_read_if(Dev->dev_i2c, data,Dev->i2c_slave_address, index,1);  //is this correct
 // printf("VL53L1_RdByte %d %d %d\n",Dev->i2c_slave_address, status,*data);
   if(status)
     return -1;

   return 0;
}


VL53L1_Error VL53L1_RdWord(VL53L1_DEV Dev, uint16_t index, uint16_t *data)
{
   int  status;
   uint8_t buffer[2] = {0,0};
   
   status = v53l1cb_i2c_read_if(Dev->dev_i2c, buffer,Dev->i2c_slave_address, index,2);  //is this correct
   if (!status)
   {
       *data = (buffer[0] << 8) + buffer[1];
   }
   return status;

}



VL53L1_Error VL53L1_RdDWord(VL53L1_DEV Dev, uint16_t index, uint32_t *data)
{
   int status;
   uint8_t buffer[4] = {0,0,0,0};

   status = v53l1cb_i2c_read_if(Dev->dev_i2c, buffer,Dev->i2c_slave_address, index,4);
   if(!status)
   {
       *data = (buffer[0] << 24) + (buffer[1] << 16) + (buffer[2] << 8) + buffer[3];
   }
   return status;

}


VL53L1_Error VL53L1_GetTickCount(
    VL53L1_DEV Dev,
    uint32_t *ptick_count_ms)
{

    /* Returns current tick count in [ms] */

    VL53L1_Error status  = VL53L1_ERROR_NONE;
    (void) Dev;

    GetTickCount( ptick_count_ms); 

#ifdef VL53L1_LOG_ENABLE
    trace_print(
        VL53L1_TRACE_LEVEL_DEBUG,
        "VL53L1_GetTickCount() = %5u ms;\n",
    *ptick_count_ms);
#endif

    return status;
}


#define trace_print(level, ...) \
    _LOG_TRACE_PRINT(VL53L1_TRACE_MODULE_PLATFORM, \
    level, VL53L1_TRACE_FUNCTION_NONE, ##__VA_ARGS__)

#define trace_i2c(...) \
    _LOG_TRACE_PRINT(VL53L1_TRACE_MODULE_NONE, \
    VL53L1_TRACE_LEVEL_NONE, VL53L1_TRACE_FUNCTION_I2C, ##__VA_ARGS__)


VL53L1_Error VL53L1_GetTimerFrequency(int32_t *ptimer_freq_hz)
{
    *ptimer_freq_hz = 0;
    
    trace_print(VL53L1_TRACE_LEVEL_INFO, "VL53L1_GetTimerFrequency: Freq : %dHz\n", *ptimer_freq_hz);
    return VL53L1_ERROR_NONE;
}


VL53L1_Error VL53L1_WaitMs(VL53L1_Dev_t *pdev, int32_t wait_time){
    (void)pdev;
    wait_us(wait_time * 1000);
    return VL53L1_ERROR_NONE;
}


VL53L1_Error VL53L1_WaitUs(VL53L1_Dev_t *pdev, int32_t wait_time){
    (void)pdev;
    wait_us(wait_time);
    return VL53L1_ERROR_NONE;
}

VL53L1_Error VL53L1_WaitValueMaskEx(
    VL53L1_Dev_t *pdev,
    uint32_t      timeout_ms,
    uint16_t      index,
    uint8_t       value,
    uint8_t       mask,
    uint32_t      poll_delay_ms)
{

    /*
     * Platform implementation of WaitValueMaskEx V2WReg script command
     *
     * WaitValueMaskEx(
     *          duration_ms,
     *          index,
     *          value,
     *          mask,
     *          poll_delay_ms);
     */

    VL53L1_Error status         = VL53L1_ERROR_NONE;
    uint32_t     start_time_ms = 0;
    uint32_t     current_time_ms = 0;
    uint32_t     polling_time_ms = 0;
    uint8_t      byte_value      = 0;
    uint8_t      found           = 0;
#ifdef VL53L1_LOG_ENABLE
    uint8_t      trace_functions = VL53L1_TRACE_FUNCTION_NONE;
#endif

    char   register_name[VL53L1_MAX_STRING_LENGTH];

    /* look up register name */
#ifdef PAL_EXTENDED
    VL53L1_get_register_name(
            index,
            register_name);
#else
    VL53L1_COPYSTRING(register_name, "");
#endif

    /* Output to I2C logger for FMT/DFT  */

    /*trace_i2c("WaitValueMaskEx(%5d, 0x%04X, 0x%02X, 0x%02X, %5d);\n",
                 timeout_ms, index, value, mask, poll_delay_ms); */
    trace_i2c("WaitValueMaskEx(%5d, %s, 0x%02X, 0x%02X, %5d);\n",
                 timeout_ms, register_name, value, mask, poll_delay_ms);

    /* calculate time limit in absolute time */

     VL53L1_GetTickCount(pdev, &start_time_ms);

     
     VL53L1_WaitMs(pdev, 10);

    /* remember current trace functions and temporarily disable
     * function logging
     */

#ifdef VL53L1_LOG_ENABLE
    trace_functions = VL53L1_get_trace_functions();
    VL53L1_set_trace_functions(VL53L1_TRACE_FUNCTION_NONE);
#endif

    /* wait until value is found, timeout reached on error occurred */
    while ((status == VL53L1_ERROR_NONE) &&
           (polling_time_ms < timeout_ms) &&
           (found == 0)) {

        if (status == VL53L1_ERROR_NONE)
            status = VL53L1_RdByte(
                            pdev,
                            index,
                            &byte_value);

        if ((byte_value & mask) == value)
            found = 1;

        if (status == VL53L1_ERROR_NONE  &&
            found == 0 &&
            poll_delay_ms > 0)
            status = VL53L1_WaitMs(
                    pdev,
                    poll_delay_ms);
        /* Update polling time (Compare difference rather than absolute to
        negate 32bit wrap around issue) */
        VL53L1_GetTickCount(pdev, &current_time_ms);
        polling_time_ms = current_time_ms - start_time_ms;

    }
 //   printf("polling_time_ms %d \n",polling_time_ms);
#ifdef VL53L1_LOG_ENABLE
    /* Restore function logging */
    VL53L1_set_trace_functions(trace_functions);
#endif

    if (found == 0 && status == VL53L1_ERROR_NONE)
        status = VL53L1_ERROR_TIME_OUT;
        
    return status;
}




