/*
 *  This VL53L1X Expansion board test application performs range measurements
 *  using the onboard embedded centre sensor.
 *
 *  Measured ranges are ouput on the Serial Port, running at 115200 baud.
 *
 *  The Black Reset button is used to restart the program.
 *
 *  *** NOTE : By default hardlinks U10, U11, U15 & U18, on the underside of
 *            the X-NUCELO-53L0A1 expansion board are not made/OFF.
 *            These links must be made to allow interrupts from the Satellite boards
 *            to be received.
 *            U11 and U18 must be made/ON to allow interrupts to be received from the
 *            INT_L & INT_R positions; or
 *            U10 and U15 must be made/ON to allow interrupts to be received from the
 *            Alternate INT_L & INT_R positions.
 *            The X_NUCLEO_53L1A1 firmware library defaults to use the INT_L/INT_R
 *            positions.
 *            INT_L is available on expansion board Arduino Connector CN5, pin 1 as D9.
 *            Alternate INT_L is on CN5 Connector pin 2 as D8.
 *            INT_R is available on expansion board Arduino Connector CN9, pin 3 as D4.
 *            Alternate INT_R is on CN9 Connector pin 5 as D2.
 *            The pinouts are shown here : https://developer.mbed.org/components/X-NUCLEO-53L1A1/
 */

#include <stdio.h>

#include "mbed.h"
#include "XNucleo53L1A1.h"
#include "vl53L1x_I2c.h"

#define VL53L1_I2C_SDA   D14
#define VL53L1_I2C_SCL   D15

static XNucleo53L1A1 *board=NULL;

volatile bool polling_stop = false;

void stop_polling(void)
{
    polling_stop = true;
}

/*=================================== Main ==================================
=============================================================================*/
int main()
{
    int status = 0;
    uint8_t ready_centre = 0;
    uint16_t distance_centre = 0;

    printf("Hello world!\r\n");

    vl53L1X_DevI2C *dev_I2C = new vl53L1X_DevI2C(VL53L1_I2C_SDA, VL53L1_I2C_SCL);

    printf("I2C device created!\r\n");

    /* creates the 53L1A1 expansion board singleton obj */
    board = XNucleo53L1A1::instance(dev_I2C, A2, D9, D2);
    printf("board created!\r\n");

    /* init the 53L1A1 expansion board with default values */
    status = board->init_board();
    if (status) {
        printf("Failed to init board!\r\n");
        return status;
    }

    printf("board initiated! - %d\r\n", status);

    /* Start ranging on the centre sensor */
    if (board->sensor_centre != NULL) {
        status = board->sensor_centre->VL53L1X_StartRanging();
        if (status != 0) {
            printf("Centre sensor failed to start ranging!\r\n");
            return status;
        }
    }

    /* Ranging loop
     * Checks each sensor for data ready
     */
    while (1)
    {
        if (polling_stop)
        {
            printf("\r\nEnding loop mode \r\n");
            break;
        }
        if (board->sensor_centre != NULL) {
            board->sensor_centre->VL53L1X_CheckForDataReady(&ready_centre);
        }
        if (ready_centre) {
            board->sensor_centre->VL53L1X_GetRangeStatus(&ready_centre);
            board->sensor_centre->VL53L1X_GetDistance(&distance_centre);
        }

        if (board->sensor_centre != NULL) {
            printf("Distance read by sensor is : %d\r\n", distance_centre);
        }
    }

    return status;
}
