/**
 *******************************************************************************
 * @file    main.cpp
 * @author  Davide Aliprandi, STMicroelectronics
 * @version V1.0.0
 * @date    May 3rd, 2018
 * @brief   mbed test application for the STMicroelectronics X-NUCLEO-IHM14A1
 *          Motor Control Expansion Board: control of 1 motor.
 * @note    (C) COPYRIGHT 2018 STMicroelectronics
 *******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2018 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *******************************************************************************
 */


/* Includes ------------------------------------------------------------------*/

/* mbed specific header files. */
#include "mbed.h"

/* Component specific header files. */
#include "STSPIN820.h"


/* Definitions ---------------------------------------------------------------*/

/* Steps. */
#define STEP_ANGLE            (18) /* Decidegrees. */
#define FULL_STEPS_PER_TURN   (3600/(STEP_ANGLE))

/* Delay in milliseconds. */
#define DELAY_1   40
#define DELAY_2  100
#define DELAY_3 1000
#define DELAY_4 2000
#define DELAY_5 4000
#define DELAY_6 5000


/* Variables -----------------------------------------------------------------*/

/* Initialization parameters. */
STSPIN820_init_t init =
{
    480,             //Acceleration rate in pulse/s2 (must be greater than 0)
    480,             //Deceleration rate in pulse/s2 (must be greater than 0)
    1600,            //Running speed in pulse/s (8 pulse/s < Maximum speed <= 10 000 pulse/s )
    400,             //Minimum speed in pulse/s (8 pulse/s <= Minimum speed < 10 000 pulse/s)
    25,              //Acceleration current torque in % (from 0 to 100)
    20,              //Deceleration current torque in % (from 0 to 100)
    15,              //Running current torque in % (from 0 to 100)
    30,              //Holding current torque in % (from 0 to 100)
    TRUE,            //Torque boost speed enable
    200,             //Torque boost speed threshold in fullstep/s
    STEP_MODE_1_32,  //Step mode via enum motor_step_mode_t  
    HOLD_MODE,       //Automatic HIZ STOP
    100000           //REF frequency (Hz)
};

/* Motor Control Component. */
STSPIN820 *motor;


/* Functions -----------------------------------------------------------------*/

/**
 * @brief  This is an example of user handler for the flag interrupt.
 * @param  None
 * @retval None
 * @note   If needed, implement it, and then attach and enable it:
 *           + motor->attach_flag_irq(&flag_irq_handler);
 *           + motor->enable_flag_irq();
 *         To disable it:
 *           + motor->disble_flag_irq();
 */
void en_fault_irq_handler(void)
{
    printf("    WARNING: \"EN_FAULT\" interrupt triggered.\r\n");

    /* When the "EN" pin is forced low by a failure, configure the GPIO as an
       ouput low. */
    motor->cmd_disable();
}

/**
  * @brief  This is an example of user handler for the errors.
  * @param  error Error code.
  * @retval None
  * @note   If needed, implement it, and then attach it:
  *           + motor->attach_error_handler(&error_handler);
  */
void error_handler(uint16_t error)
{
    /* Printing to the console. */
    printf("    ERROR: \"%d\".\r\n", error);
    
    /* Aborting the program. */
    exit(EXIT_FAILURE);
}


/* Main ----------------------------------------------------------------------*/

int main(void)
{
int32_t pos;
uint16_t mySpeed;
uint16_t myDec;
motor_step_mode_t mystep_mode;
uint32_t myFreqPwm;
uint8_t myTorque;

  
    /*----- Initialization. -----*/

    /* Initializing Motor Control Component. */
int i = 0;
printf("%d\r\n", i++);
    motor = new STSPIN820(
        D2, D8, D7,
        D9, D5, D3,
        D13, D12, D11);
printf("%d\r\n", i++);
    if (motor->init(&init) != COMPONENT_OK) {
        exit(EXIT_FAILURE);
    }
printf("%d\r\n", i++);
    /* Configuring the DECAY pin. */
    motor->set_decay_mode(MIXED_DECAY);
printf("%d\r\n", i++);
    /* Attaching and enabling interrupt handlers. */
    motor->attach_en_fault_irq(&en_fault_irq_handler);
    motor->enable_en_fault_irq();
printf("%d\r\n", i++);
    /* Attaching error handler. */
    motor->attach_error_handler(&error_handler);

    /* Printing to the console. */
    printf("Motor Control Application Example for 1 Motor\r\n\n");


    /*----- Moving. -----*/

    //----- Get the PWM frequency used for the VREFA voltage generation
    
    myFreqPwm = motor->get_bridge_input_pwm_freq();
    
    //----- Set the PWM frequency used for the VREFA voltage generation
    
    motor->set_bridge_input_pwm_freq(myFreqPwm>>1);
    
    //----- Get the predefined step mode
    
    mystep_mode = motor->get_step_mode();
    
    //----- Move device 2 turns in the FORWARD direction using predefined microsteps
    
    /* Move 2 * FULL_STEPS_PER_TURN << mystep_mode microsteps in the FORWARD direction */
    motor->move(StepperMotor::FWD, 2 * FULL_STEPS_PER_TURN << mystep_mode);
    
    /* Waiting the motor ends moving */
    motor->wait_while_active();
    
    /* Waiting. */
    wait_ms(DELAY_3);
    
    //----- Disable the power bridges
    
    /* Disable the power bridges */
    motor->cmd_disable();
    
    /* Waiting. */
    wait_ms(DELAY_3);  
    
    //----- Stop with power bridges disabled
    
    /* Turn off power bridges when motor is stopped */
    motor->set_stop_mode(HIZ_MODE);
    
    //----- Move device 2 turns in the BACKWARD direction using predefined microsteps
    
    /* Move 2 * FULL_STEPS_PER_TURN << mystep_mode microsteps in the BACKWARD direction */
    motor->move(StepperMotor::BWD, 2 * FULL_STEPS_PER_TURN << mystep_mode);
    
    /* Waiting the motor ends moving */
    motor->wait_while_active();
    
    /* Set the current position to be the Home position */
    motor->set_home();
    
    /* Waiting. */
    wait_ms(DELAY_3);
    
    //----- Stop with power bridges enabled
    
    /* Keep power bridges on when motor is stopped */
    motor->set_stop_mode(HOLD_MODE);   
    
    //----- Go to position -6400
    
    /* Request to go to position -6400 */
    motor->go_to(-6400);
    
    /* Waiting the motor ends moving */
    motor->wait_while_active();
    
    /* Get current position */
    pos = motor->get_position();

    if (pos != -6400) 
    {
        error_handler(STSPIN820_ERROR_POSITION);
    }
    
    /* Set the current position to be the Mark position */
    motor->set_mark();
    
    /* Waiting. */
    wait_ms(DELAY_3);
    
    //----- Go Home
    
    /* Request to go to Home */
    motor->go_home();
    motor->wait_while_active();
    
    /* Get current position */
    pos = motor->get_position();
    
    /* Waiting. */
    wait_ms(DELAY_3);
    
    //----- Go to position 6400
    
    /* Request to go to position 6400 */
    motor->go_to_dir(StepperMotor::FWD, 6400);
    
    /* Waiting the motor ends moving */
    motor->wait_while_active();
    
    /* Get current position */
    pos = motor->get_position();
    
    /* Waiting. */
    wait_ms(DELAY_3);
    
    //----- Go Mark which was set previously after go to -6400
    
    /* Request to go to Mark position */
    motor->go_mark();
    
    /* Waiting the motor ends moving */
    motor->wait_while_active();
    
    /* Get current position */
    pos = motor->get_position();
    
    /* Waiting. */
    wait_ms(DELAY_3);
    
    //----- Run the motor BACKWARD
    
    /* Request to run BACKWARD */
    motor->run(StepperMotor::BWD);
    
    /* Waiting. */
    wait_ms(DELAY_6);
    
    /* Get current speed */
    mySpeed = motor->get_speed();
    
    //----- Increase the speed while running
    
    /* Increase speed to 2400 microstep/s */
    motor->set_max_speed(2400);

    /* Waiting. */
    wait_ms(DELAY_6);
    
    /* Get current speed */
    mySpeed = motor->get_speed();
    
    //----- Decrease the speed while running
    
    /* Decrease speed to 1200 microstep/s */
    motor->set_max_speed(1200);

    /* Waiting. */
    wait_ms(DELAY_6);
    
    /* Get current speed */
    mySpeed = motor->get_speed();
    
    //----- Increase acceleration while running
    
    /* Increase acceleration to 2000 microstep/s^2 */
    motor->set_acceleration(2000);

    /* Waiting. */
    wait_ms(DELAY_6);
    
    /* Increase speed to 2400 microstep/s */
    motor->set_max_speed(2400);

    /* Waiting. */
    wait_ms(DELAY_6);
    
    /* Get current speed */
    mySpeed = motor->get_speed();
    
    if (mySpeed != 2400)
    {
        error_handler(STSPIN820_ERROR_SPEED);
    }
    //----- Get current deceleration
    
    myDec = motor->get_deceleration();
    
    //----- Increase deceleration while running
    
    /* Increase deceleration to 2000 microstep/s^2 */
    motor->set_deceleration(2000);

    /* Waiting. */
    wait_ms(DELAY_6);
    
    /* Decrease speed to 1200 microstep/s */
    motor->set_max_speed(1200);
    wait_ms(DELAY_6);
    
    /* Get current speed */
    mySpeed = motor->get_speed();
    
    //----- Retore previous deceleration
    
    motor->set_deceleration(myDec);
    
    //----- Soft stopped required while running
    
    /* Request soft stop */
    motor->soft_stop();
    
    /* Waiting the motor ends moving */  
    motor->wait_while_active();
    
    /* Waiting. */
    wait_ms(DELAY_4);
    
    //----- Run stopped by hard_stop
    
    /* Request to run in FORWARD direction */
    motor->run(StepperMotor::FWD);
    wait_ms(DELAY_6);
    
    /* Request to immediatly stop */
    motor->hard_stop();
    motor->wait_while_active();
    
    /* Waiting. */
    wait_ms(DELAY_4);
    
    //----- go_to stopped by soft_stop
    
    /* Request to go to position 20000  */
    motor->go_to(20000);

    /* Waiting. */
    wait_ms(DELAY_6);
    
    /* Request to perform a soft stop */
    motor->soft_stop();
    motor->wait_while_active();
    
    /* Waiting. */
    wait_ms(DELAY_4);
    
    //----- Change current torque while running (RUN_TORQUE value is kept unchanged)
    
    /* Get the running torque */
    myTorque = motor->get_torque(RUN_TORQUE);
    
    /* Request to run in FORWARD direction */
    motor->run(StepperMotor::FWD);
    
    /* When the motor is running steady for 100ms, decrease the current torque. */
    while (motor->get_device_state()!=STEADY);

    /* Waiting. */
    wait_ms(DELAY_2);
    motor->set_torque(CURRENT_TORQUE, myTorque>>1);
    
    /* Waiting. */
    wait_ms(DELAY_3);
    
    /* Request soft stop */
    motor->soft_stop();
    
    /* Waiting the motor ends moving */  
    motor->wait_while_active();
    
    /* Get the running torque and check it has not change */
    if (motor->get_torque(RUN_TORQUE) != myTorque) 
    {
        error_handler(STSPIN820_ERROR_POSITION);
    }
    
    /* Waiting. */
    wait_ms(DELAY_3); 
    
    //----- Change the holding torque (HOLD TORQUE value is changed)
    
    motor->set_torque(HOLD_TORQUE, 5);
    
    /* Waiting. */
    wait_ms(DELAY_3);
    
    ////----- Change step mode to full step mode
    
    /* Select full step mode */
    motor->set_step_mode((StepperMotor::step_mode_t) STEP_MODE_FULL);
    
    /* Set speed, acceleration and deceleration to scale with full step mode */
    if (motor->set_min_speed(motor->get_min_speed()>>mystep_mode) == FALSE)
    {
        error_handler(STSPIN820_ERROR_SET_MIN_SPEED);
    }
    if (motor->set_max_speed(motor->get_max_speed()>>mystep_mode) == FALSE)
    {
        error_handler(STSPIN820_ERROR_SET_MAX_SPEED);
    }
    if (motor->set_acceleration(motor->get_acceleration()>>mystep_mode) == FALSE)
    {
        error_handler(STSPIN820_ERROR_SET_ACCELERATION);
    }
    if (motor->set_deceleration(motor->get_deceleration()>>mystep_mode) == FALSE)
    {
        error_handler(STSPIN820_ERROR_SET_DECELERATION);
    }
    
    motor->set_home();
    
    /* Request to go position 200 (full steps) */
    motor->go_to(200);
    
    /* Waiting the motor ends moving */
    motor->wait_while_active();
    
    /* Get current position */
    pos = motor->get_position();
    
    if (pos != 200) 
    {
        error_handler(STSPIN820_ERROR_POSITION);
    }
    
    /* Waiting. */
    wait_ms(DELAY_4);
    
    //----- Restore step mode to predefined step mode
    /* Select predefined step mode */
    motor->set_step_mode((StepperMotor::step_mode_t) mystep_mode);
    
    /* Set speed, acceleration and deceleration to scale with microstep mode */
    if (motor->set_max_speed(motor->get_max_speed()<<mystep_mode) == FALSE)
    {
        error_handler(STSPIN820_ERROR_SET_MAX_SPEED);
    }
    if (motor->set_min_speed(motor->get_min_speed()<<mystep_mode) == FALSE)
    {
        error_handler(STSPIN820_ERROR_SET_MIN_SPEED);
    }
    if (motor->set_acceleration(motor->get_acceleration()<<mystep_mode) == FALSE)
    {
        error_handler(STSPIN820_ERROR_SET_ACCELERATION);
    }
    if (motor->set_deceleration(motor->get_deceleration()<<mystep_mode) == FALSE)
    {
        error_handler(STSPIN820_ERROR_SET_DECELERATION);
    }
    
    motor->set_home();
    /* Request to go to position 200*2^mystep_mode */
    pos = 200<<mystep_mode;
    motor->go_to(pos);
    
    /* Waiting the motor ends moving */
    motor->wait_while_active();
    
    /* Check current position */
    if (motor->get_position() != pos) 
    {
        error_handler(STSPIN820_ERROR_POSITION);
    }
    else
    {
    /* Set the current position to be the Home position */
    motor->set_home();
    }
    
    /* Waiting. */
    wait_ms(DELAY_4);
    
    //----- Automatic Full-step switch demonstration
    /* Increase max speed beyond torque boost speed threshold */
    mySpeed = (STSPIN820_CONF_PARAM_TORQUE_BOOST_TH<<(mystep_mode+1));
    if (motor->set_max_speed(mySpeed) == FALSE)
    {
        error_handler(STSPIN820_ERROR_SET_MAX_SPEED);
    }
    /* Increase acceleration */
    if (motor->set_acceleration(motor->get_acceleration()<<3) == FALSE)
    {
        error_handler(STSPIN820_ERROR_SET_ACCELERATION);
    }
    /* increase deceleration */
    if (motor->set_deceleration(motor->get_deceleration()<<3) == FALSE)
    {
        error_handler(STSPIN820_ERROR_SET_DECELERATION);
    }
    /* Request to run BACKWARD */
    motor->run(StepperMotor::BWD);
    wait_ms(DELAY_6);
    
    /* Request soft stop */
    motor->soft_stop();
    wait_ms(DELAY_6);
    
    //----- Slowest move
    
    /* Set the current position to be the Home position */
    motor->set_home();
    
    /* Set speed and acceleration at lowest values */
    motor->set_min_speed(8);
    motor->set_max_speed(8);
    motor->set_acceleration(8);
    motor->set_deceleration(8);
    
    /* Move forward device number of microsteps in a full step + 1 microstep */
    pos = (1<<mystep_mode)+1;
    motor->move(StepperMotor::FWD, pos);
    
    /* Waiting the motor ends moving */
    motor->wait_while_active();
    
    /* Check current position */
    if (motor->get_position() != pos) 
    {
        error_handler(STSPIN820_ERROR_POSITION);
    }
    else
    {
    /* Set the current position to be the Home position */
    motor->set_home();
    }
    
    /* Waiting 40 milliseconds */
    wait_ms(DELAY_1);
    
    //----- Stop with Standby mode
    
    /* Enter standby mode when motor is stopped */
    motor->set_stop_mode(STANDBY_MODE);
    
    /* Move forward 1 microstep */
    motor->move(StepperMotor::FWD, 1);
    motor->wait_while_active();
    motor->set_home();
    
    /* Waiting 40 milliseconds */
    wait_ms(DELAY_1);
    
    /* Move forward 1 microstep */
    motor->move(StepperMotor::FWD, 1);
    motor->wait_while_active();
    motor->set_home();
    
    /* Waiting 40 milliseconds */
    wait_ms(DELAY_1);
    
    /* Move forward 1 microstep */
    motor->move(StepperMotor::FWD, 1);
    motor->wait_while_active();
    motor->set_home();
    
    /* Waiting 40 milliseconds */
    wait_ms(DELAY_1);
    
    /* Move forward 1 microstep */
    motor->move(StepperMotor::FWD, 1);
    motor->wait_while_active();
    motor->set_home();
    
    //----- Stop with power bridges enabled
    
    /* Keep power bridges on when motor is stopped */
    motor->set_stop_mode(HOLD_MODE);
    
    //----- Set speed, acceleration and deceleration from stspin820_target_config.h 
    
    /* Set speed and acceleration from stspin820_target_config.h */
    motor->set_max_speed(STSPIN820_CONF_PARAM_RUNNING_SPEED);
    motor->set_min_speed(STSPIN820_CONF_PARAM_MIN_SPEED);
    motor->set_acceleration(STSPIN820_CONF_PARAM_ACC);
    motor->set_deceleration(STSPIN820_CONF_PARAM_DEC);
    
    /* Waiting. */
    wait_ms(DELAY_4);
    
    /* Infinite Loop. */
    while (true) {
        /* Request to run. */
        motor->run(motor->get_direction());
        wait_ms(DELAY_5);
        
        /* Request to soft stop. */
        motor->soft_stop();
        wait_ms(DELAY_6);
    }
}
