/* 
 *  This VL53L0X Expansion board test application performs range measurements
 *  using the onboard embedded centre sensor, in singleshot, polling mode. 
 *  Measured ranges are displayed on the on-board 4-digit display.
 *
 *  The User Blue button stops the current measurement and entire program, 
 *  releasing all resources.
 *
 *  The Reset button can be used to restart the program.  
 */
 
#include <string.h>
#include <stdio.h>
 
#include "mbed.h"
#include "x_nucleo_53l0a1.h"
 
#define VL53L0_I2C_SDA   D14 
#define VL53L0_I2C_SCL   D15 
 
static X_NUCLEO_53L0A1 *board = NULL;
VL53L0X_RangingMeasurementData_t data_sensor_centre;
OperatingMode operating_mode;
    
/* flags that handle interrupt request */
bool int_stop_measure = false;   
 
/* ISR callback function of the user blue button to stop program */
void StopMeasureIRQ(void)
{
   int_stop_measure = true;
}
 
/* On board 4 digit local display refresh */
void DisplayRefresh(VL53L0X_RangingMeasurementData_t sensor_range_data)
{   
   char str[5];
   
   if (sensor_range_data.RangeStatus == 0) // we have a valid range.
   {
      sprintf(str,"%d", sensor_range_data.RangeMilliMeter);
   }
   else
   {
      sprintf(str,"%s","----");
   }
     
   board->display->display_string(str);
}
 
/*=================================== Main ==================================
 Press the blue user button to stop measurements in progress    
=============================================================================*/
int main()
{
#if USER_BUTTON == PC_13  // we are cross compiling for Nucleo-f401 
   InterruptIn stop_button(USER_BUTTON);
   stop_button.rise(&StopMeasureIRQ);  
#endif   
 
   DevI2C *device_i2c = new DevI2C(VL53L0_I2C_SDA, VL53L0_I2C_SCL);     
   int status;
        
   /* creates the 53L0A1 expansion board singleton obj */
   board=X_NUCLEO_53L0A1::instance(device_i2c, A2, D8, D2);
    
   board->display->display_string("53L0");
   /* init the 53L0A1 expansion board with default values */
   status = board->init_board();
   if (status) printf("Failed to init board!\n\r");   
   operating_mode = range_single_shot_polling;
   /* start the measure on sensor centre */
   status = board->sensor_centre->start_measurement(operating_mode, NULL);
   if (!status)
   {
      while (1) // infinite loop. can be broken by pressing Blue (User) button.
      {
         status = board->sensor_centre->get_measurement(operating_mode, &data_sensor_centre);
         DisplayRefresh(data_sensor_centre);
         if (int_stop_measure) // Blue Button isr was triggered
         {
            status = board->sensor_centre->stop_measurement(operating_mode); // stop the measure and exit
            int_stop_measure = false;
            printf("\nProgram stopped!\n\n\r");
            break;
         }
      }
   }
   board->display->display_string("BYE");
   delete board;        
}
 
            