#include "mbed.h"
#include "x_nucleo_53l0a1.h"
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>

#define DEFAULT_STACK_SIZE 768
/* 
 * This VL53L0X Expansion board sample application performs range measurements using 
 * range_continuous_interrupt mode to generate a hardware interrupt each time a new 
 * measurement is ready to be read.
 * The application supports the centre, on-board, sensor and up two satellite boards.
 *
 * The measured range data is displayed on the on-board 4-digit LED display.
 *
 * The User Blue button switches between the currently selected sensor to display range 
 * results from.
 *
 * The Black Reset button is used to restart the program. 
 *
 * *** NOTE : By default hardlinks U10, U11, U15 & U18, on the underside of 
 *            the X-NUCELO-53L0A1 expansion board are not made/OFF.
 *            These links must be made to allow interrupts from the Satellite boards 
 *            to be received.
 *            U11 and U18 must be made/ON to allow interrupts to be received from the
 *            INT_L & INT_R positions; or
 *            U10 and U15 must be made/ON to allow interrupts to be received from the
 *            Alternate INT_L & INT_R positions. 
 *            The X_NUCLEO_53L0A1 firmware library defaults to use the INT_L/INT_R 
 *            positions. 
 *            INT_L is available on expansion board Arduino Connector CN5, pin 1 as D8.
 *            Alternate INT_L is on CN5 Connector pin 2 as D9.
 *            INT_R is available on expansion board Arduino Connector CN9, pin 3 as D2.
 *            Alternate INT_R is on CN9 Connector pin 5 as D4. 
 *            The pinouts are shown here : https://developer.mbed.org/components/X-NUCLEO-53L0A1/
 *   
 */

#define VL53L0_I2C_SDA   D14 
#define VL53L0_I2C_SCL   D15 

#if USER_BUTTON==PC_13  // we are cross compiling for Nucleo-64s
   InterruptIn stop_button (USER_BUTTON);
#endif   

static X_NUCLEO_53L0A1 *board=NULL;
VL53L0X_RangingMeasurementData_t data_sensor;
OperatingMode operating_mode;

/* current displayed sensor change IRQ */
volatile bool switchChanged = false;

/* interrupt requests */
volatile bool centerSensor = false;
volatile bool leftSensor = false;
volatile bool rightSensor = false; 

/* Current sensor number*/
volatile int currentSensor = 0;
/* Installed sensors count */ 
int sensorCnt = 0; 

/* installed sensors prefixes */
char installedSensors[3];

/* ISR callback function of the sensor_centre */
void SensorCenterIRQ(void)
{
   centerSensor = true;
   board->sensor_centre->disable_interrupt_measure_detection_irq();
}   

void SensorLeftIRQ(void)
{
   leftSensor = true;
   board->sensor_left->disable_interrupt_measure_detection_irq();
} 

void SensorRightIRQ(void)
{    
   rightSensor = true;  
   board->sensor_right->disable_interrupt_measure_detection_irq();
} 

/* ISR callback function of the user blue button to switch measuring sensor. */
void SwitchMeasuringSensorIRQ(void)
{
    stop_button.disable_irq();
    switchChanged = true;
}


/* On board 4 digit local display refresh */
void RefreshDisplay(const VL53L0X_RangingMeasurementData_t &data, char prefix)
{
    static char str[5];
    if (data_sensor.RangeStatus == 0) // we have a valid range.
    {
        sprintf(str, "%c%3d", prefix ,data.RangeMilliMeter);
        board->display->display_string(str);
    }
    else
    {
        sprintf(str, "%c%s", prefix, "---");
        board->display->display_string(str);
    }
}

inline void MeasureSensors(OperatingMode op_mode)
{
    bool current = false;
    if (centerSensor)
    {
        centerSensor = false;
        board->sensor_centre->handle_irq(op_mode, &data_sensor);
        current = (currentSensor == 0);
        if (current)
        {
            RefreshDisplay(data_sensor, 'C');
        }
        if (data_sensor.RangeStatus == 0) // we have a valid range.
        {
            printf("%c %4d; ", 'C', data_sensor.RangeMilliMeter);
        }
    }
    if (leftSensor)
    {
        leftSensor = false;
        board->sensor_left->handle_irq(op_mode, &data_sensor);
        current = (installedSensors[currentSensor] == 'L');
        if (current)
        {
            RefreshDisplay(data_sensor, 'L');
        }
        if (data_sensor.RangeStatus == 0) // we have a valid range.
        {
            printf("%c %4d; ", 'L', data_sensor.RangeMilliMeter);
        }

    }
    if (rightSensor)
    {
        rightSensor = false;
        board->sensor_right->handle_irq(op_mode, &data_sensor);
        current = (installedSensors[currentSensor] == 'R');        
        if (current)
        {
            RefreshDisplay(data_sensor, 'R');
        }
        if (data_sensor.RangeStatus == 0) // we have a valid range.
        {
            printf("%c %4d; ", 'R', data_sensor.RangeMilliMeter);
        }

    }
}

int InitSensorsArray()
{
    int status = 1;
    sensorCnt = 0;
    /* start the measure on the center sensor */
    if (NULL != board->sensor_centre)
    {
        installedSensors[sensorCnt] = 'C';
        status=board->sensor_centre->stop_measurement(operating_mode);
        status=board->sensor_centre->start_measurement(operating_mode, &SensorCenterIRQ); 
        ++sensorCnt;
    }
    /* start the measure on the left sensor */
    if (NULL != board->sensor_left)
    {
        installedSensors[sensorCnt] = 'L';
        status=board->sensor_left->stop_measurement(operating_mode);
        status=board->sensor_left->start_measurement(operating_mode, &SensorLeftIRQ);
        ++sensorCnt;
    }
    /* start the measure on the right sensor */    
    if (NULL != board->sensor_right)
    {
        installedSensors[sensorCnt] = 'R';
        status=board->sensor_right->stop_measurement(operating_mode);
        status=board->sensor_right->start_measurement(operating_mode, &SensorRightIRQ);
        ++sensorCnt;
    }
    currentSensor = 0;
    return status;
}


void RangeMeasure(DevI2C *device_i2c) {
   int status;

   /* creates the 53L0A1 expansion board singleton obj */
   board=X_NUCLEO_53L0A1::instance(device_i2c, A2, D8, D2);
   //board=X_NUCLEO_53L0A1::Instance(device_i2c, A2, D9, D4); // Alternate INT_L/INT_R settings.
    
   board->display->display_string("53L0");
   
   operating_mode=range_continuous_interrupt;
   
   /* init the 53L0A1 expansion board with default values */
   status=board->init_board();
   
   if(status)
   {
        printf("Failed to init board!\n\r");   
   }
   else 
   {
        status = InitSensorsArray();
   }
   
   if(!status)
   {
     printf ("\r\nEntering loop mode\r\n");
     while (true)
     { 
        MeasureSensors(operating_mode);
        if (switchChanged)
        {
            ++currentSensor;
            if (currentSensor == sensorCnt)
                currentSensor = 0;
            
            printf("Sensor changed to %c\r\n",installedSensors[currentSensor]);   
            switchChanged = false;
            stop_button.enable_irq(); 
        }
     }
   }
   delete board;        
}    

/*=================================== Main ==================================
 Press the blue user button to switch the displayed sensor.       
=============================================================================*/
int main()
{   
#if USER_BUTTON==PC_13  // we are cross compiling for Nucleo-f401 
   stop_button.rise (&SwitchMeasuringSensorIRQ);  
   stop_button.enable_irq();
#endif   
   DevI2C *device_i2c =new DevI2C(VL53L0_I2C_SDA, VL53L0_I2C_SCL);        
   RangeMeasure(device_i2c);  // start continuous measures
}



