#include <jsonlite.h>
#include "M2XStreamClient.h"
#include "x_cube_mems.h"

#include "mbed.h"
#include "GPS.h"    //GPS

//------------------------------------------------------------------------------------
// You need to configure these cellular modem / SIM parameters.
// These parameters are ignored for LISA-C200 variants and can be left NULL.
//------------------------------------------------------------------------------------
#include "MDM.h"
//! Set your secret SIM pin here (e.g. "1234"). Check your SIM manual.
#define SIMPIN      NULL
/*! The APN of your network operator SIM, sometimes it is "internet" check your 
    contract with the network operator. You can also try to look-up your settings in 
    google: https://www.google.de/search?q=APN+list */
#define APN         NULL
//! Set the user name for your APN, or NULL if not needed
#define USERNAME    NULL
//! Set the password for your APN, or NULL if not needed
#define PASSWORD    NULL 
//------------------------------------------------------------------------------------

char feedId[] = "0c76bf26149c9c01d8eec11553d1a6f2"; // Feed you want to post to
char m2xKey[] = "76ee6b8414970d7ade8c75829d0cf879"; // Your M2X access key

char name[] = "<location name>"; // Name of current location of datasource
double latitude = 33.007872;
double longitude = -96.751614; // You can also read those values from a GPS
double elevation = 697.00;
bool location_valid = false;
volatile float TEMPERATURE_Value_C, TEMPERATURE_Value_F;
volatile float HUMIDITY_Value;
volatile float PRESSURE_Value;
volatile AxesRaw_TypeDef MAG_Value;
volatile AxesRaw_TypeDef ACC_Value;
volatile AxesRaw_TypeDef GYR_Value;
Client client;
M2XStreamClient m2xClient(&client, m2xKey);

void on_data_point_found(const char* at, const char* value, int index, void* context) {
  printf("Found a data point, index: %d\r\n", index);
  printf("At: %s Value: %s\r\n", at, value);
}

void on_location_found(const char* name,
                       double latitude,
                       double longitude,
                       double elevation,
                       const char* timestamp,
                       int index,
                       void* context) {
  printf("Found a location, index: %d\r\n", index);
  printf("Name: %s  Latitude: %lf  Longitude: %lf\r\n", name, latitude, longitude);
  printf("Elevation: %lf  Timestamp: %s\r\n", elevation, timestamp);
}

int main() {
  uint8_t hts221_id;
  MDMSerial mdm(D8,D2); 
  GPSI2C gps;
  //mdm.setDebug(4); // enable this for debugging issues 
  if (!mdm.connect(SIMPIN, APN,USERNAME,PASSWORD))
    return -1;
 
  char buf[256];
  
  printf("M2X MEMS U-Blox Cellular Demo \r\n");

  static X_CUBE_MEMS *mems_expansion_board = X_CUBE_MEMS::Instance();
    
  hts221_id = mems_expansion_board->hts221.ReadID();
  printf("HTS221_ID = 0x%x\n\t\r", hts221_id);
    
  Timer tmr;
  tmr.reset();
  tmr.start();
  while (true) {
    int ret;
    // extract the location information from the GPS NMEA data
    while ((ret = gps.getMessage(buf, sizeof(buf))) > 0) {
        int len = LENGTH(ret);
        if ((PROTOCOL(ret) == GPSParser::NMEA) && (len > 6)) {
            if (!strncmp("$GPGGA", buf, 6)) {
                char ch;
                if (gps.getNmeaAngle(2,buf,len,latitude) && 
                    gps.getNmeaAngle(4,buf,len,longitude) && 
                    gps.getNmeaItem(6,buf,len,ch) &&
                    gps.getNmeaItem(9,buf,len,elevation)) {
                   printf("GPS Location: %.5f %.5f %.1f %c\r\n", latitude, longitude, elevation, ch); 
                   location_valid = ch == '1' || ch == '2' || ch == '6';
                }
            }
        }
    }
    
    if (tmr.read_ms() > 1000) {
        tmr.reset();
        tmr.start();
        int response;

        /* Read sensors */        
        mems_expansion_board->hts221.GetTemperature((float *)&TEMPERATURE_Value_C);
        mems_expansion_board->hts221.GetHumidity((float *)&HUMIDITY_Value);
        mems_expansion_board->lps25h.GetPressure((float *)&PRESSURE_Value);
        mems_expansion_board->lis3mdl.GetAxes((AxesRaw_TypeDef *)&MAG_Value);
        mems_expansion_board->lsm6ds0.Acc_GetAxes((AxesRaw_TypeDef *)&ACC_Value);
        mems_expansion_board->lsm6ds0.Gyro_GetAxes((AxesRaw_TypeDef *)&GYR_Value);
        
        /* Convert temperature to degrees Farhenheit. */
        TEMPERATURE_Value_F = (1.8f * TEMPERATURE_Value_C) + 32.0f;        
        
        printf("Temperature:\t\t %f C / %f F\r\n", TEMPERATURE_Value_C, TEMPERATURE_Value_F);
        printf("Humidity:\t\t %f%%\r\n", HUMIDITY_Value);
        printf("Pressure:\t\t %f hPa\r\n", PRESSURE_Value); 
        printf("Magnetometer (mGauss):\t X: %d, Y: %d, Z: %d\r\n", MAG_Value.AXIS_X, MAG_Value.AXIS_Y, MAG_Value.AXIS_Z);
        printf("Accelerometer (mg):\t X: %d, Y: %d, Z: %d\r\n", ACC_Value.AXIS_X, ACC_Value.AXIS_Y, ACC_Value.AXIS_Z);
        printf("Gyroscope (mdps):\t X: %d, Y: %d, Z: %d\r\n", GYR_Value.AXIS_X, GYR_Value.AXIS_Y, GYR_Value.AXIS_Z);
        printf("\r\n");        
        
        
        MDMParser::NetStatus status;
        if (mdm.checkNetStatus(&status)) {            
            /* Post Temperature to M2X */                            
            response = m2xClient.updateStreamValue(feedId, "temperature", TEMPERATURE_Value_F);
            printf("Temperature updateStreamValue response code: %d\r\n", response);
            if (response == -1) 
            {
                printf("Temperature data update error\n");
            }        
    
            /* Post Humidity to M2X */                 
            response = m2xClient.updateStreamValue(feedId, "humidity", HUMIDITY_Value);
            printf("Humidity updateStreamValue response code: %d\r\n", response);
            if (response == -1) 
            {
                printf("Humidity data update error\n");
            }        
            
            /* Post acceleration (x-axis) to the m2x stream. */
            response = m2xClient.updateStreamValue(feedId, "acceleration", ACC_Value.AXIS_X);
            printf("Acceleration updateStreamValue response code: %d\r\n", response);            
            if (response == -1) 
            {
                printf("Acceleration data update error\n");
            }
                           
        }
        
//#define READING
#ifdef READING
        // read signal strength
        response = m2xClient.fetchValues(feedId, "rssi", on_data_point_found, NULL);
        printf("Fetch response code: %d\r\n", response);
        if (response == -1) while (true) ;
#endif    
        // update location
        if (location_valid) {
            response = m2xClient.updateLocation(feedId, name, latitude, longitude, elevation);
            printf("updateLocation response code: %d\r\n", response);
            if (response == -1) while (true) ;
        }
#ifdef READING
        // read location
        response = m2xClient.readLocation(feedId, on_location_found, NULL);
        printf("readLocation response code: %d\r\n", response);
        if (response == -1) while (true) ;
#endif
        wait(30); // 30 s
        printf("\r\n");
    }
    else {
        delay(100);
    }
  }

  mdm.disconnect();
  mdm.powerOff();
}