#include <jsonlite.h>
#include "M2XStreamClient.h"
#include "mbed.h"
#include "WIZnetInterface.h"
#include "x_cube_mems.h"

#define ST_NUCLEO
char feedId[] = "0c76bf26149c9c01d8eec11553d1a6f2"; // Feed you want to post to
char m2xKey[] = "76ee6b8414970d7ade8c75829d0cf879"; // Your M2X access key
char tempStreamName[] = "temperature"; // Stream you want to post to M2X
char humidityStreamName[] = "humidity"; // Stream you want to post to M2X

char name[] = "Austin"; // Name of current location of datasource
double latitude = 30.3748076;
double longitude = -97.7386896; // You can also read those values from a GPS
double elevation = 300.00;
volatile float TEMPERATURE_Value, TEMPERATURE_Value_f;
volatile float HUMIDITY_Value;
volatile float PRESSURE_Value;
volatile AxesRaw_TypeDef MAG_Value;
volatile AxesRaw_TypeDef ACC_Value;
volatile AxesRaw_TypeDef GYR_Value;

PwmOut mypwm(PWM_OUT);

/**
 * Configure the SPI interfac to the ethernet module
 * D11 - MOSI pin
 * D12 - MISO pin
 * D13 - SCK pin
 * D10 - SEL pin
 * NC - Reset pin; use D5 otherwise the shield might get into reset loop
 */


    
SPI spi(PA_7, PA_6, PA_5); // mosi, miso, sclk
WIZnetInterface eth(&spi, PB_6, PA_10); // spi, cs, reset

/* Instantiate the M2X Stream Client */
Client client;
M2XStreamClient m2xClient(&client, m2xKey);


/* Call back function for reading back data point data from M2X */
void on_data_point_found(const char* at, const char* value, int index, void* context, int type)
{
    printf("Found a data point, index: %d\r\n", index);
    printf("At: %s Value: %s\r\n", at, value);
}

/* Call back function for reading back location data from M2X */
void on_location_found(const char* name,
                       double latitude,
                       double longitude,
                       double elevation,
                       const char* timestamp,
                       int index,
                       void* context)
{
    printf("Found a location, index: %d\r\n", index);
    printf("Name: %s  Latitude: %lf  Longitude: %lf\r\n", name, latitude, longitude);
    printf("Elevation: %lf  Timestamp: %s\r\n", elevation, timestamp);
}

int main()
{
    uint8_t mac[6];
    int response;
    uint8_t hts221_id;
    
    static X_CUBE_MEMS *mems_expansion_board = X_CUBE_MEMS::Instance();
    
    hts221_id = mems_expansion_board->hts221.ReadID();
    
    printf("HTS221_ID = 0x%x\n\t\r", hts221_id);
    
    mypwm.period_ms(500);
    mypwm.pulsewidth_ms(250);
    /* Have mbed assign the mac address */
    mbed_mac_address((char *)mac);     

    printf("Start...\n");
    wait_ms(3000);    

    /* Initialize ethernet interface */
    int ret = eth.init(mac); //Use DHCP


    if (!ret) {
        printf("Initialized, MAC: %s\n", eth.getMACAddress());
    } else {
        printf("Error eth.init() - ret = %d\n", ret);
        return -1;
    }


    /* Get IP address */
    while (1) {
        printf(">>> Get IP address...\n");
        ret = eth.connect(); // Connect to network

        printf("ret: %d\n",ret);
        if (ret < 0) {
            printf(">>> Could not connect to network! Retrying ...\n");
            wait_ms(3000);
            printf("past this point...\n");
        } else {
            break;
        }
    }

    if (!ret) {
        printf("IP: %s, MASK: %s, GW: %s\n",
                  eth.getIPAddress(), eth.getNetworkMask(), eth.getGateway());
    } else {
        printf("Error eth.connect() - ret = %d\n", ret);
        return -1;
    }

    mypwm.pulsewidth_ms(500);
 
    /* Main loop */
    while (true) { 
        /* Wait 5 secs and then loop */
        delay(2500);
        mypwm.pulsewidth_ms(0);      
              
        /* Read sensors */        
        mems_expansion_board->hts221.GetTemperature((float *)&TEMPERATURE_Value);
        mems_expansion_board->hts221.GetHumidity((float *)&HUMIDITY_Value);
        //mems_expansion_board->lps25h.GetPressure((float *)&PRESSURE_Value);
        //mems_expansion_board->lis3mdl.GetAxes((AxesRaw_TypeDef *)&MAG_Value);
        //mems_expansion_board->lsm6ds0.Acc_GetAxes((AxesRaw_TypeDef *)&ACC_Value);
        //mems_expansion_board->lsm6ds0.Gyro_GetAxes((AxesRaw_TypeDef *)&GYR_Value);
                              
        delay(2500);
        mypwm.pulsewidth_ms(500);   

        /* Post temperature to M2X */
        TEMPERATURE_Value_f =(1.8 * TEMPERATURE_Value) +  32.0;                        
        printf("TEMP: %f C  %f F \r\n", TEMPERATURE_Value, TEMPERATURE_Value_f); 
                
        response = m2xClient.updateStreamValue(feedId, tempStreamName, TEMPERATURE_Value_f);       
        printf("Post response code: %d\r\n", response);
        if (response == -1) 
        {
            mypwm.pulsewidth_ms(250);                
            printf("Temperature data transmit post error\n");
        }
        
        /* Post humidity to M2X */
        printf("HUMIDITY: %f \r\n", HUMIDITY_Value);         
        response = m2xClient.updateStreamValue(feedId, humidityStreamName, HUMIDITY_Value); 
        printf("Post response code: %d\r\n", response);
        if (response == -1) 
        {
            mypwm.pulsewidth_ms(250);                
            printf("Humidity data transmit post error\n");
        }
                        
        /* Update location data */
        response = m2xClient.updateLocation(feedId, name, latitude, longitude, elevation);
        printf("updateLocation response code: %d\r\n", response);
        if (response == -1) 
        {
            mypwm.pulsewidth_ms(250);        
            printf("Location data transmit post error\n");
        }

        printf("\r\n");
        

    }
}
