#include <jsonlite.h>
#include "M2XStreamClient.h"

#include "mbed.h"
#include "WIZnetInterface.h"
#include "LM75B.h"  //I2C Temperature Sensor

#define ST_NUCLEO
char feedId[] = "fe08906d21a70b05241234077386e041"; // Feed you want to post to
char m2xKey[] = "ca9c1e4db697886906de09c701879b19"; // Your M2X access key
char streamName[] = "temperature"; // Stream you want to post to

char name[] = "Dallas"; // Name of current location of datasource
double latitude = 33.007872;
double longitude = -96.751614; // You can also read those values from a GPS
double elevation = 697.00;

PwmOut mypwm(PWM_OUT);
AnalogIn temp_sensor(A0);  

/**
 * Configure the SPI interfac to the ethernet module
 * D11 - MOSI pin
 * D12 - MISO pin
 * D13 - SCK pin
 * D10 - SEL pin
 * NC - Reset pin; use D5 otherwise the shield might get into reset loop
 */

SPI spi(PA_7, PA_6, PA_5); // mosi, miso, sclk
WIZnetInterface eth(&spi, PB_6, PA_10); // spi, cs, reset

/* Instantiate the M2X Stream Client */
Client client;
M2XStreamClient m2xClient(&client, m2xKey);


/* Call back function for reading back data point data from M2X */
void on_data_point_found(const char* at, const char* value, int index, void* context, int type)
{
    printf("Found a data point, index: %d\r\n", index);
    printf("At: %s Value: %s\r\n", at, value);
}

/* Call back function for reading back location data from M2X */
void on_location_found(const char* name,
                       double latitude,
                       double longitude,
                       double elevation,
                       const char* timestamp,
                       int index,
                       void* context)
{
    printf("Found a location, index: %d\r\n", index);
    printf("Name: %s  Latitude: %lf  Longitude: %lf\r\n", name, latitude, longitude);
    printf("Elevation: %lf  Timestamp: %s\r\n", elevation, timestamp);
}

int main()
{
    uint8_t mac[6];
    int adc_scale = 65536; //4096; 
    int B = 3975;    
    float resistance; 
    float temperature;
    float temperature_f;  
    char amb_temp[6];    

    mypwm.period_ms(500);
    mypwm.pulsewidth_ms(250);
    /* Have mbed assign the mac address */
    mbed_mac_address((char *)mac);     

    printf("Start...\n");
    wait_ms(3000);    

    /* Initialize ethernet interface */
    int ret = eth.init(mac); //Use DHCP


    if (!ret) {
        printf("Initialized, MAC: %s\n", eth.getMACAddress());
    } else {
        printf("Error eth.init() - ret = %d\n", ret);
        return -1;
    }


    /* Get IP address */
    while (1) {
        printf(">>> Get IP address...\n");
        ret = eth.connect(); // Connect to network

        printf("ret: %d\n",ret);
        if (ret < 0) {
            printf(">>> Could not connect to network! Retrying ...\n");
            wait_ms(3000);
            printf("past this point...\n");
        } else {
            break;
        }
    }

    if (!ret) {
        printf("IP: %s, MASK: %s, GW: %s\n",
                  eth.getIPAddress(), eth.getNetworkMask(), eth.getGateway());
    } else {
        printf("Error eth.connect() - ret = %d\n", ret);
        return -1;
    }



    mypwm.pulsewidth_ms(500);
 
    /* Main loop */
    while (true) {


 
        /* Wait 5 secs and then loop */
        delay(2500);
        mypwm.pulsewidth_ms(0);      
              
        /* Read ADC value from analog sensor */
        uint16_t a = temp_sensor.read_u16();
               
        /* Calculate the temperature in Fareheight and Celsius */
        resistance = (float)(adc_scale-a)*10000/a; //get the resistance of the sensor;              
        temperature = 1/(log(resistance/10000)/B+1/298.15)-273.15;  //convert to temperature via datasheet ;        
        temperature_f =(1.8 * temperature) +  32.0;
        sprintf(amb_temp, "%0.2f", temperature_f);  
        
        printf("Temp Sensor Analog Reading is 0x%X = %d  \r\n", a, a);         
        printf("Current Temperature: %f C  %f F \r\n", temperature, temperature_f); 

        delay(2500);
        mypwm.pulsewidth_ms(500);   
        
        /* Post temperature to M2X site */
        int response = m2xClient.put(feedId, streamName, amb_temp);
        printf("Post response code: %d\r\n", response);
        if (response == -1) 
        {
            mypwm.pulsewidth_ms(250);                
            printf("Temperature data transmit post error\n");
        }
        /* Update location data */
        response = m2xClient.updateLocation(feedId, name, latitude, longitude, elevation);
        printf("updateLocation response code: %d\r\n", response);
        if (response == -1) 
        {
            mypwm.pulsewidth_ms(250);        
            printf("Location data transmit post error\n");
        }

        printf("\r\n");
        

    }
}
