#include "mbed.h"
// include target specific defines
#include "target_config.h"
// include X-CUBE-MEMS1 Library
#include "x_cube_mems.h"
// include the Wiconnect Host Library API header
#include "Wiconnect.h"
// include M2X Library
#include "M2XStreamClient.h"

/**
  * Connect the ACKme WiFi module directly to the Nucleo board.
  * Connect the Nucleo sensor module on top of the ACKme WiFi module.
  */

/**
  * Hyperterminal configuration
  * 9600 bauds, 8-bit data, no parity
  */

/**
  * This is the name of your WiFi network.
  * Look for this name in your WiFi settings.
  * (e.g. your phone's list of WiFi networks in the WiFi settings menu.)
  * tip: add double-quotes around SSID to add spaces to name.
  */
#define NETWORK_SSID "\"<YOUR NETWORK NAME HERE>\""

/**
  * This is the password of your WiFi network.
  * Leave as empty string (e.g "") to connect to OPEN network.
  */
#define NETWORK_PASSWORD "\"<YOUR NETWORK PASSWORD HERE>\""

const char key[] = "123ad8ee16ef56dfafd0c42a3a3ef109";      // Replace with your M2X API key
const char feed[] = "d3ffd3ab9f659943e2302ba232acf198";     // Replace with your blueprint Feed ID
const char tempStream[] = "temperature";                    // Replace with your stream name  
const char humStream[] = "humidity";                        // Replace with your stream name  
const char accStream[] = "acceleration";                    // Replace with your stream name  

char name[] = "redmond_st_office";  // Name of current location of datasource
double latitude = 47.633889;        // You can also read those values from a GPS
double longitude = -122.138611;
double elevation = 97.46;

/* Instantiate the serial console. */
Serial pc(SERIAL_TX, SERIAL_RX);
 
int main()
{
    /* Set the console terminal to 9600 bps. */
    pc.baud(CONSOLE_BAUD);
    
    /* Instantiate the X-CUBE-MEMS Library. */
    static X_CUBE_MEMS *mems_expansion_board = X_CUBE_MEMS::Instance();
    
    /* Read and output the humidity sensor id to confirm communication. */
    uint8_t hts221_id = mems_expansion_board->hts221.ReadID();
    
    pc.printf("HTS221_ID = 0x%x\n\t\r", hts221_id);
    
    /**
      * WIFI Setup
      */
    
    /* Setup wiconnect serial interface configuration. */
    
    /**
      * Here we only specify the rx buffer size and not rx buffer pointer, this means
      * the serial RX buffer will be dynamically allocated.
      */
    SerialConfig serialConfig(WICONNECT_RX_PIN, WICONNECT_TX_PIN, 256, NULL);
    
    /* Instantiate WiConnect Library. */
    
    /**
      * Here we only specify the buffer size and not buffer pointer, this means
      * the internal buffer will be dynamically allocated.
      */
    Wiconnect wiconnect(serialConfig, 256, NULL, WICONNECT_RESET_PIN);
    
    /* Initiate Communication with WiFi Module. */
    pc.printf("Initializing WiConnect Library...\r\n");
    
    WiconnectResult result;
    
    if (WICONNECT_FAILED(result, wiconnect.init(true)))
    {
        if (result == WICONNECT_FIRMWARE_OUTDATED)
        {
            pc.printf("** The WiFi firmware is not supported. Run the ota example to update the firmware:\r\n");
            pc.printf("https://developer.mbed.org/teams/ACKme/code/wiconnect-ota_example\r\n\r\n");
        }
        else
        {
            pc.printf("Failed to initialize communication with WiFi module!\r\n"
                      "Make sure the wires are connected correctly\r\n");
        }
        
        return -1;
    }
    
    wait(5);
    
    /* Manually connected to the specified network (to ensure SDK backward compatibility). */
    pc.printf("Setting network SSID: %s\r\n", NETWORK_SSID);
    
    if (WICONNECT_FAILED(result, wiconnect.setSetting("wlan.ssid", NETWORK_SSID)))
    {
        pc.printf("Failed to set wlan.ssid setting\r\n");
        return -1;
    }
    
    pc.printf("Setting network password\r\n");
    
    if (WICONNECT_FAILED(result, wiconnect.setSetting("wlan.passkey", NETWORK_PASSWORD)))
    {
        pc.printf("Failed to set wlan.passkey setting\r\n");
        return -1;
    }

    pc.printf("Saving settings to Non-volatile Memory\r\n");
    
    if (WICONNECT_FAILED(result, wiconnect.saveSettings()))
    {
        pc.printf("Failed save settings\r\n");
        return -1;
    }
    
    NetworkStatus status;
    
    result = wiconnect.getNetworkStatus(&status);
    
    switch (status)
    {
        case NETWORK_STATUS_DOWN:
        {
            pc.printf("NETWORK_STATUS_DOWN\r\n");
            
            return -1;
            
            break;
        }
        case NETWORK_STATUS_WIFI_ONLY:
        {
            pc.printf("NETWORK_STATUS_WIFI_ONLY\r\n");
            break;
        }
        case NETWORK_STATUS_UP:
        {
            pc.printf("NETWORK_STATUS_UP\r\n");
            break;
        }
        case NETWORK_STATUS_JOINING:
        {
            pc.printf("NETWORK_STATUS_JOINING\r\n");
            break;
        }
        default:
        {
            pc.printf("UNKNOWN\r\n");
            break;
        }
    }
        
    pc.printf("IP Address: %s\r\n", wiconnect.getIpAddress());
    pc.printf("Network joined!\r\n");
    
    /**
      * M2X Setup
      */
    
    /* Instantiate the M2X Stream Client. */
    Client client;    
    M2XStreamClient m2xClient(&client, key);
    
    /* Update device location. */
    int m2x_response = m2xClient.updateLocation(feed, name, latitude, longitude, elevation);
    
    pc.printf("updateLocation response code: %d\r\n", m2x_response);
    
    if (m2x_response == -1) 
    {
        printf("Location data update error\n");
    }

    /* Main loop */
    while(1)
    {
        volatile float TEMPERATURE_Value_C;
        volatile float TEMPERATURE_Value_F;
        volatile float HUMIDITY_Value;
        volatile float PRESSURE_Value;
        volatile AxesRaw_TypeDef MAG_Value;
        volatile AxesRaw_TypeDef ACC_Value;
        volatile AxesRaw_TypeDef GYR_Value;
        
        /* Update sensors.  */
        mems_expansion_board->hts221.GetTemperature((float *)&TEMPERATURE_Value_C);
        mems_expansion_board->hts221.GetHumidity((float *)&HUMIDITY_Value);
        mems_expansion_board->lps25h.GetPressure((float *)&PRESSURE_Value);
        mems_expansion_board->lis3mdl.GetAxes((AxesRaw_TypeDef *)&MAG_Value);
        mems_expansion_board->lsm6ds0.Acc_GetAxes((AxesRaw_TypeDef *)&ACC_Value);
        mems_expansion_board->lsm6ds0.Gyro_GetAxes((AxesRaw_TypeDef *)&GYR_Value);     
        
        /* Convert temperature to degrees Farhenheit. */
        TEMPERATURE_Value_F = (1.8f * TEMPERATURE_Value_C) + 32.0f;
        
        /* Output sensor data. */
        pc.printf("Temperature:\t\t %f C / %f F\r\n", TEMPERATURE_Value_C, TEMPERATURE_Value_F);
        pc.printf("Humidity:\t\t %f%%\r\n", HUMIDITY_Value);
        pc.printf("Pressure:\t\t %f hPa\r\n", PRESSURE_Value); 
        pc.printf("Magnetometer (mGauss):\t X: %d, Y: %d, Z: %d\r\n", MAG_Value.AXIS_X, MAG_Value.AXIS_Y, MAG_Value.AXIS_Z);
        pc.printf("Accelerometer (mg):\t X: %d, Y: %d, Z: %d\r\n", ACC_Value.AXIS_X, ACC_Value.AXIS_Y, ACC_Value.AXIS_Z);
        pc.printf("Gyroscope (mdps):\t X: %d, Y: %d, Z: %d\r\n", GYR_Value.AXIS_X, GYR_Value.AXIS_Y, GYR_Value.AXIS_Z);
        pc.printf("\r\n");
        
        /* Post temperature to the m2x stream. */
        m2x_response = m2xClient.updateStreamValue(feed, tempStream, TEMPERATURE_Value_F);
        
        pc.printf("Temperature updateStreamValue response code: %d\r\n", m2x_response);
        
        if (m2x_response == -1) 
        {
            pc.printf("Temperature data update error\n");
        }
        
        /* Post humidity to the m2x stream. */
        m2x_response = m2xClient.updateStreamValue(feed, humStream, HUMIDITY_Value);
        
        pc.printf("Humidity updateStreamValue response code: %d\r\n", m2x_response);
        
        if (m2x_response == -1) 
        {
            pc.printf("Humidity data update error\n");
        }
        
        /* Post acceleration (x-axis) to the m2x stream. */
        m2x_response = m2xClient.updateStreamValue(feed, accStream, ACC_Value.AXIS_X);
        
        pc.printf("Acceleration updateStreamValue response code: %d\r\n", m2x_response);
        
        if (m2x_response == -1) 
        {
            pc.printf("Acceleration data update error\n");
        }
        
        pc.printf("\n");
        
        wait(30); // 30 s
    }
}
 