/*-----------------------------------------------------------

 Demo usata nel Corso STM32 NUCLEO Mbed OS - Novembre 2021

-----------------------------------------------------------
*/
/*
 *  This VL53L1X Expansion board test application performs range measurements
 *  using the onboard embedded centre sensor, and 2 Satellite boards.
 *
 *  Measured ranges are ouput on the Serial Port, running at 9600 baud.
 *
 *  The Black Reset button is used to restart the program.
 *
 *  *** NOTE : By default hardlinks U10, U11, U15 & U18, on the underside of
 *            the X-NUCELO-53L0A1 expansion board are not made/OFF.
 *            These links must be made to allow interrupts from the Satellite boards
 *            to be received.
 *            U11 and U18 must be made/ON to allow interrupts to be received from the
 *            INT_L & INT_R positions; or
 *            U10 and U15 must be made/ON to allow interrupts to be received from the
 *            Alternate INT_L & INT_R positions.
 *            The X_NUCLEO_53L1A1 firmware library defaults to use the INT_L/INT_R
 *            positions.
 *            INT_L is available on expansion board Arduino Connector CN5, pin 1 as D9.
 *            Alternate INT_L is on CN5 Connector pin 2 as D8.
 *            INT_R is available on expansion board Arduino Connector CN9, pin 3 as D4.
 *            Alternate INT_R is on CN9 Connector pin 5 as D2.
 *            The pinouts are shown here : https://developer.mbed.org/components/X-NUCLEO-53L1A1/
 */

#include <stdio.h>

#include "mbed.h"
#include "XNucleo53L1A1.h"
#include "VL53L1X_I2C.h"

#define VL53L1_I2C_SDA   D14
#define VL53L1_I2C_SCL   D15

#define DISPLAY_CENTRE false
#define DISPLAY_LEFT   true
#define DISPLAY_RIGHT  false

static XNucleo53L1A1 *board=NULL;



/*=================================== Main ==================================
=============================================================================*/
int main()
{
    int status = 0;
    uint8_t ready_centre = 0;
    uint8_t ready_left = 0;
    uint8_t ready_right = 0;
    uint16_t distance_centre = 0;
    uint16_t distance_left = 0;
    uint16_t distance_right = 0;

    printf("Hello world!\r\n");

    VL53L1X_DevI2C *dev_I2C = new VL53L1X_DevI2C(VL53L1_I2C_SDA, VL53L1_I2C_SCL);

    printf("I2C device created!\r\n");

    /* creates the 53L1A1 expansion board singleton obj */
    //board = XNucleo53L1A1::instance(dev_I2C, A2, D9, D2);    /* original code */
    board = XNucleo53L1A1::instance(dev_I2C, A2, PB_13, PB_14);
    printf("board created!\r\n");

    /* init the 53L1A1 expansion board with default values */
    status = board->init_board();
    if (status) {
        printf("Failed to init board!\r\n");
        return status;
    }

    printf("board initiated! - %d\r\n", status);

    /* Start ranging on the centre sensor */
    if (board->sensor_centre != NULL) {
        status = board->sensor_centre->vl53l1x_start_ranging();
        if (status != 0) {
            printf("Centre sensor failed to start ranging!\r\n");
            return status;
        }
    }

    /* Start ranging on the left sensor */
    if (board->sensor_left != NULL) {
        status = board->sensor_left->vl53l1x_start_ranging();
        if (status != 0) {
            printf("Left sensor failed to start ranging!\r\n");
            return status;
        }
    }

    /* Start ranging on the right sensor */
    if (board->sensor_right != NULL) {
        status = board->sensor_right->vl53l1x_start_ranging();
        if (status != 0) {
            printf("Right sensor failed to start ranging!\r\n");
            return status;
        }
    }

    /* Ranging loop
     * Checks each sensor for data ready
     */
    while (1)
    {
        if (board->sensor_centre != NULL) {
            board->sensor_centre->vl53l1x_check_for_data_ready(&ready_centre);
        }
        if (board->sensor_left != NULL) {
            board->sensor_left->vl53l1x_check_for_data_ready(&ready_left);
        }
        if (board->sensor_right != NULL) {
            board->sensor_right->vl53l1x_check_for_data_ready(&ready_right);
        }
        if (ready_centre) {
            board->sensor_centre->vl53l1x_get_range_status(&ready_centre);
            board->sensor_centre->vl53l1x_get_distance(&distance_centre);
        }
        if (ready_left) {
            board->sensor_left->vl53l1x_get_range_status(&ready_left);
            board->sensor_left->vl53l1x_get_distance(&distance_left);
        }
        if (ready_right) {
            board->sensor_right->vl53l1x_get_range_status(&ready_right);
            board->sensor_right->vl53l1x_get_distance(&distance_right);
        }
        #if DISPLAY_CENTRE
        if (board->sensor_centre != NULL) {
            printf("Distance read by centre sensor is : %d mm\r\n", distance_centre);
        }
        #endif
        #if DISPLAY_LEFT
        if (board->sensor_left != NULL) {
            printf("Distance read by left satellite sensor is : %d mm\r\n", distance_left);
        }
        #endif
        #if DISPLAY_RIGHT
        if (board->sensor_right != NULL) {
            printf("Distance read by right satellite sensor is : %d mm\r\n", distance_right);
        }
        #endif
    }

    return status;
}
