#ifndef MPU6050_H
#define MPU6050_H

#include "mbed.h"

/** MPU6050
 *
 * 三軸加速度&ジャイロセンサー
 * 説明用に最低限のドキュメントを作成
 */
class MPU6050
{
protected:

public:
    // Set initial input parameters
    enum Ascale {
        AFS_2G = 0,
        AFS_4G,
        AFS_8G,
        AFS_16G
    };

    enum Gscale {
        GFS_250DPS = 0,
        GFS_500DPS,
        GFS_1000DPS,
        GFS_2000DPS
    };

//===================================================================================================================
//====== Set of useful function to access acceleratio, gyroscope, and temperature data
//===================================================================================================================
    /** センサの初期設定をする宣言
    * @param 12c_sda SDAをつないだピン名
    * @param i2c_scl SCLをつないだピン名
    * @param adO AD0ピンがhighとlowのどちらになっているか、普通は書かなくてもよい
    */
    MPU6050( PinName i2c_sda, PinName i2c_scl, int ad0 = 0)
    : i2c_p( new I2C( i2c_sda, i2c_scl ) ), i2c( *i2c_p ) {

        // Using the GY-521 breakout board, I set ADO to 0 by grounding through a 4k7 resistor
        // Seven-bit device address is 110100 for ADO = 0 and 110101 for ADO = 1
        if(ad0 == 0) {
            adr = 0x68 << 1;
        } else {
            adr = 0x69 << 1;
        }

        // Specify sensor full scale
        _Gscale = GFS_250DPS;
        _Ascale = AFS_2G;

        _q[0] = 1.0f;
        _q[1] = 0.0f;
        _q[2] = 0.0f;
        _q[3] = 0.0f;
        deltat = 0.0f;

        // parameters for 6 DoF sensor fusion calculations
        float PI = 3.14159265358979323846f;
        float GyroMeasError = PI * (60.0f / 180.0f);     // gyroscope measurement error in rads/s (start at 60 deg/s), then reduce after ~10 s to 3
        beta = sqrt(3.0f / 4.0f) * GyroMeasError;  // compute beta
        float GyroMeasDrift = PI * (1.0f / 180.0f);      // gyroscope measurement drift in rad/s/s (start at 0.0 deg/s/s)
        zeta = sqrt(3.0f / 4.0f) * GyroMeasDrift;  // compute zeta, the other free parameter in the Madgwick scheme usually set to a small or zero value

        float gyroBias[3] = {0, 0, 0}, accelBias[3] = {0, 0, 0}; // Bias corrections for gyro and accelerometer
        float SelfTest[6];

        MPU6050SelfTest(SelfTest);
        resetMPU6050();
        calibrateMPU6050(gyroBias, accelBias);
        initMPU6050();
    }

    // scale resolutions per LSB for the sensors
    float getGres() {
        float gRes;
        switch (_Gscale) {
                // Possible gyro scales (and their register bit settings) are:
                // 250 DPS (00), 500 DPS (01), 1000 DPS (10), and 2000 DPS  (11).
                // Here's a bit of an algorith to calculate DPS/(ADC tick) based on that 2-bit value:
            case GFS_250DPS:
                gRes = 250.0/32768.0;
                break;
            case GFS_500DPS:
                gRes = 500.0/32768.0;
                break;
            case GFS_1000DPS:
                gRes = 1000.0/32768.0;
                break;
            case GFS_2000DPS:
                gRes = 2000.0/32768.0;
                break;
        }
        return gRes;
    }

    float getAres() {
        float aRes;
        switch (_Ascale) {
                // Possible accelerometer scales (and their register bit settings) are:
                // 2 Gs (00), 4 Gs (01), 8 Gs (10), and 16 Gs  (11).
                // Here's a bit of an algorith to calculate DPS/(ADC tick) based on that 2-bit value:
            case AFS_2G:
                aRes = 2.0/32768.0;
                break;
            case AFS_4G:
                aRes = 4.0/32768.0;
                break;
            case AFS_8G:
                aRes = 8.0/32768.0;
                break;
            case AFS_16G:
                aRes = 16.0/32768.0;
                break;
        }
        return aRes;
    }


    void readAccelData(int * destination) {
        /** 加速度の読み出し
         *  @param destination int[3]の配列を渡してください、加速度をxyz順に返します
         */
        uint8_t rawData[6];  // x/y/z accel register data stored here
        readBytes(ACCEL_XOUT_H, 6, &rawData[0]);  // Read the six raw data registers into data array
        destination[0] = (int)(((int8_t)rawData[0] << 8) | rawData[1]) ;  // Turn the MSB and LSB into a signed 16-bit value
        destination[1] = (int)(((int8_t)rawData[2] << 8) | rawData[3]) ;
        destination[2] = (int)(((int8_t)rawData[4] << 8) | rawData[5]) ;
    }

    void readGyroData(int * destination) {
        /** 角速度の読み出し
         *  @param destination int[3]の配列を渡してください、角速度をxyz順に返します
         */
        uint8_t rawData[6];  // x/y/z gyro register data stored here
        readBytes(GYRO_XOUT_H, 6, &rawData[0]);  // Read the six raw data registers sequentially into data array
        destination[0] = (int)(((int8_t)rawData[0] << 8) | rawData[1]) ;  // Turn the MSB and LSB into a signed 16-bit value
        destination[1] = (int)(((int8_t)rawData[2] << 8) | rawData[3]) ;
        destination[2] = (int)(((int8_t)rawData[4] << 8) | rawData[5]) ;
    }

    int readTempData() {
        /** 温度の読み出し
         *  @return int型の変数に代入してください、温度を返します
         */
        uint8_t rawData[2];  // x/y/z gyro register data stored here
        readBytes(TEMP_OUT_H, 2, &rawData[0]);  // Read the two raw data registers sequentially into data array
        return (int)(((int8_t)rawData[0]) << 8 | rawData[1]) ;  // Turn the MSB and LSB into a 16-bit value
    }



// Configure the motion detection control for low power accelerometer mode
    void LowPowerAccelOnly() {

// The sensor has a high-pass filter necessary to invoke to allow the sensor motion detection algorithms work properly
// Motion detection occurs on free-fall (acceleration below a threshold for some time for all axes), motion (acceleration
// above a threshold for some time on at least one axis), and zero-motion toggle (acceleration on each axis less than a
// threshold for some time sets this flag, motion above the threshold turns it off). The high-pass filter takes gravity out
// consideration for these threshold evaluations; otherwise, the flags would be set all the time!

        uint8_t c = readByte(PWR_MGMT_1);
        writeByte(PWR_MGMT_1, c & ~0x30); // Clear sleep and cycle bits [5:6]
        writeByte(PWR_MGMT_1, c |  0x30); // Set sleep and cycle bits [5:6] to zero to make sure accelerometer is running

        c = readByte(PWR_MGMT_2);
        writeByte(PWR_MGMT_2, c & ~0x38); // Clear standby XA, YA, and ZA bits [3:5]
        writeByte(PWR_MGMT_2, c |  0x00); // Set XA, YA, and ZA bits [3:5] to zero to make sure accelerometer is running

        c = readByte(ACCEL_CONFIG);
        writeByte(ACCEL_CONFIG, c & ~0x07); // Clear high-pass filter bits [2:0]
// Set high-pass filter to 0) reset (disable), 1) 5 Hz, 2) 2.5 Hz, 3) 1.25 Hz, 4) 0.63 Hz, or 7) Hold
        writeByte(ACCEL_CONFIG,  c | 0x00);  // Set ACCEL_HPF to 0; reset mode disbaling high-pass filter

        c = readByte(CONFIG);
        writeByte(CONFIG, c & ~0x07); // Clear low-pass filter bits [2:0]
        writeByte(CONFIG, c |  0x00);  // Set DLPD_CFG to 0; 260 Hz bandwidth, 1 kHz rate

        c = readByte(INT_ENABLE);
        writeByte(INT_ENABLE, c & ~0xFF);  // Clear all interrupts
        writeByte(INT_ENABLE, 0x40);  // Enable motion threshold (bits 5) interrupt only

// Motion detection interrupt requires the absolute value of any axis to lie above the detection threshold
// for at least the counter duration
        writeByte(MOT_THR, 0x80); // Set motion detection to 0.256 g; LSB = 2 mg
        writeByte(MOT_DUR, 0x01); // Set motion detect duration to 1  ms; LSB is 1 ms @ 1 kHz rate

        wait(0.1);  // Add delay for accumulation of samples

        c = readByte(ACCEL_CONFIG);
        writeByte(ACCEL_CONFIG, c & ~0x07); // Clear high-pass filter bits [2:0]
        writeByte(ACCEL_CONFIG, c |  0x07);  // Set ACCEL_HPF to 7; hold the initial accleration value as a referance

        c = readByte(PWR_MGMT_2);
        writeByte(PWR_MGMT_2, c & ~0xC7); // Clear standby XA, YA, and ZA bits [3:5] and LP_WAKE_CTRL bits [6:7]
        writeByte(PWR_MGMT_2, c |  0x47); // Set wakeup frequency to 5 Hz, and disable XG, YG, and ZG gyros (bits [0:2])

        c = readByte(PWR_MGMT_1);
        writeByte(PWR_MGMT_1, c & ~0x20); // Clear sleep and cycle bit 5
        writeByte(PWR_MGMT_1, c |  0x20); // Set cycle bit 5 to begin low power accelerometer motion interrupts

    }


    void resetMPU6050() {
        // reset device
        writeByte(PWR_MGMT_1, 0x80); // Write a one to bit 7 reset bit; toggle reset device
        wait(0.1);
    }


    void initMPU6050() {
// Initialize MPU6050 device
// wake up device
        writeByte(PWR_MGMT_1, 0x00); // Clear sleep mode bit (6), enable all sensors
        wait(0.1); // Delay 100 ms for PLL to get established on x-axis gyro; should check for PLL ready interrupt

// get stable time source
        writeByte(PWR_MGMT_1, 0x01);  // Set clock source to be PLL with x-axis gyroscope reference, bits 2:0 = 001

// Configure Gyro and Accelerometer
// Disable FSYNC and set accelerometer and gyro bandwidth to 44 and 42 Hz, respectively;
// DLPF_CFG = bits 2:0 = 010; this sets the sample rate at 1 kHz for both
// Maximum delay is 4.9 ms which is just over a 200 Hz maximum rate
        writeByte(CONFIG, 0x03);

// Set sample rate = gyroscope output rate/(1 + SMPLRT_DIV)
        writeByte(SMPLRT_DIV, 0x04);  // Use a 200 Hz rate; the same rate set in CONFIG above

// Set gyroscope full scale range
// Range selects FS_SEL and AFS_SEL are 0 - 3, so 2-bit values are left-shifted into positions 4:3
        uint8_t c =  readByte(GYRO_CONFIG);
        writeByte(GYRO_CONFIG, c & ~0xE0); // Clear self-test bits [7:5]
        writeByte(GYRO_CONFIG, c & ~0x18); // Clear AFS bits [4:3]
        writeByte(GYRO_CONFIG, c | _Gscale << 3); // Set full scale range for the gyro

// Set accelerometer configuration
        c =  readByte(ACCEL_CONFIG);
        writeByte(ACCEL_CONFIG, c & ~0xE0); // Clear self-test bits [7:5]
        writeByte(ACCEL_CONFIG, c & ~0x18); // Clear AFS bits [4:3]
        writeByte(ACCEL_CONFIG, c | _Ascale << 3); // Set full scale range for the accelerometer

        // Configure Interrupts and Bypass Enable
        // Set interrupt pin active high, push-pull, and clear on read of INT_STATUS, enable I2C_BYPASS_EN so additional chips
        // can join the I2C bus and all can be controlled by the Arduino as master
        writeByte(INT_PIN_CFG, 0x22);
        writeByte(INT_ENABLE, 0x01);  // Enable data ready (bit 0) interrupt
    }

// Function which accumulates gyro and accelerometer data after device initialization. It calculates the average
// of the at-rest readings and then loads the resulting offsets into accelerometer and gyro bias registers.
    void calibrateMPU6050(float * dest1, float * dest2) {
        uint8_t data[12]; // data array to hold accelerometer and gyro x, y, z, data
        uint16_t ii, packet_count, fifo_count;
        int32_t gyro_bias[3] = {0, 0, 0}, accel_bias[3] = {0, 0, 0};

// reset device, reset all registers, clear gyro and accelerometer bias registers
        writeByte(PWR_MGMT_1, 0x80); // Write a one to bit 7 reset bit; toggle reset device
        wait(0.1);

// get stable time source
// Set clock source to be PLL with x-axis gyroscope reference, bits 2:0 = 001
        writeByte(PWR_MGMT_1, 0x01);
        writeByte(PWR_MGMT_2, 0x00);
        wait(0.2);

// Configure device for bias calculation
        writeByte(INT_ENABLE, 0x00);   // Disable all interrupts
        writeByte(FIFO_EN, 0x00);      // Disable FIFO
        writeByte(PWR_MGMT_1, 0x00);   // Turn on internal clock source
        writeByte(I2C_MST_CTRL, 0x00); // Disable I2C master
        writeByte(USER_CTRL, 0x00);    // Disable FIFO and I2C master modes
        writeByte(USER_CTRL, 0x0C);    // Reset FIFO and DMP
        wait(0.015);

// Configure MPU6050 gyro and accelerometer for bias calculation
        writeByte(CONFIG, 0x01);      // Set low-pass filter to 188 Hz
        writeByte(SMPLRT_DIV, 0x00);  // Set sample rate to 1 kHz
        writeByte(GYRO_CONFIG, 0x00);  // Set gyro full-scale to 250 degrees per second, maximum sensitivity
        writeByte(ACCEL_CONFIG, 0x00); // Set accelerometer full-scale to 2 g, maximum sensitivity

        uint16_t  gyrosensitivity  = 131;   // = 131 LSB/degrees/sec
        uint16_t  accelsensitivity = 16384;  // = 16384 LSB/g

// Configure FIFO to capture accelerometer and gyro data for bias calculation
        writeByte(USER_CTRL, 0x40);   // Enable FIFO
        writeByte(FIFO_EN, 0x78);     // Enable gyro and accelerometer sensors for FIFO  (max size 1024 bytes in MPU-6050)
        wait(0.08); // accumulate 80 samples in 80 milliseconds = 960 bytes

// At end of sample accumulation, turn off FIFO sensor read
        writeByte(FIFO_EN, 0x00);        // Disable gyro and accelerometer sensors for FIFO
        readBytes(FIFO_COUNTH, 2, &data[0]); // read FIFO sample count
        fifo_count = ((uint16_t)data[0] << 8) | data[1];
        packet_count = fifo_count/12;// How many sets of full gyro and accelerometer data for averaging

        for (ii = 0; ii < packet_count; ii++) {
            int16_t accel_temp[3] = {0, 0, 0}, gyro_temp[3] = {0, 0, 0};
            readBytes(FIFO_R_W, 12, &data[0]); // read data for averaging
            accel_temp[0] = (int16_t) (((int16_t)data[0] << 8) | data[1]  ) ;  // Form signed 16-bit integer for each sample in FIFO
            accel_temp[1] = (int16_t) (((int16_t)data[2] << 8) | data[3]  ) ;
            accel_temp[2] = (int16_t) (((int16_t)data[4] << 8) | data[5]  ) ;
            gyro_temp[0]  = (int16_t) (((int16_t)data[6] << 8) | data[7]  ) ;
            gyro_temp[1]  = (int16_t) (((int16_t)data[8] << 8) | data[9]  ) ;
            gyro_temp[2]  = (int16_t) (((int16_t)data[10] << 8) | data[11]) ;

            accel_bias[0] += (int32_t) accel_temp[0]; // Sum individual signed 16-bit biases to get accumulated signed 32-bit biases
            accel_bias[1] += (int32_t) accel_temp[1];
            accel_bias[2] += (int32_t) accel_temp[2];
            gyro_bias[0]  += (int32_t) gyro_temp[0];
            gyro_bias[1]  += (int32_t) gyro_temp[1];
            gyro_bias[2]  += (int32_t) gyro_temp[2];

        }
        accel_bias[0] /= (int32_t) packet_count; // Normalize sums to get average count biases
        accel_bias[1] /= (int32_t) packet_count;
        accel_bias[2] /= (int32_t) packet_count;
        gyro_bias[0]  /= (int32_t) packet_count;
        gyro_bias[1]  /= (int32_t) packet_count;
        gyro_bias[2]  /= (int32_t) packet_count;

        if(accel_bias[2] > 0L) {
            accel_bias[2] -= (int32_t) accelsensitivity;   // Remove gravity from the z-axis accelerometer bias calculation
        } else {
            accel_bias[2] += (int32_t) accelsensitivity;
        }

// Construct the gyro biases for push to the hardware gyro bias registers, which are reset to zero upon device startup
        data[0] = (-gyro_bias[0]/4  >> 8) & 0xFF; // Divide by 4 to get 32.9 LSB per deg/s to conform to expected bias input format
        data[1] = (-gyro_bias[0]/4)       & 0xFF; // Biases are additive, so change sign on calculated average gyro biases
        data[2] = (-gyro_bias[1]/4  >> 8) & 0xFF;
        data[3] = (-gyro_bias[1]/4)       & 0xFF;
        data[4] = (-gyro_bias[2]/4  >> 8) & 0xFF;
        data[5] = (-gyro_bias[2]/4)       & 0xFF;

// Push gyro biases to hardware registers
        writeByte(XG_OFFS_USRH, data[0]);
        writeByte(XG_OFFS_USRL, data[1]);
        writeByte(YG_OFFS_USRH, data[2]);
        writeByte(YG_OFFS_USRL, data[3]);
        writeByte(ZG_OFFS_USRH, data[4]);
        writeByte(ZG_OFFS_USRL, data[5]);

        dest1[0] = (float) gyro_bias[0]/(float) gyrosensitivity; // construct gyro bias in deg/s for later manual subtraction
        dest1[1] = (float) gyro_bias[1]/(float) gyrosensitivity;
        dest1[2] = (float) gyro_bias[2]/(float) gyrosensitivity;

// Construct the accelerometer biases for push to the hardware accelerometer bias registers. These registers contain
// factory trim values which must be added to the calculated accelerometer biases; on boot up these registers will hold
// non-zero values. In addition, bit 0 of the lower byte must be preserved since it is used for temperature
// compensation calculations. Accelerometer bias registers expect bias input as 2048 LSB per g, so that
// the accelerometer biases calculated above must be divided by 8.

        int32_t accel_bias_reg[3] = {0, 0, 0}; // A place to hold the factory accelerometer trim biases
        readBytes(XA_OFFSET_H, 2, &data[0]); // Read factory accelerometer trim values
        accel_bias_reg[0] = (int16_t) ((int16_t)data[0] << 8) | data[1];
        readBytes(YA_OFFSET_H, 2, &data[0]);
        accel_bias_reg[1] = (int16_t) ((int16_t)data[0] << 8) | data[1];
        readBytes(ZA_OFFSET_H, 2, &data[0]);
        accel_bias_reg[2] = (int16_t) ((int16_t)data[0] << 8) | data[1];

        uint32_t mask = 1uL; // Define mask for temperature compensation bit 0 of lower byte of accelerometer bias registers
        uint8_t mask_bit[3] = {0, 0, 0}; // Define array to hold mask bit for each accelerometer bias axis

        for(ii = 0; ii < 3; ii++) {
            if(accel_bias_reg[ii] & mask) mask_bit[ii] = 0x01; // If temperature compensation bit is set, record that fact in mask_bit
        }

        // Construct total accelerometer bias, including calculated average accelerometer bias from above
        accel_bias_reg[0] -= (accel_bias[0]/8); // Subtract calculated averaged accelerometer bias scaled to 2048 LSB/g (16 g full scale)
        accel_bias_reg[1] -= (accel_bias[1]/8);
        accel_bias_reg[2] -= (accel_bias[2]/8);

        data[0] = (accel_bias_reg[0] >> 8) & 0xFF;
        data[1] = (accel_bias_reg[0])      & 0xFF;
        data[1] = data[1] | mask_bit[0]; // preserve temperature compensation bit when writing back to accelerometer bias registers
        data[2] = (accel_bias_reg[1] >> 8) & 0xFF;
        data[3] = (accel_bias_reg[1])      & 0xFF;
        data[3] = data[3] | mask_bit[1]; // preserve temperature compensation bit when writing back to accelerometer bias registers
        data[4] = (accel_bias_reg[2] >> 8) & 0xFF;
        data[5] = (accel_bias_reg[2])      & 0xFF;
        data[5] = data[5] | mask_bit[2]; // preserve temperature compensation bit when writing back to accelerometer bias registers

        // Push accelerometer biases to hardware registers
//  writeByte(XA_OFFSET_H, data[0]);
//  writeByte(XA_OFFSET_L_TC, data[1]);
//  writeByte(YA_OFFSET_H, data[2]);
//  writeByte(YA_OFFSET_L_TC, data[3]);
//  writeByte(ZA_OFFSET_H, data[4]);
//  writeByte(ZA_OFFSET_L_TC, data[5]);

// Output scaled accelerometer biases for manual subtraction in the main program
        dest2[0] = (float)accel_bias[0]/(float)accelsensitivity;
        dest2[1] = (float)accel_bias[1]/(float)accelsensitivity;
        dest2[2] = (float)accel_bias[2]/(float)accelsensitivity;
    }


// Accelerometer and gyroscope self test; check calibration wrt factory settings
    void MPU6050SelfTest(float * destination) { // Should return percent deviation from factory trim values, +/- 14 or less deviation is a pass
        uint8_t rawData[4] = {0, 0, 0, 0};
        uint8_t selfTest[6];
        float factoryTrim[6];

        // Configure the accelerometer for self-test
        writeByte(ACCEL_CONFIG, 0xF0); // Enable self test on all three axes and set accelerometer range to +/- 8 g
        writeByte(GYRO_CONFIG,  0xE0); // Enable self test on all three axes and set gyro range to +/- 250 degrees/s
        wait(0.25);  // Delay a while to let the device execute the self-test
        rawData[0] = readByte(SELF_TEST_X); // X-axis self-test results
        rawData[1] = readByte(SELF_TEST_Y); // Y-axis self-test results
        rawData[2] = readByte(SELF_TEST_Z); // Z-axis self-test results
        rawData[3] = readByte(SELF_TEST_A); // Mixed-axis self-test results
        // Extract the acceleration test results first
        selfTest[0] = (rawData[0] >> 3) | (rawData[3] & 0x30) >> 4 ; // XA_TEST result is a five-bit unsigned integer
        selfTest[1] = (rawData[1] >> 3) | (rawData[3] & 0x0C) >> 4 ; // YA_TEST result is a five-bit unsigned integer
        selfTest[2] = (rawData[2] >> 3) | (rawData[3] & 0x03) >> 4 ; // ZA_TEST result is a five-bit unsigned integer
        // Extract the gyration test results first
        selfTest[3] = rawData[0]  & 0x1F ; // XG_TEST result is a five-bit unsigned integer
        selfTest[4] = rawData[1]  & 0x1F ; // YG_TEST result is a five-bit unsigned integer
        selfTest[5] = rawData[2]  & 0x1F ; // ZG_TEST result is a five-bit unsigned integer
        // Process results to allow final comparison with factory set values
        factoryTrim[0] = (4096.0f*0.34f)*(pow( (0.92f/0.34f) , ((selfTest[0] - 1.0f)/30.0f))); // FT[Xa] factory trim calculation
        factoryTrim[1] = (4096.0f*0.34f)*(pow( (0.92f/0.34f) , ((selfTest[1] - 1.0f)/30.0f))); // FT[Ya] factory trim calculation
        factoryTrim[2] = (4096.0f*0.34f)*(pow( (0.92f/0.34f) , ((selfTest[2] - 1.0f)/30.0f))); // FT[Za] factory trim calculation
        factoryTrim[3] =  ( 25.0f*131.0f)*(pow( 1.046f , (selfTest[3] - 1.0f) ));             // FT[Xg] factory trim calculation
        factoryTrim[4] =  (-25.0f*131.0f)*(pow( 1.046f , (selfTest[4] - 1.0f) ));             // FT[Yg] factory trim calculation
        factoryTrim[5] =  ( 25.0f*131.0f)*(pow( 1.046f , (selfTest[5] - 1.0f) ));             // FT[Zg] factory trim calculation

//  Output self-test results and factory trim calculation if desired
//  Serial.println(selfTest[0]); Serial.println(selfTest[1]); Serial.println(selfTest[2]);
//  Serial.println(selfTest[3]); Serial.println(selfTest[4]); Serial.println(selfTest[5]);
//  Serial.println(factoryTrim[0]); Serial.println(factoryTrim[1]); Serial.println(factoryTrim[2]);
//  Serial.println(factoryTrim[3]); Serial.println(factoryTrim[4]); Serial.println(factoryTrim[5]);

// Report results as a ratio of (STR - FT)/FT; the change from Factory Trim of the Self-Test Response
// To get to percent, must multiply by 100 and subtract result from 100
        for (int i = 0; i < 6; i++) {
            destination[i] = 100.0f + 100.0f*(selfTest[i] - factoryTrim[i])/factoryTrim[i]; // Report percent differences
        }

    }


// Implementation of Sebastian Madgwick's "...efficient orientation filter for... inertial/magnetic sensor arrays"
// (see http://www.x-io.co.uk/category/open-source/ for examples and more details)
// which fuses acceleration and rotation rate to produce a quaternion-based estimate of relative
// device orientation -- which can be converted to yaw, pitch, and roll. Useful for stabilizing quadcopters, etc.
// The performance of the orientation filter is at least as good as conventional Kalman-based filtering algorithms
// but is much less computationally intensive---it can be performed on a 3.3 V Pro Mini operating at 8 MHz!
    void MadgwickQuaternionUpdate(float ax, float ay, float az, float gx, float gy, float gz) {
        float q1 = _q[0], q2 = _q[1], q3 = _q[2], q4 = _q[3];         // short name local variable for readability
        float norm;                                               // vector norm
        float f1, f2, f3;                                         // objective funcyion elements
        float J_11or24, J_12or23, J_13or22, J_14or21, J_32, J_33; // objective function Jacobian elements
        float qDot1, qDot2, qDot3, qDot4;
        float hatDot1, hatDot2, hatDot3, hatDot4;
        float gerrx, gerry, gerrz, gbiasx, gbiasy, gbiasz;  // gyro bias error

        // Auxiliary variables to avoid repeated arithmetic
        float _halfq1 = 0.5f * q1;
        float _halfq2 = 0.5f * q2;
        float _halfq3 = 0.5f * q3;
        float _halfq4 = 0.5f * q4;
        float _2q1 = 2.0f * q1;
        float _2q2 = 2.0f * q2;
        float _2q3 = 2.0f * q3;
        float _2q4 = 2.0f * q4;
//            float _2q1q3 = 2.0f * q1 * q3;
//            float _2q3q4 = 2.0f * q3 * q4;

        // Normalise accelerometer measurement
        norm = sqrt(ax * ax + ay * ay + az * az);
        if (norm == 0.0f) return; // handle NaN
        norm = 1.0f/norm;
        ax *= norm;
        ay *= norm;
        az *= norm;

        // Compute the objective function and Jacobian
        f1 = _2q2 * q4 - _2q1 * q3 - ax;
        f2 = _2q1 * q2 + _2q3 * q4 - ay;
        f3 = 1.0f - _2q2 * q2 - _2q3 * q3 - az;
        J_11or24 = _2q3;
        J_12or23 = _2q4;
        J_13or22 = _2q1;
        J_14or21 = _2q2;
        J_32 = 2.0f * J_14or21;
        J_33 = 2.0f * J_11or24;

        // Compute the gradient (matrix multiplication)
        hatDot1 = J_14or21 * f2 - J_11or24 * f1;
        hatDot2 = J_12or23 * f1 + J_13or22 * f2 - J_32 * f3;
        hatDot3 = J_12or23 * f2 - J_33 *f3 - J_13or22 * f1;
        hatDot4 = J_14or21 * f1 + J_11or24 * f2;

        // Normalize the gradient
        norm = sqrt(hatDot1 * hatDot1 + hatDot2 * hatDot2 + hatDot3 * hatDot3 + hatDot4 * hatDot4);
        hatDot1 /= norm;
        hatDot2 /= norm;
        hatDot3 /= norm;
        hatDot4 /= norm;

        // Compute estimated gyroscope biases
        gerrx = _2q1 * hatDot2 - _2q2 * hatDot1 - _2q3 * hatDot4 + _2q4 * hatDot3;
        gerry = _2q1 * hatDot3 + _2q2 * hatDot4 - _2q3 * hatDot1 - _2q4 * hatDot2;
        gerrz = _2q1 * hatDot4 - _2q2 * hatDot3 + _2q3 * hatDot2 - _2q4 * hatDot1;

        // Compute and remove gyroscope biases
        gbiasx += gerrx * deltat * zeta;
        gbiasy += gerry * deltat * zeta;
        gbiasz += gerrz * deltat * zeta;
//           gx -= gbiasx;
//           gy -= gbiasy;
//           gz -= gbiasz;

        // Compute the quaternion derivative
        qDot1 = -_halfq2 * gx - _halfq3 * gy - _halfq4 * gz;
        qDot2 =  _halfq1 * gx + _halfq3 * gz - _halfq4 * gy;
        qDot3 =  _halfq1 * gy - _halfq2 * gz + _halfq4 * gx;
        qDot4 =  _halfq1 * gz + _halfq2 * gy - _halfq3 * gx;

        // Compute then integrate estimated quaternion derivative
        q1 += (qDot1 -(beta * hatDot1)) * deltat;
        q2 += (qDot2 -(beta * hatDot2)) * deltat;
        q3 += (qDot3 -(beta * hatDot3)) * deltat;
        q4 += (qDot4 -(beta * hatDot4)) * deltat;

        // Normalize the quaternion
        norm = sqrt(q1 * q1 + q2 * q2 + q3 * q3 + q4 * q4);    // normalise quaternion
        norm = 1.0f/norm;
        _q[0] = q1 * norm;
        _q[1] = q2 * norm;
        _q[2] = q3 * norm;
        _q[3] = q4 * norm;

    }
    
private:
// Define registers per MPU6050, Register Map and Descriptions, Rev 4.2, 08/19/2013 6 DOF Motion sensor fusion device
// Invensense Inc., www.invensense.com
// See also MPU-6050 Register Map and Descriptions, Revision 4.0, RM-MPU-6050A-00, 9/12/2012 for registers not listed in
// above document; the MPU6050 and MPU 9150 are virtually identical but the latter has an on-board magnetic sensor
    enum register_adr{
        XGOFFS_TC           = 0x00, // Bit 7 PWR_MODE, bits 6:1 XG_OFFS_TC, bit 0 OTP_BNK_VLD
        YGOFFS_TC           = 0x01,
        ZGOFFS_TC           = 0x02,
        X_FINE_GAIN         = 0x03, // [7:0] fine gain
        Y_FINE_GAIN         = 0x04,
        Z_FINE_GAIN         = 0x05,
        XA_OFFSET_H         = 0x06, // User-defined trim values for accelerometer
        XA_OFFSET_L_TC      = 0x07,
        YA_OFFSET_H         = 0x08,
        YA_OFFSET_L_TC      = 0x09,
        ZA_OFFSET_H         = 0x0A,
        ZA_OFFSET_L_TC      = 0x0B,
        SELF_TEST_X         = 0x0D,
        SELF_TEST_Y         = 0x0E,
        SELF_TEST_Z         = 0x0F,
        SELF_TEST_A         = 0x10,
        XG_OFFS_USRH        = 0x13, // User-defined trim values for gyroscope; supported in MPU-6050?
        XG_OFFS_USRL        = 0x14,
        YG_OFFS_USRH        = 0x15,
        YG_OFFS_USRL        = 0x16,
        ZG_OFFS_USRH        = 0x17,
        ZG_OFFS_USRL        = 0x18,
        SMPLRT_DIV          = 0x19,
        CONFIG              = 0x1A,
        GYRO_CONFIG         = 0x1B,
        ACCEL_CONFIG        = 0x1C,
        FF_THR              = 0x1D, // Free-fall
        FF_DUR              = 0x1E, // Free-fall
        MOT_THR             = 0x1F, // Motion detection threshold bits [7:0]
        MOT_DUR             = 0x20, // Duration counter threshold for motion interrupt generation, 1 kHz rate, LSB = 1 ms
        ZMOT_THR            = 0x21, // Zero-motion detection threshold bits [7:0]
        ZRMOT_DUR           = 0x22, // Duration counter threshold for zero motion interrupt generation, 16 Hz rate, LSB = 64 ms
        FIFO_EN             = 0x23,
        I2C_MST_CTRL        = 0x24,
        I2C_SLV0_ADDR       = 0x25,
        I2C_SLV0_REG        = 0x26,
        I2C_SLV0_CTRL       = 0x27,
        I2C_SLV1_ADDR       = 0x28,
        I2C_SLV1_REG        = 0x29,
        I2C_SLV1_CTRL       = 0x2A,
        I2C_SLV2_ADDR       = 0x2B,
        I2C_SLV2_REG        = 0x2C,
        I2C_SLV2_CTRL       = 0x2D,
        I2C_SLV3_ADDR       = 0x2E,
        I2C_SLV3_REG        = 0x2F,
        I2C_SLV3_CTRL       = 0x30,
        I2C_SLV4_ADDR       = 0x31,
        I2C_SLV4_REG        = 0x32,
        I2C_SLV4_DO         = 0x33,
        I2C_SLV4_CTRL       = 0x34,
        I2C_SLV4_DI         = 0x35,
        I2C_MST_STATUS      = 0x36,
        INT_PIN_CFG         = 0x37,
        INT_ENABLE          = 0x38,
        DMP_INT_STATUS      = 0x39, // Check DMP interrupt
        INT_STATUS          = 0x3A,
        ACCEL_XOUT_H        = 0x3B,
        ACCEL_XOUT_L        = 0x3C,
        ACCEL_YOUT_H        = 0x3D,
        ACCEL_YOUT_L        = 0x3E,
        ACCEL_ZOUT_H        = 0x3F,
        ACCEL_ZOUT_L        = 0x40,
        TEMP_OUT_H          = 0x41,
        TEMP_OUT_L          = 0x42,
        GYRO_XOUT_H         = 0x43,
        GYRO_XOUT_L         = 0x44,
        GYRO_YOUT_H         = 0x45,
        GYRO_YOUT_L         = 0x46,
        GYRO_ZOUT_H         = 0x47,
        GYRO_ZOUT_L         = 0x48,
        EXT_SENS_DATA_00    = 0x49,
        EXT_SENS_DATA_01    = 0x4A,
        EXT_SENS_DATA_02    = 0x4B,
        EXT_SENS_DATA_03    = 0x4C,
        EXT_SENS_DATA_04    = 0x4D,
        EXT_SENS_DATA_05    = 0x4E,
        EXT_SENS_DATA_06    = 0x4F,
        EXT_SENS_DATA_07    = 0x50,
        EXT_SENS_DATA_08    = 0x51,
        EXT_SENS_DATA_09    = 0x52,
        EXT_SENS_DATA_10    = 0x53,
        EXT_SENS_DATA_11    = 0x54,
        EXT_SENS_DATA_12    = 0x55,
        EXT_SENS_DATA_13    = 0x56,
        EXT_SENS_DATA_14    = 0x57,
        EXT_SENS_DATA_15    = 0x58,
        EXT_SENS_DATA_16    = 0x59,
        EXT_SENS_DATA_17    = 0x5A,
        EXT_SENS_DATA_18    = 0x5B,
        EXT_SENS_DATA_19    = 0x5C,
        EXT_SENS_DATA_20    = 0x5D,
        EXT_SENS_DATA_21    = 0x5E,
        EXT_SENS_DATA_22    = 0x5F,
        EXT_SENS_DATA_23    = 0x60,
        MOT_DETECT_STATUS   = 0x61,
        I2C_SLV0_DO         = 0x63,
        I2C_SLV1_DO         = 0x64,
        I2C_SLV2_DO         = 0x65,
        I2C_SLV3_DO         = 0x66,
        I2C_MST_DELAY_CTRL  = 0x67,
        SIGNAL_PATH_RESET   = 0x68,
        MOT_DETECT_CTRL     = 0x69,
        USER_CTRL           = 0x6A, // Bit 7 enable DMP, bit 3 reset DMP
        PWR_MGMT_1          = 0x6B, // Device defaults to the SLEEP mode
        PWR_MGMT_2          = 0x6C,
        DMP_BANK            = 0x6D, // Activates a specific bank in the DMP
        DMP_RW_PNT          = 0x6E, // Set read/write pointer to a specific start address in specified DMP bank
        DMP_REG             = 0x6F, // Register in DMP from which to read or to which to write
        DMP_REG_1           = 0x70,
        DMP_REG_2           = 0x71,
        FIFO_COUNTH         = 0x72,
        FIFO_COUNTL         = 0x73,
        FIFO_R_W            = 0x74,
        WHO_AM_I_MPU6050    = 0x75, // Should return 0x68
    };
    
    int _Gscale;
    int _Ascale;

    float _q[4]; // vector to hold quaternion
    float beta;
    float zeta;
    float deltat;    // integration interval for both filter schemes

    //I2C
    I2C     *i2c_p;
    I2C     &i2c;
    char    adr;

    void writeByte(uint8_t address, uint8_t data) {
        char data_write[2];
        data_write[0] = address;
        data_write[1] = data;
        i2c.write(adr, data_write, 2, 0);
    }

    char readByte(uint8_t address) {
        char data[1]; // `data` will store the register data
        char data_write[1];
        data_write[0] = address;
        i2c.write(adr, data_write, 1, 1); // no stop
        i2c.read(adr, data, 1, 0);
        return data[0];
    }

    void readBytes(uint8_t address, uint8_t count, uint8_t * dest) {
        char data[14];
        char data_write[1];
        data_write[0] = address;
        i2c.write(adr, data_write, 1, 1); // no stop
        i2c.read(adr, data, count, 0);
        for(int ii = 0; ii < count; ii++) {
            dest[ii] = data[ii];
        }
    }

};
#endif