/// @file IHM_
/// @mainpage
///
///
///
/// 
///
/// @note
/// History
/// v3.00 31/10/2019


#include "mbed.h"

typedef int int32_t;


#include <time.h>

/// RealTimeClock is an interface to the mbed RTC module
///
/// It provides interfaces to get and set the time in text representation
/// as well as a means to get and set the timezone offset so that it
/// maintains time in UTC format.
///

///
/// @note It is recommended that the system has a battery connected to
/// the mbed battery backup pin.
///

///
/// Example:
/// @code
///#include "mbed.h"
///#include "RTC_IUT.h"
///Rtc rtc;
///
///int main()
///{
///    //char TimeTest[] = "31/12/1999 23:59:50";
///    //char TimeTest[] = "19/01/2038 03:13:50";
///    char TimeTest[] = "03/06/1970 10:00:00";
///    time_t epoc;
///    char buf[50];
///    rtc.SetTimeString(TimeTest);
///    rtc.GetTimeString(buf);
///    epoc =rtc.GetTimeValue();
///    printf("epoc: %8X, time: %s\r\n", epoc, buf);
///    while(1) {
///        rtc.SetYear(2020);   // Set the year
///       rtc.GetTimeString(buf);  // read date time string DD:MM:YYY hh:mm:ss
///        epoc =rtc.GetTimeValue();  // Read second since 01/01/1970
///        printf("epoc: %8X, time: %s\r\n", epoc, buf);
///        wait(0.9999);
///    }
///}
/// @endcode
///
class Rtc
{
public:
    /// Constructor for the RealTimeClock interface, usually implemented
    /// outside of main( )
    ///
    Rtc();

    /// Destructor for the RealTimeClock, usually not executed
    ///
    ~Rtc();

    /// GetTimeValue gets the current time in time_t format
    ///
    /// Gets the current time from the rtc and returns the time in time_t format.
    /// This is functionally identical to "return time(NULL);",
    /// simply wrapped in this class for convenience.
    ///
    /// @returns current time in time_t format
    ///
    time_t GetTimeValue();

    /// SetTimeValue will set the current time in time_t format
    ///
    /// example SetTimeValue(1463673534) set the current time
    ///  19 May 2016 15:58:54
    ///  the min value is 978307200 (01/01/2001 00:00:00)
    void SetTimeValue(time_t t);

    /// GetTimeString gets the formatted time
    ///
    /// It places the formatted string into the buffer in the format of
    ///  DD/MM/YYYY HH:MM:SS   30/12/2016 07:19:24
    ///
    /// @returns nothing
    ///
    void GetTimeString(char * buffer);


    /// SetTimeString will set the Real Time Clock from a well formatted string
    ///
    /// This will read a string, and if it is well formed, it will then
    /// set the RTC based on this string. The string should be of the form:
    /// 'DD/MM/YYYY HH:MM:SS'
    ///
    /// example: 30/12/2011 13:15:07
    ///
    /// @param timestring is the string to parse
    /// @returns true if the time was set
    ///
    bool SetTimeString(char * timestring);


    /// SetYear writes the year
    ///
    /// year format YYYY (1970 - 2038)
    void SetYear(unsigned short yyyy);

    /// GetYear reads the year
    ///
    /// @param void
    /// @returns the year unsigned short
    unsigned short GetYear(void);

    /// SetMon writes the month
    ///
    /// month format mm (1 - 12)
    void SetMon(unsigned char mm);

    /// GetMon reads the month
    ///
    /// @param void
    /// @returns the month unsigned char
    unsigned char GetMon(void);

    /// SetDay writes the day
    ///
    /// day format dd (1 - 31)
    void SetDay(unsigned char dd);

    /// GetDay reads the day
    ///
    /// @param void
    /// @returns the day unsigned char
    unsigned char GetDay(void);

    /// SetHour writes the hour
    ///
    /// hour format hh (0 - 23)
    void SetHour(unsigned char hh);

    /// GetHour reads the hour
    ///
    /// @param void
    /// @returns the hour unsigned char
    unsigned char GetHour(void);

    /// SetMin writes the minutes
    ///
    /// minute format mm (0 - 59)
    void SetMin(unsigned char mm);

    /// GetMin reads the minute
    ///
    /// @param void
    /// @returns the minute unsigned char
    unsigned char GetMin(void);

    /// SetSec writes the seconds
    ///
    /// second format ss (0 - 59)
    void SetSec(unsigned char ss);

    /// GetSec reads the second
    ///
    /// @param void
    /// @returns the second unsigned char
    unsigned char GetSec(void);

private:

};


