class UberVestService {
public:
    const static uint16_t UBER_VEST_SERVICE_UUID   = 0xB000;
    const static uint16_t ECG_CHARACTERISTIC_UUID  = 0xB002;
    const static uint16_t TEMP_CHARACTERISTIC_UUID = 0xB003;

    UberVestService(BLE &_ble, int8_t ecgInitial, uint8_t tempInitial) :
        ble(_ble),
        ecgState(ECG_CHARACTERISTIC_UUID, &ecgInitial, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        tempState(TEMP_CHARACTERISTIC_UUID, &tempInitial, 5, 5, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY)
    {
        GattCharacteristic *charTable[] = {&ecgState, &tempState};
        GattService         uberVestService(UberVestService::UBER_VEST_SERVICE_UUID, charTable, sizeof(charTable) / sizeof(GattCharacteristic *));
        
        ble.gattServer().addService(uberVestService);
    }

    void updateEcg(int8_t value) {
        ble.gattServer().write(ecgState.getValueHandle(), (uint8_t *)&value, sizeof(int8_t));
    }
    
    // The following adapted from https://developer.mbed.org/users/donalm/code/BLE_Health_Thermometer_Blog/file/f11df1469db2/main.cpp:
    void updateTemp(float value) {
        uint8_t  thermTempPayload[5] = { 0, 0, 0, 0, 0 };
        uint32_t temp_ieee11073      = quick_ieee11073_from_float(value);
        
        memcpy(thermTempPayload + 1, &temp_ieee11073, 4);
        
        ble.gattServer().write(tempState.getValueHandle(), thermTempPayload, sizeof(thermTempPayload));
    }
 
    /**
     * @brief A very quick conversion between a float temperature and 11073-20601 FLOAT-Type.
     * @param temperature The temperature as a float.
     * @return The temperature in 11073-20601 FLOAT-Type format.
     */
    uint32_t quick_ieee11073_from_float(float temperature)
    {
        uint8_t  exponent = 0xFF; //exponent is -1
        uint32_t mantissa = (uint32_t)(temperature*10);
        
        return ( ((uint32_t)exponent) << 24) | mantissa;
    }

private:
    BLE &ble;
    ReadOnlyGattCharacteristic<int8_t> ecgState;
    GattCharacteristic                 tempState;
};
