/*
Copyright 2017 (c) Analog Devices, Inc.

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
  - Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  - Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in
    the documentation and/or other materials provided with the
    distribution.
  - Neither the name of Analog Devices, Inc. nor the names of its
    contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.
  - The use of this software may or may not infringe the patent rights
    of one or more patent holders. This license does not release you
    from the requirement that you obtain separate licenses from these
    patent holders to use this software.
  - Use of the software either in source or binary form, must be run
    on or directly connected to an Analog Devices Inc. component.

THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, INTELLECTUAL PROPERTY RIGHTS, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*!
 ******************************************************************************
 * @file:
 * @brief:  Sensor type definitions for ADSNS1000.
 *-----------------------------------------------------------------------------
 */

#ifndef __ADI_SENSE_1000_SENSOR_TYPES_H__
#define __ADI_SENSE_1000_SENSOR_TYPES_H__

/*! @addtogroup ADSNS1000_Api
 *  @{
 */

#ifdef __cplusplus
extern "C" {
#endif

/*! ADSNS1000 measurement channel identifiers */
typedef enum {
    ADI_SENSE_1000_CHANNEL_ID_NONE  = -1,
    /*!< Used to indicate when no channel is selected (e.g. compensation channel) */

    ADI_SENSE_1000_CHANNEL_ID_CJC_0 = 0,
    /*!< Cold-Juction Compensation channel #0 */
    ADI_SENSE_1000_CHANNEL_ID_CJC_1,
    /*!< Cold-Juction Compensation channel #1 */
    ADI_SENSE_1000_CHANNEL_ID_SENSOR_0,
    /*!< Analog Sensor channel #0 */
    ADI_SENSE_1000_CHANNEL_ID_SENSOR_1,
    /*!< Analog Sensor channel #1 */
    ADI_SENSE_1000_CHANNEL_ID_SENSOR_2,
    /*!< Analog Sensor channel #2 */
    ADI_SENSE_1000_CHANNEL_ID_SENSOR_3,
    /*!< Analog Sensor channel #3 */
    ADI_SENSE_1000_CHANNEL_ID_VOLTAGE_0,
    /*!< Analog 0-10V Voltage Sensor channel #0 */
    ADI_SENSE_1000_CHANNEL_ID_CURRENT_0,
    /*!< Analog 4-20mA Current Sensor channel #0 */
    ADI_SENSE_1000_CHANNEL_ID_I2C_0,
    /*!< Digital I2C Sensor channel #0 */
    ADI_SENSE_1000_CHANNEL_ID_I2C_1,
    /*!< Digital I2C Sensor channel #1 */
    ADI_SENSE_1000_CHANNEL_ID_SPI_0,
    /*!< Digital SPI Sensor channel #0 */
    ADI_SENSE_1000_CHANNEL_ID_SPI_1,
    /*!< Digital SPI Sensor channel #1 */
    ADI_SENSE_1000_CHANNEL_ID_SPI_2,
    /*!< Digital SPI Sensor channel #2 */
    ADI_SENSE_1000_CHANNEL_ID_UART,
    /*!< Digital UART Sensor channel #0 */

    ADI_SENSE_1000_MAX_CHANNELS,
    /*!< Maximum number of measurement channels on ADSNS1000 */
} ADI_SENSE_1000_CHANNEL_ID;

/*! ADSNS1000 analog sensor type options
 *
 *  Select the sensor type that is connected to an ADC analog measurement
 *  channel.
 *
 *  @note Some channels may only support a subset of the available sensor types
 *        below.
 *
 *  @note The sensor type name may include a classification suffix:
 *        - _DEF_L1: pre-defined sensor using built-in linearisation data
 *        - _DEF_L2: pre-defined sensor using user-supplied linearisation data
 *        Where the suffix is absent, assume the _DEF_L1 classification above.
 */
typedef enum {
    ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_T_DEF_L1 = 0,
    /*!< Standard T-type Thermocouple temperature sensor with default
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_J_DEF_L1 = 1,
    /*!< Standard J-type Thermocouple temperature sensor with default
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_K_DEF_L1 = 2,
    /*!< Standard K-type Thermocouple temperature sensor with default
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_1_DEF_L2 = 8,
    /*!< Standard thermocouple temperature sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_2_DEF_L2 = 9,
    /*!< Standard thermocouple temperature sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_3_DEF_L2 = 10,
    /*!< Standard thermocouple temperature sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_4_DEF_L2 = 11,
    /*!< Standard thermocouple temperature sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_T_ADV_L1 = 16,
    /*!< T-type thermocouple temperature sensor with default linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_J_ADV_L1 = 17,
    /*!< T-type thermocouple temperature sensor with default linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_K_ADV_L1 = 18,
    /*!< T-type thermocouple temperature sensor with default linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_1_ADV_L2 = 24,
    /*!< Thermocouple temperature sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_2_ADV_L2 = 25,
    /*!< Thermocouple temperature sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_3_ADV_L2 = 26,
    /*!< Thermocouple temperature sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_4_ADV_L2 = 27,
    /*!< Thermocouple temperature sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_2WIRE_PT100_DEF_L1 = 32,
    /*!< Standard 2-wire PT100 RTD temperature sensor with default
     *   linearisation and default configuration options
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_2WIRE_PT1000_DEF_L1 = 33,
    /*!< Standard 2-wire PT1000 RTD temperature sensor with default
     *   linearisation and default configuration options
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_2WIRE_1_DEF_L2 = 40,
    /*!< 2-wire RTD temperature sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_2WIRE_2_DEF_L2 = 41,
    /*!< 2-wire RTD temperature sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_2WIRE_3_DEF_L2 = 42,
    /*!< 2-wire RTD temperature sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_2WIRE_4_DEF_L2 = 43,
    /*!< 2-wire RTD temperature sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_2WIRE_PT100_ADV_L1 = 48,
    /*!< Standard 2-wire PT100 RTD temperature sensor with default
     *   linearisation and advanced configuration options
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_2WIRE_PT1000_ADV_L1 = 49,
    /*!< Standard 2-wire PT1000 RTD temperature sensor with default
     *   linearisation and advanced configuration options
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_2WIRE_1_ADV_L2 = 56,
    /*!< 2-wire RTD temperature sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_2WIRE_2_ADV_L2 = 57,
    /*!< 2-wire RTD temperature sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_2WIRE_3_ADV_L2 = 58,
    /*!< 2-wire RTD temperature sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_2WIRE_4_ADV_L2 = 59,
    /*!< 2-wire RTD temperature sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_3WIRE_PT100_DEF_L1 = 64,
    /*!< Standard 3-wire PT100 RTD temperature sensor with default
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_3WIRE_PT1000_DEF_L1 = 65,
    /*!< Standard 3-wire PT1000 RTD temperature sensor with default
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_3WIRE_1_DEF_L2 = 72,
    /*!< 3-wire RTD temperature sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_3WIRE_2_DEF_L2 = 73,
    /*!< 3-wire RTD temperature sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_3WIRE_3_DEF_L2 = 74,
    /*!< 3-wire RTD temperature sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_3WIRE_4_DEF_L2 = 75,
    /*!< 3-wire RTD temperature sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_3WIRE_PT100_ADV_L1 = 80,
    /*!< Standard 3-wire PT100 RTD temperature sensor with default
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_3WIRE_PT1000_ADV_L1 = 81,
    /*!< Standard 3-wire PT1000 RTD temperature sensor with default
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_3WIRE_1_ADV_L2 = 88,
    /*!< 3-wire RTD temperature sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_3WIRE_2_ADV_L2 = 89,
    /*!< 3-wire RTD temperature sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_3WIRE_3_ADV_L2 = 90,
    /*!< 3-wire RTD temperature sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_3WIRE_4_ADV_L2 = 91,
    /*!< 3-wire RTD temperature sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_4WIRE_PT100_DEF_L1 = 96,
    /*!< Standard 4-wire PT100 RTD temperature sensor with default
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_4WIRE_PT1000_DEF_L1 = 97,
    /*!< Standard 4-wire PT1000 RTD temperature sensor with default
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_4WIRE_1_DEF_L2 = 104,
    /*!< 4-wire RTD temperature sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_4WIRE_2_DEF_L2 = 105,
    /*!< 4-wire RTD temperature sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_4WIRE_3_DEF_L2 = 106,
    /*!< 4-wire RTD temperature sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_4WIRE_4_DEF_L2 = 107,
    /*!< 4-wire RTD temperature sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_4WIRE_PT100_ADV_L1 = 112,
    /*!< Standard 4-wire PT100 RTD temperature sensor with default
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_4WIRE_PT1000_ADV_L1 = 113,
    /*!< Standard 4-wire PT1000 RTD temperature sensor with default
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_4WIRE_1_ADV_L2 = 120,
    /*!< 4-wire RTD temperature sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_4WIRE_2_ADV_L2 = 121,
    /*!< 4-wire RTD temperature sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_4WIRE_3_ADV_L2 = 122,
    /*!< 4-wire RTD temperature sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_4WIRE_4_ADV_L2 = 123,
    /*!< 4-wire RTD temperature sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMISTOR_A_10K_DEF_L1 = 128,
    /*!< Standard 10kOhm NTC Thermistor temperature sensor with Steinhart–Hart
     *   linearisation equation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMISTOR_B_10K_DEF_L1 = 129,
    /*!< Standard 10kOhm NTC Thermistor temperature sensor with Beta
     *   linearisation equation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMISTOR_1_DEF_L2 = 136,
    /*!< Thermistor sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMISTOR_2_DEF_L2 = 137,
    /*!< Thermistor sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMISTOR_3_DEF_L2 = 138,
    /*!< Thermistor sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMISTOR_4_DEF_L2 = 139,
    /*!< Thermistor sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMISTOR_A_10K_ADV_L1 = 144,
    /*!< 10kOhm NTC Thermistor temperature sensor with Steinhart–Hart
     *   linearisation equation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMISTOR_B_10K_ADV_L1 = 145,
    /*!< 10kOhm NTC Thermistor temperature sensor with Beta
     *   linearisation equation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMISTOR_1_ADV_L2 = 152,
    /*!< Thermistor sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMISTOR_2_ADV_L2 = 153,
    /*!< Thermistor sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMISTOR_3_ADV_L2 = 154,
    /*!< Thermistor sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMISTOR_4_ADV_L2 = 155,
    /*!< Thermistor sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_4WIRE_1_DEF_L2 = 168,
    /*!< Standard 4-wire Bridge Transducer sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_4WIRE_2_DEF_L2 = 169,
    /*!< Standard 4-wire Bridge Transducer sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_4WIRE_3_DEF_L2 = 170,
    /*!< Standard 4-wire Bridge Transducer sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_4WIRE_4_DEF_L2 = 171,
    /*!< Standard 4-wire Bridge Transducer sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_4WIRE_1_ADV_L2 = 184,
    /*!< Standard 4-wire Bridge Transducer sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_4WIRE_2_ADV_L2 = 185,
    /*!< Standard 4-wire Bridge Transducer sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_4WIRE_3_ADV_L2 = 186,
    /*!< Standard 4-wire Bridge Transducer sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_4WIRE_4_ADV_L2 = 187,
    /*!< Standard 4-wire Bridge Transducer sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_6WIRE_1_DEF_L2 = 200,
    /*!< Standard 6-wire Bridge Transducer sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_6WIRE_2_DEF_L2 = 201,
    /*!< Standard 6-wire Bridge Transducer sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_6WIRE_3_DEF_L2 = 202,
    /*!< Standard 6-wire Bridge Transducer sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_6WIRE_4_DEF_L2 = 203,
    /*!< Standard 6-wire Bridge Transducer sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_6WIRE_1_ADV_L2 = 216,
    /*!< Standard 6-wire Bridge Transducer sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_6WIRE_2_ADV_L2 = 217,
    /*!< Standard 6-wire Bridge Transducer sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_6WIRE_3_ADV_L2 = 218,
    /*!< Standard 6-wire Bridge Transducer sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_6WIRE_4_ADV_L2 = 219,
    /*!< Standard 6-wire Bridge Transducer sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_DIODE_2C_TYPEA_DEF_L1 = 224,
    /*!< Standard Diode two current temperature sensor with default
     *   linearisation equation and default configuration options
     *
     * @note For use with  Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_DIODE_3C_TYPEA_DEF_L1 = 225,
    /*!< Standard Diode three current temperature sensor with default
     *   linearisation equation and default configuration options
     *
     * @note For use with  Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_DIODE_2C_1_DEF_L2 = 232,
    /*!< Standard Diode two current sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with  Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_DIODE_3C_1_DEF_L2 = 233,
    /*!< Standard Diode three current sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with  Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_DIODE_2C_TYPEA_ADV_L1 = 240,
    /*!< Standard Diode two current temperature sensor with default
     *   linearisation equation and advanced configuration options
     *
     * @note For use with  Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_DIODE_3C_TYPEA_ADV_L1 = 241,
    /*!< Standard Diode three current sensor with default linearisation and
     *   advanced configuration options
     *
     * @note For use with  Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_DIODE_2C_1_ADV_L2 = 248,
    /*!< Standard Diode two current sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with  Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_DIODE_3C_1_ADV_L2 = 249,
    /*!< Standard Diode three current sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with  Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_MICROPHONE_A_DEF_L1 = 256,
    /*!< Generic microphone sensor without external amplifier, and with
     *   default linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_MICROPHONE_B_DEF_L1 = 257,
    /*!< Generic microphone sensor with external amplifier and bias, and with
     *   default linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_MICROPHONE_1_DEF_L2 = 264,
    /*!< Generic microphone sensor without external amplifier, and with
     *   user-defined linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_MICROPHONE_2_DEF_L2 = 265,
    /*!< Generic microphone sensor with external amplifier and bias, and with
     *   user-defined linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_MICROPHONE_A_ADV_L1 = 272,
    /*!< Generic microphone sensor without external amplifier, and with
     *   default linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_MICROPHONE_B_ADV_L1 = 273,
    /*!< Generic microphone sensor with external amplifier and bias, and with
     *   default linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_MICROPHONE_1_ADV_L2 = 278,
    /*!< Generic microphone sensor without external amplifier, and with
     *   user-defined linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_MICROPHONE_2_ADV_L2 = 279,
    /*!< Generic microphone sensor with external amplifier and bias, and with
     *   user-defined linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_VOLTAGE = 512,
    /*!< Generic voltage sensor with no linearisation applied
     *
     * @note For use with Analog 0-10V Voltage Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_VOLTAGE_PRESSURE_A_DEF_L1 = 544,
    /*!< Honeywell Pressure voltage sensor (HSCMRNN1.6BAAA3) with default
     *   linearisation and default configuration options
     *
     * @note For use with Analog 0-10V Voltage Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_VOLTAGE_PRESSURE_B_DEF_L1 = 545,
    /*!< Amphenol Pressure voltage sensor (NPA-300B-015A) with default
     *   linearisation and default configuration options
     *
     * @note For use with Analog 0-10V Voltage Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_VOLTAGE_PRESSURE_1_DEF_L2 = 552,
    /*!< Generic pressure voltage sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog 0-10V Voltage Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_VOLTAGE_PRESSURE_2_DEF_L2 = 553,
    /*!< Generic pressure voltage sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog 0-10V Voltage Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_VOLTAGE_PRESSURE_A_ADV_L1 = 560,
    /*!< Honeywell Pressure voltage sensor (HSCMRNN1.6BAAA3) with default
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog 0-10V Voltage Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_VOLTAGE_PRESSURE_B_ADV_L1 = 561,
    /*!< Amphenol Pressure voltage sensor (NPA-300B-015A) with default
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog 0-10V Voltage Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_VOLTAGE_PRESSURE_1_ADV_L2 = 568,
    /*!< Generic pressure voltage sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog 0-10V Voltage Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_VOLTAGE_PRESSURE_2_ADV_L2 = 569,
    /*!< Generic pressure voltage sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog 0-10V Voltage Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_CURRENT = 768,
    /*!< Generic current sensor with no linearisation applied
     *
     * @note For use with Analog 4-20mA Current Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_CURRENT_PRESSURE_A_DEF_L1 = 800,
    /*!< Honeywell Pressure current sensor (PX2CN2XX100PACH) with default
     *   linearisation and default configuration options
     *
     * @note For use with Analog 4-20mA Current Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_CURRENT_PRESSURE_1_DEF_L2 = 808,
    /*!< Generic pressure current sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog 4-20mA Current Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_CURRENT_PRESSURE_2_DEF_L2 = 809,
    /*!< Generic pressure current sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog 4-20mA Current Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_CURRENT_PRESSURE_A_ADV_L1 = 816,
    /*!< Honeywell Pressure current sensor (PX2CN2XX100PACH) with default
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog 4-20mA Current Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_CURRENT_PRESSURE_1_ADV_L2 = 824,
    /*!< Generic pressure current sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog 4-20mA Current Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_CURRENT_PRESSURE_2_ADV_L2 = 825,
    /*!< Generic pressure current sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog 4-20mA Current Sensor channels only
     */
} ADI_SENSE_1000_ADC_SENSOR_TYPE;

/*! ADSNS1000 I2C digital sensor type options
 *
 *  Select the sensor type that is connected to an I2C digital measurement
 *  channel.
 *
 *  @note These are pre-defined sensors using built-in linearisation data
 */
typedef enum {
    ADI_SENSE_1000_I2C_SENSOR_HUMIDITY_A_DEF_L1 = 2112,
    /*!< Honeywell HiH9000-series humidity sensor with default linearisation
     *   and default configuration options
     *
     * @note For use with I2C Digital Sensor channels only
     */
    ADI_SENSE_1000_I2C_SENSOR_HUMIDITY_B_DEF_L1 = 2113,
    /*!< Sensirion SHT35-DIS-B humidity sensor with default linearisation
     *   and default configuration options
     *
     * @note For use with I2C Digital Sensor channels only
     */
    ADI_SENSE_1000_I2C_SENSOR_HUMIDITY_A_DEF_L2 = 2120,
    /*!< Honeywell HiH9000-series humidity sensor with user-defined linearisation
     *   and default configuration options
     *
     * @note For use with I2C Digital Sensor channels only
     */
    ADI_SENSE_1000_I2C_SENSOR_HUMIDITY_B_DEF_L2 = 2121,
    /*!< Sensirion SHT35-DIS-B humidity sensor with user-defined linearisation
     *   and default configuration options
     *
     * @note For use with I2C Digital Sensor channels only
     */
    ADI_SENSE_1000_I2C_SENSOR_HUMIDITY_A_ADV_L1 = 2128,
    /*!< Honeywell HiH9000-series humidity sensor with default linearisation
     *   and advanced configuration options
     *
     * @note For use with I2C Digital Sensor channels only
     */
    ADI_SENSE_1000_I2C_SENSOR_HUMIDITY_B_ADV_L1 = 2129,
    /*!< Sensirion SHT35-DIS-B humidity sensor with default linearisation
     *   and advanced configuration options
     *
     * @note For use with I2C Digital Sensor channels only
     */
    ADI_SENSE_1000_I2C_SENSOR_HUMIDITY_A_ADV_L2 = 2136,
    /*!< Honeywell HiH9000-series humidity sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with I2C Digital Sensor channels only
     */
    ADI_SENSE_1000_I2C_SENSOR_HUMIDITY_B_ADV_L2 = 2137,
    /*!< Sensirion SHT35-DIS-B humidity sensor with user-defined linearisation
     *   and advanced configuration options
     *
     * @note For use with I2C Digital Sensor channels only
     */
    ADI_SENSE_1000_I2C_SENSOR_AMBIENTLIGHT_A_DEF_L1 = 2176,
    /*!< ON-Semiconductor NOA1305 ambient light sensor with default
     *   linearisation and default configuration options
     *
     * @note For use with I2C Digital Sensor channels only
     */
    ADI_SENSE_1000_I2C_SENSOR_AMBIENTLIGHT_A_DEF_L2 = 2184,
    /*!< ON-Semiconductor NOA1305 ambient light sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with I2C Digital Sensor channels only
     */
    ADI_SENSE_1000_I2C_SENSOR_AMBIENTLIGHT_A_ADV_L1 = 2192,
    /*!< ON-Semiconductor NOA1305 ambient light sensor with default
     *   linearisation and advanced configuration options
     *
     * @note For use with I2C Digital Sensor channels only
     */
    ADI_SENSE_1000_I2C_SENSOR_AMBIENTLIGHT_A_ADV_L2 = 2200,
    /*!< ON-Semiconductor NOA1305 ambient light sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with I2C Digital Sensor channels only
     */
} ADI_SENSE_1000_I2C_SENSOR_TYPE;

/*! ADSNS1000 SPI digital sensor type options
 *
 *  Select the sensor type that is connected to an SPI digital measurement
 *  channel.
 *
 *  @note These are pre-defined sensors using built-in linearisation data
 */
typedef enum {
    ADI_SENSE_1000_SPI_SENSOR_PRESSURE_A_DEF_L1 = 3072,
    /*!< Honeywell HSCDRNN1.6BASA3 pressure sensor with default linearisation
     * and default configuration options
     *
     * @note For use with SPI Digital Sensor channels only
     */
    ADI_SENSE_1000_SPI_SENSOR_PRESSURE_A_DEF_L2 = 3080,
    /*!< Honeywell HSCDRNN1.6BASA3 pressure sensor with user-defined
     * linearisation and default configuration options
     *
     * @note For use with SPI Digital Sensor channels only
     */
    ADI_SENSE_1000_SPI_SENSOR_PRESSURE_A_ADV_L1 = 3088,
    /*!< Honeywell HSCDRNN1.6BASA3 pressure sensor with default linearisation
     * and advanced configuration options
     *
     * @note For use with SPI Digital Sensor channels only
     */
    ADI_SENSE_1000_SPI_SENSOR_PRESSURE_A_ADV_L2 = 3096,
    /*!< Honeywell HSCDRNN1.6BASA3 pressure sensor with user-defined
     * linearisation and advanced configuration options
     *
     * @note For use with SPI Digital Sensor channels only
     */
    ADI_SENSE_1000_SPI_SENSOR_ACCELEROMETER_A_DEF_L1 = 3200,
    /*!< Analog Devices ADxL362 3-axis accelerometer sensor with default
     *   linearisation and default configuration options(*)
     *
     * @note For use with SPI Digital Sensor channels only
     *
     * @note This sensor requires the use of 3 SPI Digital Sensor channels, with
     *       the sensor measurements from the X/Y/Z axes each output on a
     *       seperate dedicated channel (SPI#0/SPI#1/SPI#2, respectively)
     */
    ADI_SENSE_1000_SPI_SENSOR_ACCELEROMETER_B_DEF_L1 = 3201,
    /*!< Analog Devices ADxL355 3-axis accelerometer sensor with default
     *   linearisation and default configuration options(*)
     *
     * @note For use with SPI Digital Sensor channels only
     *
     * @note This sensor requires the use of 3 SPI Digital Sensor channels, with
     *       the sensor measurements from the X/Y/Z axes each output on a
     *       seperate dedicated channel (SPI#0/SPI#1/SPI#2, respectively)
     */
    ADI_SENSE_1000_SPI_SENSOR_ACCELEROMETER_A_DEF_L2 = 3208,
    /*!< Analog Devices ADxL362 3-axis accelerometer sensor with user-defined
     *   linearisation and default configuration options(*)
     *
     * @note For use with SPI Digital Sensor channels only
     *
     * @note This sensor requires the use of 3 SPI Digital Sensor channels, with
     *       the sensor measurements from the X/Y/Z axes each output on a
     *       seperate dedicated channel (SPI#0/SPI#1/SPI#2, respectively)
     */
    ADI_SENSE_1000_SPI_SENSOR_ACCELEROMETER_B_DEF_L2 = 3209,
    /*!< Analog Devices ADxL355 3-axis accelerometer sensor with user-defined
     *   linearisation and default configuration options(*)
     *
     * @note For use with SPI Digital Sensor channels only
     *
     * @note This sensor requires the use of 3 SPI Digital Sensor channels, with
     *       the sensor measurements from the X/Y/Z axes each output on a
     *       seperate dedicated channel (SPI#0/SPI#1/SPI#2, respectively)
     */
    ADI_SENSE_1000_SPI_SENSOR_ACCELEROMETER_A_ADV_L1 = 3216,
    /*!< Analog Devices ADxL362 3-axis accelerometer sensor with default
     *   linearisation and advanced configuration options(*)
     *
     * @note For use with SPI Digital Sensor channels only
     *
     * @note This sensor requires the use of 3 SPI Digital Sensor channels, with
     *       the sensor measurements from the X/Y/Z axes each output on a
     *       seperate dedicated channel (SPI#0/SPI#1/SPI#2, respectively)
     */
    ADI_SENSE_1000_SPI_SENSOR_ACCELEROMETER_B_ADV_L1 = 3217,
    /*!< Analog Devices ADxL355 3-axis accelerometer sensor with default
     *   linearisation and advanced configuration options(*)
     *
     * @note For use with SPI Digital Sensor channels only
     *
     * @note This sensor requires the use of 3 SPI Digital Sensor channels, with
     *       the sensor measurements from the X/Y/Z axes each output on a
     *       seperate dedicated channel (SPI#0/SPI#1/SPI#2, respectively)
     */
    ADI_SENSE_1000_SPI_SENSOR_ACCELEROMETER_A_ADV_L2 = 3224,
    /*!< Analog Devices ADxL362 3-axis accelerometer sensor with user-defined
     *   linearisation and advanced configuration options(*)
     *
     * @note For use with SPI Digital Sensor channels only
     *
     * @note This sensor requires the use of 3 SPI Digital Sensor channels, with
     *       the sensor measurements from the X/Y/Z axes each output on a
     *       seperate dedicated channel (SPI#0/SPI#1/SPI#2, respectively)
     */
    ADI_SENSE_1000_SPI_SENSOR_ACCELEROMETER_B_ADV_L2 = 3225,
    /*!< Analog Devices ADxL355 3-axis accelerometer sensor with user-defined
     *   linearisation and advanced configuration options(*)
     *
     * @note For use with SPI Digital Sensor channels only
     *
     * @note This sensor requires the use of 3 SPI Digital Sensor channels, with
     *       the sensor measurements from the X/Y/Z axes each output on a
     *       seperate dedicated channel (SPI#0/SPI#1/SPI#2, respectively)
     */
} ADI_SENSE_1000_SPI_SENSOR_TYPE;

/*! ADSNS1000 UART digital sensor type options
 *
 *  Select the sensor type that is connected to an UART digital measurement
 *  channel.
 *
 *  @note These are pre-defined sensors using built-in linearisation data
 */
typedef enum {
    ADI_SENSE_1000_UART_SENSOR_UART_CO2_A_DEF_L1 = 3584,
    /*!< COZIR AMB 2,000 ppm CO2 Sensor with default linearisation
     * and default configuration options
     *
     * @note For use with UART Digital Sensor channels only
     */
    ADI_SENSE_1000_UART_SENSOR_UART_CO2_B_DEF_L1 = 3585,
    /*!< COZIR LP Miniature 5,000ppm CO2 Sensor with default linearisation
     * and default configuration options
     *
     * @note For use with UART Digital Sensor channels only
     */
    ADI_SENSE_1000_UART_SENSOR_UART_CO2_A_DEF_L2 = 3592,
    /*!< COZIR AMB 2,000 ppm CO2 Sensor with user-defined linearisation
     * and default configuration options
     *
     * @note For use with UART Digital Sensor channels only
     */
    ADI_SENSE_1000_UART_SENSOR_UART_CO2_B_DEF_L2 = 3593,
    /*!< COZIR LP Miniature 5,000ppm CO2 Sensor with user-defined linearisation
     * and default configuration options
     *
     * @note For use with UART Digital Sensor channels only
     */
    ADI_SENSE_1000_UART_SENSOR_UART_CO2_A_ADV_L1 = 3600,
    /*!< COZIR AMB 2,000 ppm CO2 Sensor with default linearisation
     * and advanced configuration options
     *
     * @note For use with UART Digital Sensor channels only
     */
    ADI_SENSE_1000_UART_SENSOR_UART_CO2_B_ADV_L1 = 3601,
    /*!< COZIR LP Miniature 5,000ppm CO2 Sensor with default linearisation
     * and advanced configuration options
     *
     * @note For use with UART Digital Sensor channels only
     */
    ADI_SENSE_1000_UART_SENSOR_UART_CO2_A_ADV_L2 = 3608,
    /*!< COZIR AMB 2,000 ppm CO2 Sensor with user-defined linearisation
     * and advanced configuration options
     *
     * @note For use with UART Digital Sensor channels only
     */
    ADI_SENSE_1000_UART_SENSOR_UART_CO2_B_ADV_L2 = 3609,
    /*!< COZIR LP Miniature 5,000ppm CO2 Sensor with user-defined linearisation
     * and advanced configuration options
     *
     * @note For use with UART Digital Sensor channels only
     */
} ADI_SENSE_1000_UART_SENSOR_TYPE;

#ifdef __cplusplus
}
#endif

/*!
 * @}
 */

#endif /* __ADI_SENSE_1000_SENSOR_TYPES_H__ */
