/*
CONFIDENTIAL AND PROPRIETARY INFORMATION

Copyright (c) 2018 Emutex Ltd. All rights reserved.
This software and documentation contain confidential and
proprietary information that is the property of
Emutex Ltd. The software and documentation are
furnished under a license agreement and may be used
or copied only in accordance with the terms of the license
agreement. No part of the software and documentation
may be reproduced, transmitted, or translated, in any
form or by any means, electronic, mechanical, manual,
optical, or otherwise, without prior written permission
of Emutex Ltd., or as expressly provided by the license agreement.
Reverse engineering is prohibited, and reproduction,
disclosure or use without specific written authorization
of Emutex Ltd. is strictly forbidden.
 *
Copyright 2017 (c) Analog Devices, Inc.

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
  - Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  - Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in
    the documentation and/or other materials provided with the
    distribution.
  - Neither the name of Analog Devices, Inc. nor the names of its
    contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.
  - The use of this software may or may not infringe the patent rights
    of one or more patent holders. This license does not release you
    from the requirement that you obtain separate licenses from these
    patent holders to use this software.
  - Use of the software either in source or binary form, must be run
    on or directly connected to an Analog Devices Inc. component.

THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, INTELLECTUAL PROPERTY RIGHTS, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *****************************************************************************/
/*!
 ******************************************************************************
 * @file:
 *-----------------------------------------------------------------------------
 *
 */
#include "mbed.h"
#include "inc/adi_sense_api.h"
#include "inc/adi_sense_1000/adi_sense_1000_api.h"
#include "inc/adi_sense_log.h"
#include "common/utils.h"

extern ADI_SENSE_CONFIG babyseat_conf;
extern ADI_SENSE_1000_LUT_DESCRIPTOR *sample_lut_desc_list[];
extern ADI_SENSE_1000_LUT_TABLE_DATA *sample_lut_data_list[];
extern unsigned                       sample_lut_num_tables;


/* Change the following pointer to select any of the configurations above */
static ADI_SENSE_CONFIG *pSelectedConfig = &babyseat_conf;

static ADI_SENSE_CONNECTION connectionInfo = {
    .type = ADI_SENSE_CONNECTION_TYPE_SPI,
    .spi = {
        .mosiPin    = SPI_MOSI,
        .misoPin    = SPI_MISO,
        .sckPin     = SPI_SCK,
        .csPin      = D10,
        .maxSpeedHz = 2000000,
    },
    .gpio = {
        .resetPin     = D6,
        .errorPin     = D3,
        .alertPin     = D4,
        .datareadyPin = D5,
    },
    .log = {
        .txPin        = PA_11,
        .rxPin        = PA_12,
        .baudRate     = 115200,
        .disableLogs  = false,
    },
};

int main()
{
    ADI_SENSE_RESULT res;
    ADI_SENSE_STATUS status;
    ADI_SENSE_DEVICE_HANDLE hDevice;
    ADI_SENSE_MEASUREMENT_MODE eMeasurementMode = ADI_SENSE_MEASUREMENT_MODE_NORMAL;
    bool_t bDeviceReady;

    /*
     * Open an ADI Sense device instance.
     */
    res = adi_sense_Open(0, &connectionInfo, &hDevice);
    if (res != ADI_SENSE_SUCCESS)
    {
        ADI_SENSE_LOG_ERROR("Failed to open device instance");
        return res;
    }

    /*
     * Reset the given ADI Sense device....
     */
    ADI_SENSE_LOG_INFO("Resetting ADI Sense device, please wait...");
    res = adi_sense_Reset(hDevice);
    if (res != ADI_SENSE_SUCCESS)
    {
        ADI_SENSE_LOG_ERROR("Failed to reset device");
        return res;
    }
    /*
     * ...and wait until the device is ready.
     */
    do {
        wait_ms(100);
        res = adi_sense_GetDeviceReadyState(hDevice, &bDeviceReady);
        if (res != ADI_SENSE_SUCCESS)
        {
            ADI_SENSE_LOG_ERROR("Failed to get device ready-state");
            return res;
        }
    } while (! bDeviceReady);
    ADI_SENSE_LOG_INFO("ADI Sense device ready");

  /*
     * Assemble the list of user-defined Look-Up Tables from sample_lut_data.c
     * into the single contiguous data format required by the device.
     */
    unsigned lutBufferSize = ADI_SENSE_LUT_MAX_SIZE;
    ADI_SENSE_1000_LUT *pLutBuffer = (ADI_SENSE_1000_LUT *) ::operator new (lutBufferSize);
    if (pLutBuffer == NULL)
    {
        ADI_SENSE_LOG_ERROR("Failed to allocate memory for user-defined LUT data buffer");
        return ADI_SENSE_NO_MEM;
    }

    ADI_SENSE_LOG_INFO("Assembling LUT data");
    res = adi_sense_1000_AssembleLutData(pLutBuffer, lutBufferSize,
                                         sample_lut_num_tables,
                                         sample_lut_desc_list,
                                         sample_lut_data_list);
    if (res != ADI_SENSE_SUCCESS)
    {
        ADI_SENSE_LOG_ERROR("Failed to assemble user-defined LUT data");
        return res;
    }

    /*
     * Write assembled user-defined Look-Up Table data structure to the device
     * User-defined LUT data is not applied until adi_sense_ApplyConfigUpdates() is called.
     */
    ADI_SENSE_LOG_INFO("Setting LUT data");
    res = adi_sense_1000_SetLutData(hDevice, pLutBuffer);
    if (res != ADI_SENSE_SUCCESS)
    {
        ADI_SENSE_LOG_ERROR("Failed to set user-defined LUT data");
        return res;
    }

    delete pLutBuffer;
    /*
     * Write configuration settings to the device registers.
     * Settings are not applied until adi_sense_ApplyConfigUpdates() is called.
     */
    ADI_SENSE_LOG_INFO("Setting device configuration");
    res = adi_sense_SetConfig(hDevice, pSelectedConfig);
    if (res != ADI_SENSE_SUCCESS)
    {
        ADI_SENSE_LOG_ERROR("Failed to set device configuration");
        return res;
    }
    res = adi_sense_ApplyConfigUpdates(hDevice);
    if (res != ADI_SENSE_SUCCESS)
    {
        ADI_SENSE_LOG_ERROR("Failed to apply device configuration");
        return res;
    }
    /*
     * Check device status after updating the configuration
     */
    res = adi_sense_GetStatus(hDevice, &status);
    if (res != ADI_SENSE_SUCCESS)
    {
        ADI_SENSE_LOG_ERROR("Failed to retrieve device status");
        return res;
    }
    if (status.deviceStatus &
        (ADI_SENSE_DEVICE_STATUS_ERROR | ADI_SENSE_DEVICE_STATUS_ALERT))
    {
        utils_printStatus(&status);
    }

    /*
     * Kick off the measurement cycle here
     */
    ADI_SENSE_LOG_INFO("Configuration completed, starting measurement cycles");
    utils_runMeasurement(hDevice, eMeasurementMode);

    /*
     * Clean up and exit
     */
    res = adi_sense_Close(hDevice);
    if (res != ADI_SENSE_SUCCESS)
    {
        ADI_SENSE_LOG_ERROR("Failed to close device instance");
        return res;
    }

    return 0;
}
