#include "mbed.h"
#include "SakuraIO.h"

Serial pc(USBTX, USBRX);
DigitalOut myled(LED1);

#if defined(TARGET_NUCLEO_F042K6) || defined(TARGET_NUCLEO_F303K8)
// Nucleo-32 series
SPI spi(A6, A5, A4); // mosi, miso, sclk
DigitalOut cs(A3);
I2C i2c(D4, D5);        // sda, scl
#elif defined(TARGET_NUCLEO_F401RE) || defined(TARGET_NUCLEO_F411RE) || defined(TARGET_NUCLEO_F446RE) || defined(TARGET_NUCLEO_L476RG)
// Nucleo-64 series
SPI spi(D11, D12, D13); // mosi, miso, sclk
DigitalOut cs(D10);
I2C i2c(D14, D15);
#elif defined(TARGET_SAKURAIO_EVB_01)
// sakura.io Evaluation Board(SCO-EVB-01)
SPI spi(PB_15, PB_14, PB_13); // mosi, miso, sclk
DigitalOut cs(PB_12);
I2C i2c(PC_9, PA_8);
#else
// LPC1768 or others.
SPI spi(p5, p6, p7); // mosi, miso, sclk
DigitalOut cs(p8);
I2C i2c(p9, p10);        // sda, scl
#endif

//SakuraIO_SPI sakuraio( spi, cs );
SakuraIO_I2C sakuraio(i2c);

uint8_t updateFirmware()
{
    uint8_t ret;
    char version[33] = {0};

    wait(1);

    // Check module
    uint16_t productId = sakuraio.getProductID();
    switch(productId) {
        case 0x0001:
            pc.printf("SCM-LTE-Beta\r\n");
            break;
        case 0x0002:
            pc.printf("SCM-LTE-01\r\n");
            break;
        default:
            pc.printf("Please check connection %d\r\n", productId);
            return 1;
    }
    
    
    pc.printf("Get current version\r\n");
    ret = sakuraio.getFirmwareVersion(version);
    if((ret = sakuraio.getFirmwareVersion(version)) != CMD_ERROR_NONE) {
        pc.printf("Error code=%d\r\n", ret);
        return 1;
    }
    pc.printf("Current: %s\r\n", version);


    // Waiting for online
    pc.printf("Waiting for online\r\n");
    while(1) {
        pc.printf(".");
        wait_ms(2000);
        if((sakuraio.getConnectionStatus() & 0x80) != 0x00) {
            break;
        }
    }
    pc.printf("\r\nOnline\r\n");


    wait_ms(1000);


    // Request unlock
    pc.printf("Unlock\r\n");
    if((ret = sakuraio.unlock()) != CMD_ERROR_NONE) {
        pc.printf("Error code=%d\r\n", ret);
        return 1;
    }

    wait_ms(1000);

    // Request firmware update
    pc.printf("Starting update\r\n");
    if((ret = sakuraio.updateFirmware()) != CMD_ERROR_NONE) {
        pc.printf("Error code=%d\r\n", ret);
        return 1;
    }

    wait_ms(1000);

    // Check update status
    uint8_t errCode = 0x00;
    pc.printf("Waiting for update\r\n");
    while(1) {

        wait_ms(1000);
        pc.printf(".");

        uint8_t updateStatus = sakuraio.getFirmwareUpdateStatus();
        if(updateStatus == 0xff || (updateStatus & 0x80) != 0x00) {
            continue;
        } else if(updateStatus == 0x00) {
            // Success
            break;
        } else {
            // Error
            errCode = updateStatus & 0x3f;
            break;
        }
    }

    pc.printf("\r\n");
    switch(errCode) {
        case 0x00:
            return 0;
        case 0x01:
            pc.printf("Already updated\r\n");
            return 0;
        default:
            pc.printf("Error code=%d\r\n", errCode);
            break;
    }

    return 1;
}

int main()
{
    if(updateFirmware() == 0) {
        pc.printf("Update successfull\r\n");
    } else {
        pc.printf("Update failure\r\n");
    }
    while(1) {
        myled = !myled;
        wait(1);
    }

}
