#include "Custom_UBloxGPS/neo-m8m.h"

#define     NEOM8M_ADR_GPS          0x84
#define     NEOM8M_REG_GPS_LENH     0xFD
#define     NEOM8M_REG_GPS_LENL     0xFE
#define     NEOM8M_REG_GPS_DATA     0xFE

char NeoM8M::sleepCommand[12] {0xB5, 0x62, 0x06, 0x04, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00};
char NeoM8M::wakeCommand[12] {0xB5, 0x62, 0x06, 0x04, 0x04, 0x00, 0x00, 0x00, 0x09, 0x00};
char NeoM8M::resetDefaults[21] {0xB5, 0x62, 0x06, 0x09, 0x0D, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0x03}; // UBX-CFG-CFG -- Reset to Defaults (also leave 2 bytes)

// COSNTRUCTOR

/**
    Basic, default constructor for the NeoM8M object
 */
NeoM8M::NeoM8M()
{
    
}

/**
    Constructor for the NeoM8M object where the I2C Object is passed in
 */
NeoM8M::NeoM8M(I2C *_i2c)
{
    i2c = _i2c;
}

// DECONSTRUCTOR

/**
    Basic, default deconstructor for the NeoM8M object
 */
NeoM8M::~NeoM8M()
{

}

// PUBLIC METHODS

/**
    Initialization method for the NeoM8M object.
    
    Reset GPS module to default status.
    
    Object takes in:
        -Pointer to Buffer
        -Pointer to Position Tracker
        -Pointer to MyTime storage variable
        -Pointer to MyLat storage variable
        -Pointer to MyLon storage variable
    And sets variables.
    
    GPS Read is performed in order to get the time
 */
void NeoM8M::init(char *_buf, uint8_t *_pos, uint32_t *_mytime, double *_mylat, double *_mylon)
{
    printf("In gpsInitSequence()... Resetting Defaults... \r\n");
    
    // Setup Commands    
    executeCommand(resetDefaults, sizeof(resetDefaults));
    
    // Setting variables necessary for future reads
    buf = _buf;
    pos = _pos;
    mytime = _mytime;
    mylat = _mylat;
    mylon = _mylon;
    
    read();   
}

/**
    Setter for the NeoM8M's I2C Variable
 */
void NeoM8M::setI2C(I2C *_i2c)
{
    i2c = _i2c;
}

/**
    GPS Read method.
    
    
 */
int NeoM8M::read()
{
    bool gpsDone = false;
    bool fixGood = false;
    
    uint8_t crcPass = 0;
    uint8_t crcFail = 0;
    
    uint32_t tDate = 0;
    uint32_t tTime = 0;
    
    char ret = 0xFF;
    
    char cmd[2];
    cmd[0] = 0xFF;
    
    i2c->write(NEOM8M_ADR_GPS, cmd, 1);
    while(!gpsDone)
    {
        while (ret == 0xFF)
        {
            i2c->read(NEOM8M_ADR_GPS, &ret, 1);
        }   
        while (ret != 0xFF)
        {
            buf[*pos++] = ret;
            i2c->read(NEOM8M_ADR_GPS, &ret, 1);
            if (ret == '\r')
            {
                i2c->read(NEOM8M_ADR_GPS, &ret, 1);
                if (ret == '\n')
                {
                    buf[*pos] = 0x00;
                    // NMEA Validation
                    uint16_t crc = 0;
                    char clr = '\0';
                    
                    if (buf[0] == '$' && buf[*pos-3] == '*')
                    {
                        for (int i = 1; i < *pos-3; i++)
                        {
                            crc ^= buf[i];
                        }
                    }
                    
                    if (crc == ((buf[*pos-2] < 'A' ? buf[*pos-2] - '0' : buf[*pos-2] - '7') << 4 | (buf[*pos-1] < 'A' ? buf[*pos-1] - '0' : buf[*pos-1] - '7')))
                    {
                        clr = '2'; // 2 = Green
                        crcPass++;
                    }
                    
                    else
                    {
                        clr = '1'; // 1 = Red
                        crcFail++;
                    }
                    
                    printf("GPS: [\u001b[3%cm%02X\u001b[0m] |%s|\r\n", clr, crc, buf);
                    
                    if (clr == '2')
                    {
                        // Global Positioning System Fix Data
                        if(strncmp(buf, "$GNGGA", 6) == 0)
                        {
                            printf("GNGGA> ");
                            float fldTim, fldAlt;
                            double fldLat, fldLon;
                            char fldN_S, fldE_W;
                            int fldFix, fldSat;
                            if (sscanf(buf, "$GNGGA,%f,%lf,%c,%lf,%c,%d,%d,%*f,%f", &fldTim, &fldLat, &fldN_S, &fldLon, &fldE_W, &fldFix, &fldSat, &fldAlt) == 8)
                                printf("Sec: %.2f, Lat: %.5f %c, Lon: %.5f %c, Fix: %d, Sat: %d, Alt: %.1f M\r\n", fldTim, fldLat, fldN_S, fldLon, fldE_W, fldFix, fldSat, fldAlt);
                            if (clr == '2')
                            {
                                *mylat = fldLat / (fldN_S == 'S' ? -100 : 100);
                                *mylon = fldLon / (fldE_W == 'W' ? -100 : 100);
                            }
                        }
                        
                        // Satellite status
                        if(strncmp(buf, "$GNGSA", 6) == 0)
                        {
                            char fldTyp;
                            int fldDim, fldSat;
                            sscanf(buf, "$GNGSA,%c,%d,%d", &fldTyp, &fldDim, &fldSat);
                        }
                        
                        // Geographic position, Latitude and Longitude
                        if(strncmp(buf, "$GNGLL", 6) == 0)
                        {
                            float fldTim;
                            double fldLat, fldLon;
                            char fldN_S, fldE_W;
                            sscanf(buf, "$GNGLL,%lf,%c,%lf,%c,%f", &fldLat, &fldN_S, &fldLon, &fldE_W, &fldTim);
                        }
                        
                        // Geographic position, Latitude and Longitude
                        if(strncmp(buf, "$GNRMC", 6) == 0)
                        {
                            printf("GNRMC> ");
                            float fldTim, fldSpd, fldTrk;
                            fldTrk = 0;
                            double fldLat, fldLon;
                            char fldSts, fldN_S, fldE_W;
                            uint32_t fldDat;
                            if (sscanf(buf, "$GNRMC,,%c", &fldSts) != 1 && 
                                sscanf(buf, "$GNRMC,%f,%c,,,,,,,%d", &fldTim, &fldSts, &fldDat) != 3 && 
                                sscanf(buf, "$GNRMC,%f,%c,%lf,%c,%lf,%c,%f,,%d", &fldTim, &fldSts, &fldLat, &fldN_S, &fldLon, &fldE_W, &fldSpd, &fldDat) != 8 &&
                                sscanf(buf, "$GNRMC,%f,%c,%lf,%c,%lf,%c,%f,%f,%d", &fldTim, &fldSts, &fldLat, &fldN_S, &fldLon, &fldE_W, &fldSpd, &fldTrk, &fldDat) != 9)
                                printf("[\u001b[33mWARN\u001b[0m] Invalid GNRMC packet detected.\r\n");
                            
                            if (fldN_S == 'N' || fldN_S == 'S' && fldE_W == 'E' || fldE_W == 'W')
                                printf("Sec: %.2f, Sts: %c, Lat: %.5f %c, Lon: %.5f %c, Spd: %.3f, Dat: %06d\r\n", fldTim, fldSts, fldLat, fldN_S, fldLon, fldE_W, fldSpd, fldDat);
                            else
                                printf("\r\n");
                                
                            if (clr == '2')
                            {
                                tTime = (uint32_t)fldTim;
                                tDate = fldDat;
                                if (fldSts == 'A')
                                    fixGood = true;
                            }
                        }
                    }
                    
                    *pos = 0;
                    i2c->read(NEOM8M_ADR_GPS, &ret, 1);
                }
                else
                {
                    printf("[\u001b[33mWARN\u001b[0m] Expected '0A', received '%02X'.\r\n", ret);
                }
            }
            
            //else if (pos == 82)
            else if (*pos == 191)
            {
                buf[*pos] = 0x00;
                printf("GPS: |%s| ...\r\n", buf);
                *pos = 0;
                i2c->read(NEOM8M_ADR_GPS, &ret, 1);
            }
        }
        
        buf[*pos] = 0x00;
        gpsDone = true;
    }
    
    if (*pos > 0)
        printf("GPS: |%s|\r\n", buf);

    if (crcFail)
        printf("[\u001b[33mWARN\u001b[0m] CRC PASS: %d FAIL: %d\r\n", crcPass, crcFail);

    struct tm ts;
    time_t t;
    
    uint8_t tDay = tDate / 10000;
    uint8_t tMon = (tDate - (tDay * 10000)) / 100;
    uint8_t tYear = (tDate - ((tDay * 10000) + (tMon * 100))) + 100;
    uint8_t tHour = tTime / 10000;
    uint8_t tMin = (tTime - (tHour * 10000)) / 100;
    uint8_t tSec = (tTime - ((tHour * 10000) + (tMin * 100)));
    
    ts.tm_year = tYear;
    ts.tm_mon = tMon - 1;
    ts.tm_mday = tDay;
    ts.tm_hour = tHour;
    ts.tm_min = tMin;
    ts.tm_sec = tSec;
    t = mktime(&ts);

    printf("GPS: %08X\t%s", t, ctime(&t));
    *mytime = t;

    if (fixGood)
        return 0;
    else
        return 1;
}

/**
    Places the NeoM8M GPS Module to sleep by calling inner private method
 */        
void NeoM8M::sleep()
{
    return executeCommand(sleepCommand, sizeof(sleepCommand));
}

/**
    Wakes the NeoM8M GPS Module up by calling inner private method
 */
void NeoM8M::wake()
{
    return executeCommand(wakeCommand, sizeof(wakeCommand));
}

/**
    
 */
void NeoM8M::ubxRead()
{
    uint8_t crcPass = 0;
    uint8_t crcFail = 0;
    
    *pos = 0;
    char ret = 0xFF;
    
    char cmd[2];
    cmd[0] = 0xFF;
    
    i2c->write(NEOM8M_ADR_GPS, cmd, 1);
    
    while (ret == 0xFF)
    {
        i2c->read(NEOM8M_ADR_GPS, &ret, 1);
    }
    
    while (ret != 0xFF)
    {
        buf[*pos++] = ret;
        i2c->read(NEOM8M_ADR_GPS, &ret, 1);
    }
    
    printf("UBX: | ");
    
    for (int i = 0; i <= *pos; i++)
        printf("%02X ", buf[i]);
    printf("|\r\n", buf);
    
}

// PRIVATE METHODS

void NeoM8M::executeCommand(char *command, int length)
{
    uint8_t crcA = 0;
    uint8_t crcB = 0;
    
    for(int i = 2; i <= (length - 3); i++)
    {
        crcA = crcA + command[i];
        crcB = crcB + crcA;
    }
    
    command[(length - 2)] = crcA;
    command[(length - 1)] = crcB;
    
    printf("UBX CRC: %02X %02X\r\n", crcA, crcB);
    
    i2c->write(NEOM8M_ADR_GPS, command, length);
    ubxRead();
    
    printf("Command Executed \r\n");
}