/* Rohm SensorShield-EVK-001 Example
 * Copyright (c) 2019 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include "mbed.h"
#define BH1790GLC_CONNECTED     0

#include "BM1383AGLV.h"
#include "BM1422AGMV.h"
#include "KX224.h"
#include "BH1749NUC.h"
#include "rohm-rpr0521/rohm-rpr0521/rpr0521.h"
#include "rohm-rpr0521/rohm-rpr0521/rpr0521_driver.h"
#include "rohm-sensor-hal/rohm-sensor-hal/I2CCommon.h"
#include "rohm-bh1790glc-driver/bh1790glc.h"

BM1383AGLV pressure(I2C_SDA, I2C_SCL);
BM1422AGMV magnetometer(I2C_SDA, I2C_SCL);
KX224 accelerometer(I2C_SDA, I2C_SCL);
BH1749NUC color(I2C_SDA, I2C_SCL);
#if BH1790GLC_CONNECTED == 1
RegisterWriter i2c_rw_bh(I2C_SDA, I2C_SCL);
BH1790GLC heartrate(i2c_rw_bh);
#endif

#ifdef TARGET_K66F
#warning A2 pin is disabled on THIS BOARD, please use A0 pin for BD1020HFV!!
#endif
AnalogIn temperature(A0); // BD1020HFV

int main() {
    float axis_ac[3], temp1_bd1020hfv, temp2_bd1020hfv ;
    bool err_rpr0521;
    int ret;
    uint16_t buf[5];

    printf("\r\nExample program for using following Rohm sensor libraries\r\n");
    printf("      - BM1383AGLV: Pressure/Temperature sensor\r\n");
    printf("      - BM1422AGMV: 3 axis magnetic sensor\r\n");
    printf("      - KX224-1053: 3 axis accelerometer\r\n");
#if BH1790GLC_CONNECTED == 1
    printf("      - BH1790GLC:  optical sensor for heart rate monitor\r\n");
#endif
    printf("      - RPR-0521RS: Light & Proximity Sensor\r\n");
    printf("      - BD1020HFV:  Temperature Sensor, Formula values should be calibrated before use.\r\n");
    printf("      - BH1749NUC:  Color Sensor\r\n");

    // Initialize KX224-1053
    ret = accelerometer.initialize();
    if (ret != 0) {
    	printf("\r\nKX224-1053  :   Failed initialization \n");
    }

#if BH1790GLC_CONNECTED == 1
	// Initialize BH1790GLC
    do {
        ret = heartrate.set_default_on();
        wait_ms(300);
    } while (ret);
#endif

    // Initialize RPR-0521RS
    I2CCommonBegin();
    rpr0521_wait_until_found();
    printf("\nRPR-0521RS Sensor found.\r\n");
    rpr0521_initial_setup();

	// Initialize BH1749NUC
    ret = color.init();
    if (ret != 0) {
    	printf("\r\nBH1749NUC  :   Failed initialization \n");
    }

    while (1) {
        printf("\n++++++++++++++++++\n");
        printf("BM1383AGLV  :   pressure=%7.2f,  temperature=%5.3f\r\n", pressure.getPressure(), pressure.getTemperature());
        magnetometer.get_val(&axis_ac[0]);
        printf("BM1422AGMV  :   X=%7.2f,    Y=%7.2f,    Z=%7.2f\r\n", axis_ac[0], axis_ac[1], axis_ac[2]);
        accelerometer.get_val(&axis_ac[0]);
       	printf("KX224-1053  :   X=%7.2f,    Y=%7.2f,    Z=%7.2f\r\n", axis_ac[0], axis_ac[1], axis_ac[2]);

#if BH1790GLC_CONNECTED == 1
        ret = heartrate.getresults(buf);
        if (ret) {
            printf("BH1790GLC   :   Failed reading value from BH1790GLC\r\n");
        } else {
            printf("BH1790GLC   :   value= %d, \t%d\n\r", buf[1], buf[0]);
        }
#endif

        err_rpr0521 = rpr0521_read_data(buf);
        if (err_rpr0521) {
            printf("RPR-0521RS  :   Failed reading value from RPR-0521RS\r\n");
        } else {
            printf("RPR-0521RS  :   PS= %4u,     Als0= %4u,   Als1= %4u\r\n", buf[0], buf[1], buf[2]);
        }

        // Input voltage 0-3.3V == 0.0-1.0f
        // -40'C == 1.87V
        //   0'C == 1.546V
        // 192'C == 0V (out of scale)
        // 1.87V / 232'C = 0.008060V/'C
        temp1_bd1020hfv = -(1000 * (temperature * 3.3f) - 1546) / 8.2f;
        temp2_bd1020hfv = 192 - ( temperature * 3.3f / 0.008060f );
        printf("BD1020HFV   :   formula1= %5.3f,     formula2= %5.3f\r\n", temp1_bd1020hfv, temp2_bd1020hfv);

        color.get_val(buf);
        printf("BH1749NUC   :   [R] %04x, [G] %04x, [B] %04x, [IR] %04x, [G2] %04x\n", buf[0], buf[1], buf[2], buf[3], buf[4]);

        wait(3);
    }
}
